package com.linkedin.lift.eval

import com.linkedin.lift.lib.{DivergenceUtils, PermutationTestUtils, StatsUtils}
import com.linkedin.lift.types.{BenefitMap, Distribution, FairnessResult, ModelPrediction}
import org.apache.spark.sql.functions._
import org.apache.spark.sql.types.IntegerType
import org.apache.spark.sql.{DataFrame, DataFrameReader, SaveMode, SparkSession}

/**
  * Utilities that stitch together various fairness metrics methods,
  * to provide more higher level APIs.
  */
object FairnessMetricsUtils {

  /**
    * Extract a flattened DataFrame containing (id, label, score) columns.
    * This input DataFrame would have an ID field, a label field and a score field.
    *
    * @param df The input DataFrame
    * @param uidField The unique ID field, like a Member ID
    * @param labelField The label field
    * @param scoreField The score field
    * @param groupIdField The grouping ID field
    * @return A flattened DataFrame containing the above 3 fields
    */
  def projectIdLabelsAndScores(df: DataFrame, uidField: String,
    labelField: String, scoreField: String, groupIdField: String): DataFrame = {
    if (groupIdField.isEmpty) {
      df.select(col(uidField), col(labelField), col(scoreField))
    } else {
      val allFields = df.schema.fieldNames
      if (allFields.contains(groupIdField)) {
        df.select(col(uidField), col(labelField), col(scoreField), col(groupIdField))
      } else {
        df.select(col(uidField), col(labelField), col(scoreField))
      }
    }
  }

  /**
    * Evaluate if the difference in metric values is significant, for every
    * pair of protected attribute values (in the given set of predictions),
    * using permutation testing.
    *
    * @param predictions The input predictions to evaluate
    * @param dimType The dimension type, such as gender or age.
    * @param metrics The metrics of interest
    * @param numTrials Number of trials to run the permutation test for
    * @param seed A random seed.
    * @return A map containing the results of the permutation test
    */
  def computePermutationTestMetrics(predictions: Seq[ModelPrediction],
    dimType: String, metrics: Seq[String], numTrials: Int,
    seed: Long): Seq[FairnessResult] = {
    // Compute all subsets of size 2 - nC2 such pairs in total
    val allDimValPairs = predictions.map(_.dimensionValue)
      .toSet
      .subsets(2)
      .toList

    metrics.flatMap { metric =>
      allDimValPairs.map { dimValPair =>
        PermutationTestUtils.permutationTest(predictions, dimType,
          dimValPair.head, dimValPair.last, metric, numTrials, seed)
      }
    }
  }

  /**
    * Evaluate a requested set of overall fairness metrics on a benefit
    * vector generated by computing a benefit metric for each
    * protected attribute value (in the given set of predictions).
    *
    * @param sampledDF The sampled input DataFrame
    * @param args The Model Fairness Measurement command line args
    * @return A sequence of various model-performance-related fairness metrics.
    */
  def computeModelPerformanceMetrics(sampledDF: DataFrame,
    args: MeasureModelFairnessMetricsCmdLineArgs): Seq[FairnessResult] = {
    val samplePredictions = ModelPrediction.compute(sampledDF,
      args.labelField, args.scoreField, args.groupIdField,
      args.protectedAttributeField)

    val permutationTestMetrics = computePermutationTestMetrics(
      samplePredictions, args.protectedAttributeField,
      args.permutationMetrics, args.numTrials, args.seed)

    val benefitMaps = args.performanceBenefitMetrics.map { benefitMetric =>
      BenefitMap.compute(samplePredictions,
        args.protectedAttributeField, benefitMetric)
    }
    val overallFairnessMetrics =
      benefitMaps.flatMap(_.computeOverallMetrics(args.overallMetrics))

    permutationTestMetrics ++ overallFairnessMetrics
  }

  /**
    * Join the input DataFrame with the protectedAttribute DataFrame, and return
    * the input DataFrame appended with the protectedAttribute.
    *
    * @param dfReader The DataFrameReader to be used
    * @param df The input DataFrame
    * @param uidField The unique ID field, such as memberId
    * @param protectedDatasetPath Path to the protected dataset. If empty, we
    *                             attempt to load the right dataset based on the
    *                             protectedAttribute specified.
    * @param uidProtectedAttributeField The uid field of the protected
    *                                   attribute dataset
    * @param protectedAttributeField The protected attribute field in the
    *                                protectedAttribute DataFrame
    * @return The joined DataFrame
    */
  def computeJoinedDF(dfReader: DataFrameReader, df: DataFrame, uidField: String,
    protectedDatasetPath: String, uidProtectedAttributeField: String,
    protectedAttributeField: String): DataFrame = {
    val protectedDF = dfReader
      .load(protectedDatasetPath)
    protectedDF.select(col(uidProtectedAttributeField).as(uidField),
      col(protectedAttributeField))
      .join(df, uidField)
  }

  /**
    * Computes a reference distribution based on the input dataset's
    * distribution.
    *
    * @param inputDistr The input dataset's distribution of protectedAttributes only
    * @param referenceDistribution The kind of reference distribution desired.
    *                              Currently, only a uniform distribution is supported.
    * @return The computed reference distribution, or None if the specified
    *         referenceDistribution parameter is invalid.
    */
  def computeReferenceDistributionOpt(inputDistr: Distribution,
    referenceDistribution: String):
    Option[Distribution] = {
    if (referenceDistribution != "UNIFORM") {
      None
    } else {
      val numDims = inputDistr.entries.size
      val uniformWeight: Double = 1.0 / numDims
      val uniformEntries = inputDistr.entries.map { case (dimVal, _) =>
        (dimVal, uniformWeight)
      }
      Some(Distribution(uniformEntries))
    }
  }

  /**
    * Computes all requested distance and divergence metrics for a given
    * distribution. It compares it to a reference distribution if necessary.
    *
    * We assume that the original and reference distributions are for the training
    * data, and are over (label, protectedAttribute), and that the reference
    * distribution also contains similar dimensions.
    *
    * @param distribution The distribution of (label, protectedAttribute) for the
    *                     training dataset
    * @param referenceDistrOpt An optional field that contains a reference
    *                          distribution of (label, protectedAttribute) to
    *                          compare against. If not provided, distance and
    *                          divergence metrics that perform comparisons will
    *                          return empty results.
    * @param args The set of parsed command line arguments for this job.
    * @return A sequence of Metric values that contain the name of the metric,
    *         any parameters used, and the result of the computation.
    */
  def computeDatasetMetrics(distribution: Distribution,
    referenceDistrOpt: Option[Distribution],
    args: MeasureDatasetFairnessMetricsCmdLineArgs,
    scoreField: String = ""): Seq[FairnessResult] = {
    val computedDistanceMetrics = DivergenceUtils.computeDistanceMetrics(
      args.distanceMetrics, distribution, referenceDistrOpt,
      args.labelField, scoreField, args.protectedAttributeField)

    val benefitMaps = DivergenceUtils.computeDistanceMetrics(
      args.benefitMetrics, distribution, referenceDistrOpt,
      args.labelField, scoreField, args.protectedAttributeField)
      .map(_.toBenefitMap)
    val computedOverallMetrics =
      benefitMaps.flatMap(_.computeOverallMetrics(args.overallMetrics))

    computedDistanceMetrics ++ computedOverallMetrics
  }

  /**
    * Computes a DataFrame with probabilities for the score field. If the
    * threshold is specified, we use it to obtain 0/1 values for the scores,
    * thus giving us probabilities. If we have raw scores, we convert it into
    * probabilities using the sigmoid function. If we have probabilities, we
    * return them as is.
    *
    * @param df The input DataFrame
    * @param thresholdOpt An optional threshold that can be provided to convert
    *                     the scores into 0/1 predictions directly.
    * @param labelField The label field
    * @param scoreField The score field
    * @param protectedAttributeField The protected attribute field
    * @param scoreType Whether the scores are raw scores or probabilities
    * @return The input DataFrame with its scores transformed into probabilities.
    */
  def computeProbabilityDF(df: DataFrame, thresholdOpt: Option[Double],
    labelField: String, scoreField: String, protectedAttributeField: String,
    scoreType: String): DataFrame = {
    val probDF =
      if (scoreType.equals("RAW")) {
        // Compute sigmoid(x) = 1.0 / (1.0 + exp(-x))
        df.select(col(labelField),
          (lit(1.0) /
            (lit(1.0) + exp(-col(scoreField))))
            .as(scoreField),
          col(protectedAttributeField))
      } else {
        df.select(labelField, scoreField, protectedAttributeField)
      }

    thresholdOpt.map { threshold =>
      probDF.select(col(labelField),
        (col(scoreField) > threshold).cast(IntegerType).as(scoreField),
        col(protectedAttributeField))
    }.getOrElse(probDF)
  }

  /**
    * Computes all requested model-related fairness metrics for a given
    * dataset. We assume that the dataset has the (label, score/prediction,
    * protectedAttribute) fields at the very least.
    *
    * At a high level, there are three kinds of metrics. The first involves
    * checking for statistically significant differences in a particular metric
    * across various protected groups. The second kind computes conventional
    * notions of fairness such as Demographic Parity and Equalized Odds. The
    * third kind is to compute aggregate metrics. This can further be divided
    * into two kinds: the first is used to summarize metrics such as Demographic
    * Parity and Equalized Odds, while the second is used to summarize model
    * performance metrics across various groups (such as precision, TPR, FPR, AUC etc.).
    *
    * This method works by sampling the input data and computes metrics. Typically,
    * 50k-100k rows of data are more than sufficient to compute good estimates
    * without having to analyze the entire output. If the number of data points
    * in the input DataFrame is less than this, the entire dataset is analyzed.
    *
    * @param df The input DataFrame
    * @param referenceDistrOpt An optional reference distribution to compare against
    * @param args The set of model-related fairness measurement command line args
    * @return A sequence of Metric values that contain the name of the metric,
    *         any parameters used, and the result of the computation.
    */
  def computeModelMetrics(df: DataFrame, referenceDistrOpt: Option[Distribution],
    args: MeasureModelFairnessMetricsCmdLineArgs): Seq[FairnessResult] = {
    val probabilityDF = computeProbabilityDF(df, args.thresholdOpt,
      args.labelField, args.scoreField,
      args.protectedAttributeField, args.scoreType)

    val distribution = DivergenceUtils
      .computeGeneralizedPredictionCountDistribution(probabilityDF,
        args.labelField, args.scoreField, args.protectedAttributeField)

    val distMetrics = computeDatasetMetrics(distribution, referenceDistrOpt,
      MeasureDatasetFairnessMetricsCmdLineArgs(
        datasetPath = args.datasetPath,
        protectedDatasetPath = args.protectedDatasetPath,
        uidField = args.uidField,
        labelField = args.labelField,
        protectedAttributeField = args.protectedAttributeField,
        outputPath = args.outputPath,
        referenceDistribution = args.referenceDistribution,
        distanceMetrics = args.distanceMetrics,
        overallMetrics = args.overallMetrics,
        benefitMetrics = args.distanceBenefitMetrics), args.scoreField)

    val sampledDF =
      if (args.groupIdField.isEmpty) {
        StatsUtils.sampleDataFrame(probabilityDF, args.labelField,
          args.approxRows, args.labelZeroPercentage, args.seed)
      } else {
        StatsUtils.sampleDataFrameByGroupId(df, args.labelField, args.scoreField,
          args.groupIdField, args.protectedAttributeField, args.approxRows, args.seed)
      }

    val modelPerfMetrics = computeModelPerformanceMetrics(sampledDF, args)
    distMetrics ++ modelPerfMetrics
  }

  /**
    * Writes a sequence of FairnessResults out to disk.
    *
    * @param spark The Spark Session to use
    * @param dataFormat Output data format
    * @param dataOptions options for the DataFrameWriter
    * @param outputPath Output path for the results
    * @param fairnessResults Sequence of results to be written out
    */
  def writeFairnessResults(spark: SparkSession, dataFormat: String,
    dataOptions: Map[String, String], outputPath: String,
    fairnessResults: Seq[FairnessResult]): Unit = {
    FairnessResult.toDF(spark, fairnessResults)
      .repartition(1)
      .write
      .mode(SaveMode.Overwrite)
      .format(dataFormat)
      .options(dataOptions)
      .save(outputPath)
  }

  /**
    * Computes dataset metrics and logs the output for tracking purposes.
    *
    * @param df The input data
    * @param referenceDistrOpt An optional reference distribution
    * @param args Command line args for dataset metrics measurement
    */
  def computeAndWriteDatasetMetrics(df: DataFrame,
    referenceDistrOpt: Option[Distribution],
    args: MeasureDatasetFairnessMetricsCmdLineArgs): Unit = {
    // Compute the label-protected attribute distribution of the input data
    val distribution = Distribution.compute(df,
      Set(args.labelField, args.protectedAttributeField))

    // Passing in the appropriate parameters to this API returns the fairness metrics
    val fairnessMetrics = FairnessMetricsUtils.computeDatasetMetrics(
      distribution, referenceDistrOpt, args)

    // The above fairness metrics can be written out to HDFS
    writeFairnessResults(df.sparkSession, args.dataFormat, args.dataOptions,
      args.outputPath, fairnessMetrics)
  }

  /**
    * Computes model metrics and logs the output for tracking purposes.
    *
    * @param df The input data
    * @param referenceDistrOpt An optional reference distribution
    * @param args Command line args for model metrics measurement
    */
  def computeAndWriteModelMetrics(df: DataFrame,
    referenceDistrOpt: Option[Distribution],
    args: MeasureModelFairnessMetricsCmdLineArgs): Unit = {
    // Passing in the appropriate parameters to this API returns the fairness metrics
    val fairnessMetrics = FairnessMetricsUtils.computeModelMetrics(df,
      referenceDistrOpt, args)

    // The above fairness metrics can be written out to HDFS
    writeFairnessResults(df.sparkSession, args.dataFormat, args.dataOptions,
      args.outputPath, fairnessMetrics)
  }
}
