package co.tomlee.nifty;

import com.facebook.nifty.client.NettyClientConfig;
import com.facebook.nifty.client.NettyClientConfigBuilder;
import com.facebook.nifty.client.NiftyClient;
import com.google.common.base.Preconditions;
import io.airlift.units.Duration;
import org.apache.commons.pool2.impl.GenericKeyedObjectPoolConfig;
import org.apache.thrift.TServiceClient;
import org.apache.thrift.protocol.TBinaryProtocol;
import org.apache.thrift.protocol.TProtocolFactory;

import java.util.concurrent.TimeUnit;

public final class TNiftyClientChannelTransportPoolConfig {
    final NiftyClient niftyClient;
    final Class<? extends TServiceClient> clientClass;

    GenericKeyedObjectPoolConfig poolConfig = new GenericKeyedObjectPoolConfig();
    int maxFrameSize = 1024 * 1024;
    Duration connectTimeout = new Duration(5, TimeUnit.SECONDS);
    Duration readTimeout = new Duration(5, TimeUnit.SECONDS);
    Duration receiveTimeout = new Duration(5, TimeUnit.SECONDS);
    Duration sendTimeout = new Duration(5, TimeUnit.SECONDS);
    TProtocolFactory protocolFactory = new TBinaryProtocol.Factory();
    TNiftyClientTransportValidator checkTransport;

    public TNiftyClientChannelTransportPoolConfig(final Class<? extends TServiceClient> clientClass) {
        this(new NettyClientConfigBuilder().build(), clientClass);
    }

    public TNiftyClientChannelTransportPoolConfig(final int bossThreads, final int workerThreads, final Class<? extends TServiceClient> clientClass) {
        this(new NettyClientConfigBuilder().setWorkerThreadCount(workerThreads).setBossThreadCount(bossThreads).build(), clientClass);
    }

    public TNiftyClientChannelTransportPoolConfig(final NettyClientConfig nettyClientConfig, final Class<? extends TServiceClient> clientClass) {
        this(new NiftyClient(nettyClientConfig), clientClass);
    }

    public TNiftyClientChannelTransportPoolConfig(final NiftyClient niftyClient, final Class<? extends TServiceClient> clientClass) {
        this.niftyClient = niftyClient;
        this.clientClass = clientClass;
        poolConfig.setMaxTotal(512);
        poolConfig.setMaxTotalPerKey(8);
        poolConfig.setMaxIdlePerKey(6);
        poolConfig.setMinIdlePerKey(2);
        poolConfig.setBlockWhenExhausted(true);
        poolConfig.setLifo(false);
    }

    void validate() {
        Preconditions.checkNotNull(niftyClient);
        Preconditions.checkNotNull(clientClass);
    }

    public TNiftyClientChannelTransportPoolConfig connectTimeout(final int connectTimeout, final TimeUnit timeUnit) {
        this.connectTimeout = new Duration(connectTimeout, timeUnit);
        return this;
    }

    public TNiftyClientChannelTransportPoolConfig receiveTimeout(final int receiveTimeout, final TimeUnit timeUnit) {
        this.receiveTimeout = new Duration(receiveTimeout, timeUnit);
        return this;
    }

    public TNiftyClientChannelTransportPoolConfig readTimeout(final int readTimeout, final TimeUnit timeUnit) {
        this.readTimeout = new Duration(readTimeout, timeUnit);
        return this;
    }

    public TNiftyClientChannelTransportPoolConfig sendTimeout(final int sendTimeout, final TimeUnit timeUnit) {
        this.sendTimeout = new Duration(sendTimeout, timeUnit);
        return this;
    }

    public TNiftyClientChannelTransportPoolConfig maxFrameSize(final int maxFrameSize) {
        this.maxFrameSize = maxFrameSize;
        return this;
    }

    public TNiftyClientChannelTransportPoolConfig protocolFactory(final TProtocolFactory protocolFactory) {
        this.protocolFactory = protocolFactory;
        return this;
    }

    public TNiftyClientChannelTransportPoolConfig maxTotalConnections(final int maxTotalConnections) {
        poolConfig.setMaxTotal(maxTotalConnections);
        return this;
    }

    public TNiftyClientChannelTransportPoolConfig maxConnectionsPerEndpoint(final int maxConnectionsPerEndpoint) {
        poolConfig.setMaxTotalPerKey(maxConnectionsPerEndpoint);
        return this;
    }

    public TNiftyClientChannelTransportPoolConfig maxIdlePerEndpoint(final int maxIdlePerEndpoint) {
        poolConfig.setMaxIdlePerKey(maxIdlePerEndpoint);
        return this;
    }

    public TNiftyClientChannelTransportPoolConfig minIdlePerEndpoint(final int minIdlePerEndpoint) {
        poolConfig.setMinIdlePerKey(minIdlePerEndpoint);
        return this;
    }

    public TNiftyClientChannelTransportPoolConfig blockWhenExhausted(final boolean blockWhenExhausted) {
        poolConfig.setBlockWhenExhausted(blockWhenExhausted);
        return this;
    }

    public TNiftyClientChannelTransportPoolConfig maxWaitTime(final int maxWaitTime, final TimeUnit maxWaitUnit) {
        poolConfig.setMaxWaitMillis(maxWaitUnit.toMillis(maxWaitTime));
        return this;
    }

    public TNiftyClientChannelTransportPoolConfig validateOnBorrow(final boolean validateOnBorrow) {
        poolConfig.setTestOnBorrow(validateOnBorrow);
        return this;
    }

    public TNiftyClientChannelTransportPoolConfig validateOnReturn(final boolean validateOnReturn) {
        poolConfig.setTestOnReturn(validateOnReturn);
        return this;
    }

    public TNiftyClientChannelTransportPoolConfig validateWhileIdle(final boolean validateWhileIdle) {
        poolConfig.setTestWhileIdle(validateWhileIdle);
        return this;
    }

    public TNiftyClientChannelTransportPoolConfig minEvictableIdleTime(final int time, final TimeUnit timeUnit) {
        poolConfig.setMinEvictableIdleTimeMillis(timeUnit.toMillis(time));
        return this;
    }

    public TNiftyClientChannelTransportPoolConfig timeBetweenEvictionRuns(final int time, final TimeUnit timeUnit) {
        poolConfig.setTimeBetweenEvictionRunsMillis(timeUnit.toMillis(time));
        return this;
    }

    public TNiftyClientChannelTransportPoolConfig numTestsPerEvictionRun(final int numTestsPerEvictionRun) {
        poolConfig.setNumTestsPerEvictionRun(numTestsPerEvictionRun);
        return this;
    }

    public TNiftyClientChannelTransportPoolConfig enableJmx() {
        poolConfig.setJmxEnabled(true);
        return this;
    }

    public TNiftyClientChannelTransportPoolConfig lifo(final boolean lifo) {
        poolConfig.setLifo(lifo);
        return this;
    }
}

