package co.tomlee.nifty;

import com.google.common.net.HostAndPort;
import org.apache.thrift.TServiceClient;
import org.apache.thrift.TServiceClientFactory;
import org.apache.thrift.protocol.TProtocol;
import org.apache.thrift.transport.TTransport;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Proxy;
import java.net.InetSocketAddress;
import java.time.Duration;

/**
 * Client proxy factory wrapping a connection pool.
 *
 * @param <U>    the service interface type
 */
public final class PoolingThriftClientProxyFactory<U> implements ThriftClientProxyFactory<U> {
    private final TNiftyClientChannelTransportPool connectionPool;
    private final Constructor<U> proxyConstructor;

    /**
     * Initialize a new client proxy factory supporting connection pooling.
     *
     * @param connectionPool    the connection pool
     * @param serviceInterface  service interface type
     */
    @SuppressWarnings("unchecked")
    public PoolingThriftClientProxyFactory(final TNiftyClientChannelTransportPool connectionPool, final Class<U> serviceInterface) {
        this.connectionPool = connectionPool;
        final Class<U> proxyClass =
            (Class<U>) Proxy.getProxyClass(getClass().getClassLoader(), serviceInterface);
        try {
            proxyConstructor = proxyClass.getConstructor(InvocationHandler.class);
        }
        catch (Exception e) {
            throw new IllegalStateException(e);
        }
    }

    /**
     * Create a new client proxy for a remote Thrift service endpoint.
     *
     * @param host    the remote hostname or IP
     * @param port    the port number
     * @return the newly instantiated proxy
     * @throws Exception
     */
    public U newProxy(final String host, final int port) throws Exception {
        return newProxy(new InetSocketAddress(host, port));
    }

    /**
     * Create a new client proxy for a remote Thrift service endpoint.
     *
     * @param hostAndPort the remote hostname or IP and port
     * @return the newly instantiated proxy
     * @throws Exception
     */
    public U newProxy(final HostAndPort hostAndPort) throws Exception {
        return newProxy(new InetSocketAddress(hostAndPort.getHostText(), hostAndPort.getPort()));
    }

    /**
     * Create a new client proxy for a remote Thrift service endpoint.
     *
     * @param address    the remote hostname or IP and port
     * @return the newly instantiated proxy
     * @throws Exception
     */
    public U newProxy(final InetSocketAddress address) throws Exception {
        return proxyConstructor.newInstance(newInvocationHandler(address, null));
    }

    /**
     * Create a new client proxy for a remote Thrift service endpoint including a borrow timeout.
     *
     * @param host    the remote hostname or IP
     * @param port    the port number
     * @return the newly instantiated proxy
     * @throws Exception
     */
    public U newProxy(final String host, final int port, final Duration timeout) throws Exception {
        final InetSocketAddress address = new InetSocketAddress(host, port);
        return newProxy(address, timeout);
    }

    /**
     * Create a new client proxy for a remote Thrift service endpoint including a borrow timeout.
     *
     * @param hostAndPort the remote hostname or IP and port
     * @return the newly instantiated proxy
     * @throws Exception
     */
    public U newProxy(final HostAndPort hostAndPort, final Duration timeout) throws Exception {
        final InetSocketAddress address = new InetSocketAddress(hostAndPort.getHostText(), hostAndPort.getPort());
        return newProxy(address, timeout);
    }

    /**
     * Create a new client proxy for a remote Thrift service endpoint including a borrow timeout.
     *
     * @param address    the remote hostname or IP and port
     * @return the newly instantiated proxy
     * @throws Exception
     */
    public U newProxy(final InetSocketAddress address, final Duration timeout) throws Exception {
        return proxyConstructor.newInstance(newInvocationHandler(address, timeout));
    }

    @SuppressWarnings("unchecked")
    private U makeClient(final TProtocol protocol) throws Exception {
        return (U) connectionPool.clientConstructor().newInstance(protocol);
    }

    private InvocationHandler newInvocationHandler(final InetSocketAddress address, final Duration timeout) {
        return (proxy, method, args) -> {
            final TTransport transport;
            if (timeout == null) {
                transport = connectionPool.getTransport(address);
            }
            else {
                transport = connectionPool.getTransport(address, timeout);
            }
            try {
                final TProtocol protocol = connectionPool.protocolFactory().getProtocol(transport);
                final U client = makeClient(protocol);
                return method.invoke(client, args);
            }
            finally {
                transport.close();
            }
        };
    }
}
