package co.tomlee.nifty;

import com.facebook.nifty.client.NettyClientConfig;
import com.facebook.nifty.client.NettyClientConfigBuilder;
import com.facebook.nifty.client.NiftyClient;
import com.google.common.base.Preconditions;
import io.airlift.units.Duration;
import org.apache.commons.pool2.impl.GenericKeyedObjectPoolConfig;
import org.apache.thrift.TServiceClient;
import org.apache.thrift.protocol.TCompactProtocol;
import org.apache.thrift.protocol.TProtocolFactory;

import java.util.concurrent.TimeUnit;

public final class ThriftClientConnectionPoolConfig<T extends TServiceClient, U> {
    final NiftyClient niftyClient;
    final Class<T> clientClass;
    final Class<U> serviceInterface;

    GenericKeyedObjectPoolConfig poolConfig = new GenericKeyedObjectPoolConfig();
    int maxFrameSize = 1024 * 1024;
    Duration connectTimeout = new Duration(5, TimeUnit.SECONDS);
    Duration readTimeout = new Duration(5, TimeUnit.SECONDS);
    Duration receiveTimeout = new Duration(5, TimeUnit.SECONDS);
    Duration sendTimeout = new Duration(5, TimeUnit.SECONDS);
    TProtocolFactory protocolFactory = new TCompactProtocol.Factory(maxFrameSize);
    TNiftyClientTransportValidator checkTransport;

    public ThriftClientConnectionPoolConfig(final Class<T> clientClass, final Class<U> serviceInterface) {
        this(new NettyClientConfigBuilder().build(), clientClass, serviceInterface);
    }

    public ThriftClientConnectionPoolConfig(final int bossThreads, final int workerThreads, final Class<T> clientClass, final Class<U> serviceInterface) {
        this(new NettyClientConfigBuilder().setWorkerThreadCount(workerThreads).setBossThreadCount(bossThreads).build(), clientClass, serviceInterface);
    }

    public ThriftClientConnectionPoolConfig(final NettyClientConfig nettyClientConfig, final Class<T> clientClass, final Class<U> serviceInterface) {
        this(new NiftyClient(nettyClientConfig), clientClass, serviceInterface);
    }

    public ThriftClientConnectionPoolConfig(final NiftyClient niftyClient, final Class<T> clientClass, final Class<U> serviceInterface) {
        this.niftyClient = niftyClient;
        this.clientClass = clientClass;
        this.serviceInterface = serviceInterface;
        poolConfig.setMaxTotal(512);
        poolConfig.setMaxTotalPerKey(8);
        poolConfig.setMaxIdlePerKey(6);
        poolConfig.setMinIdlePerKey(2);
        poolConfig.setBlockWhenExhausted(true);
        poolConfig.setLifo(false);
    }

    void validate() {
        Preconditions.checkNotNull(niftyClient);
        Preconditions.checkNotNull(clientClass);
        Preconditions.checkNotNull(serviceInterface);
    }

    public ThriftClientConnectionPoolConfig<T, U> connectTimeout(final int connectTimeout, final TimeUnit timeUnit) {
        this.connectTimeout = new Duration(connectTimeout, timeUnit);
        return this;
    }

    public ThriftClientConnectionPoolConfig<T, U> receiveTimeout(final int receiveTimeout, final TimeUnit timeUnit) {
        this.receiveTimeout = new Duration(receiveTimeout, timeUnit);
        return this;
    }

    public ThriftClientConnectionPoolConfig<T, U> readTimeout(final int readTimeout, final TimeUnit timeUnit) {
        this.readTimeout = new Duration(readTimeout, timeUnit);
        return this;
    }

    public ThriftClientConnectionPoolConfig<T, U> sendTimeout(final int sendTimeout, final TimeUnit timeUnit) {
        this.sendTimeout = new Duration(sendTimeout, timeUnit);
        return this;
    }

    public ThriftClientConnectionPoolConfig<T, U> maxFrameSize(final int maxFrameSize) {
        this.maxFrameSize = maxFrameSize;
        return this;
    }

    public ThriftClientConnectionPoolConfig<T, U> protocolFactory(final TProtocolFactory protocolFactory) {
        this.protocolFactory = protocolFactory;
        return this;
    }

    public ThriftClientConnectionPoolConfig<T, U> maxTotalConnections(final int maxTotalConnections) {
        poolConfig.setMaxTotal(maxTotalConnections);
        return this;
    }

    public ThriftClientConnectionPoolConfig<T, U> maxConnectionsPerEndpoint(final int maxConnectionsPerEndpoint) {
        poolConfig.setMaxTotalPerKey(maxConnectionsPerEndpoint);
        return this;
    }

    public ThriftClientConnectionPoolConfig<T, U> maxIdlePerEndpoint(final int maxIdlePerEndpoint) {
        poolConfig.setMaxIdlePerKey(maxIdlePerEndpoint);
        return this;
    }

    public ThriftClientConnectionPoolConfig<T, U> minIdlePerEndpoint(final int minIdlePerEndpoint) {
        poolConfig.setMinIdlePerKey(minIdlePerEndpoint);
        return this;
    }

    public ThriftClientConnectionPoolConfig<T, U> blockWhenExhausted(final boolean blockWhenExhausted) {
        poolConfig.setBlockWhenExhausted(blockWhenExhausted);
        return this;
    }

    public ThriftClientConnectionPoolConfig<T, U> maxWaitTime(final int maxWaitTime, final TimeUnit maxWaitUnit) {
        poolConfig.setMaxWaitMillis(maxWaitUnit.toMillis(maxWaitTime));
        return this;
    }

    public ThriftClientConnectionPoolConfig<T, U> validateOnBorrow(final boolean validateOnBorrow) {
        poolConfig.setTestOnBorrow(validateOnBorrow);
        return this;
    }

    public ThriftClientConnectionPoolConfig<T, U> validateOnReturn(final boolean validateOnReturn) {
        poolConfig.setTestOnReturn(validateOnReturn);
        return this;
    }

    public ThriftClientConnectionPoolConfig<T, U> validateWhileIdle(final boolean validateWhileIdle) {
        poolConfig.setTestWhileIdle(validateWhileIdle);
        return this;
    }

    public ThriftClientConnectionPoolConfig<T, U> minEvictableIdleTime(final int time, final TimeUnit timeUnit) {
        poolConfig.setMinEvictableIdleTimeMillis(timeUnit.toMillis(time));
        return this;
    }

    public ThriftClientConnectionPoolConfig<T, U> timeBetweenEvictionRuns(final int time, final TimeUnit timeUnit) {
        poolConfig.setTimeBetweenEvictionRunsMillis(timeUnit.toMillis(time));
        return this;
    }

    public ThriftClientConnectionPoolConfig<T, U> numTestsPerEvictionRun(final int numTestsPerEvictionRun) {
        poolConfig.setNumTestsPerEvictionRun(numTestsPerEvictionRun);
        return this;
    }

    public ThriftClientConnectionPoolConfig<T, U> enableJmx() {
        poolConfig.setJmxEnabled(true);
        return this;
    }

    public ThriftClientConnectionPoolConfig<T, U> lifo(final boolean lifo) {
        poolConfig.setLifo(lifo);
        return this;
    }
}

