package co.datadome.api.common;

import javax.servlet.ServletException;
import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;

import static co.datadome.api.common.DataDomeHeaders.*;

public class DataDomeRequestConsumer {

    private static final Logger logger = Logger.getLogger(DataDomeRequestConsumer.class.getSimpleName());

    private final DataDomeService dataDomeService;
    private final Pattern regex;
    private final Pattern exclusionRegex;

    public DataDomeRequestConsumer(DataDomeService dataDomeService, String regex, String exclusionRegex) {
        this.dataDomeService = dataDomeService;

        this.regex = nullOrEmpty(regex) ? null : Pattern.compile(regex);
        this.exclusionRegex = nullOrEmpty(exclusionRegex) ? null : Pattern.compile(exclusionRegex);
    }

    private static boolean nullOrEmpty(String value) {
        return value == null || value.length() == 0;
    }

    public void accept(final HttpRequest httpRequest) throws IOException, ServletException {
        DataDomeRequest dataDomeRequest = buildDataDomeRequest(httpRequest);

        if (!isRegexMatched(dataDomeRequest.getUri())) {
            logger.log(Level.FINE, "DataDome regex miss");
            httpRequest.next();
            return;
        }

        DataDomeResponse dataDomeResponse = validateRequest(dataDomeRequest, httpRequest);

        if (dataDomeResponse == null) {
            httpRequest.next();
            return;
        }

        httpRequest.addHeadersInRequest(dataDomeResponse.getRequestHeaders().entrySet());
        httpRequest.addHeadersInResponse(dataDomeResponse.getResponseHeaders().entrySet());

        // block if 403 and show captcha
        if (dataDomeResponse.shouldBeBlocked()) {
            httpRequest.block(dataDomeResponse);
            return;
        }

        httpRequest.next();
    }


    protected boolean isRegexMatched(String uri) {
        if (uri == null) {
            return false;
        }

        if (exclusionRegex != null && exclusionRegex.matcher(uri).find()) {
            return false;
        }

        if (regex != null) {
            return regex.matcher(uri).find();
        }

        return true;
    }

    private DataDomeResponse validateRequest(DataDomeRequest dataDomeRequest, HttpRequest httpRequest) throws IOException {
        long startTime = System.currentTimeMillis();

        DataDomeResponse dataDomeResponse = dataDomeService.validateRequest(dataDomeRequest);
        long elapsedTime = System.currentTimeMillis() - startTime;
        logger.log(Level.INFO, "DataDome request/response time in milliseconds: {0}", elapsedTime);

        httpRequest.timeSpent(elapsedTime);

        return dataDomeResponse;
    }

    public DataDomeRequest buildDataDomeRequest(HttpRequest request) {
        DataDomeRequest.Builder requestBuilder = getRequestBuilder();

        requestBuilder.setUserAgent(request.getHeader(USER_AGENT_HEADER));
        requestBuilder.setIp(request.getIp());
        requestBuilder.setPort(request.getPort());
        requestBuilder.setClientID(request.getCookie(DATADOME_COOKIE));
        requestBuilder.setHost(request.getHeader(HOST_HEADER));
        requestBuilder.setReferer(request.getHeader(REFERER_HEADER));

        requestBuilder.setUri(request.uri());
        requestBuilder.setRequest(uriQuery(request.uri(), request.query()));

        requestBuilder.setProtocol(request.protocol());
        requestBuilder.setMethod(request.method());
        requestBuilder.setCookiesLen(Integer.toString(getHeaderLen(request, COOKIE_HEADER)));

        // Java 9 has `Instant.now()` with up to nanoseconds resolution but here we should support an old one
        requestBuilder.setTimeRequest(Long.toString(getRequestTimeStampInMicro()));

        requestBuilder.setServerHostname(request.getHeader(HOST_HEADER));
        requestBuilder.setPostParamLen(request.getHeader(CONTENT_LENGTH_HEADER));
        requestBuilder.setForwardedForIP(request.getHeader(X_FORWARDED_FOR_HEADER));

        requestBuilder.setHeadersList(headerList(request.headers()));

        requestBuilder.setAuthorizationLen(Integer.toString(getHeaderLen(request, AUTHORIZATION_HEADER)));
        requestBuilder.setxRequestedWith(request.getHeader(X_REQUESTED_WITH_HEADER));
        requestBuilder.setOrigin(request.getHeader(ORIGIN_HEADER));
        requestBuilder.setConnection(request.getHeader(CONNECTION_HEADER));
        requestBuilder.setPragma(request.getHeader(PRAGMA_HEADER));
        requestBuilder.setCacheControl(request.getHeader(CACHE_CONTROL_HEADER));
        requestBuilder.setContentType(request.getHeader(CONTENT_TYPE_HEADER));
        requestBuilder.setFrom(request.getHeader(FROM_HEADER));
        requestBuilder.setxRealIP(request.getHeader(X_REAL_IP_HEADER));
        requestBuilder.setVia(request.getHeader(VIA_HEADER));
        requestBuilder.setTrueClientIP(request.getHeader(TRUE_CLIENT_IP_HEADER));
        requestBuilder.setAccept(request.getHeader(ACCEPT_HEADER));
        requestBuilder.setAcceptCharset(request.getHeader(ACCEPT_CHARSET_HEADER));
        requestBuilder.setAcceptEncoding(request.getHeader(ACCEPT_ENCODING_HEADER));
        requestBuilder.setAcceptLanguage(request.getHeader(ACCEPT_LANGUAGE_HEADER));

        return requestBuilder.build();
    }

    private static int getHeaderLen(HttpRequest request, String name) {
        String value = request.getHeader(name);

        return value == null ? 0 : value.length();
    }


    public DataDomeRequest.Builder getRequestBuilder() {
        return DataDomeRequest.builder();
    }

    public long getRequestTimeStampInMicro() {
        return System.currentTimeMillis()*1000;
    }

}
