package co.datadome.api.vertx;

import co.datadome.api.common.DataDomeEnvironment;
import co.datadome.api.common.DataDomeRequest;
import co.datadome.api.common.DataDomeResponse;
import co.datadome.api.common.DataDomeService;
import io.vertx.core.http.HttpServerRequest;
import io.vertx.core.http.HttpServerResponse;
import io.vertx.ext.web.RoutingContext;

import java.io.IOException;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import static co.datadome.api.servlet.DataDomeFilter.CONTENT_TYPE_HEADER;
import static co.datadome.api.servlet.DataDomeFilter.REFERER_HEADER;
import static co.datadome.api.servlet.DataDomeFilter.USER_AGENT_HEADER;

public class DataDomeRouteHandler {

    private static final Logger logger = Logger.getLogger(DataDomeRouteHandler.class.getCanonicalName());

    private final DataDomeService dataDomeService;

    public DataDomeRouteHandler(DataDomeRouteHandlerConfig config) {
        dataDomeService = new DataDomeService(config.getApiKey(),
                config.getApiHost(),
                config.isApiSSL(),
                config.getRegex(),
                config.getExclusionRegex(),
                config.getProxyServer(),
                config.getProxyPort(),
                config.isProxySSL(),
                config.getConnectTimeout(),
                config.getReadTimeout(),
                config.getMaxConnections());

        DataDomeEnvironment.setServerInfo("vertex-web/3.5.3");
    }

    public void handleRequest(RoutingContext routingContext) {
        final HttpServerRequest request = routingContext.request();
        if (request == null) {
            logger.log(Level.WARNING, "DataDome can process only HttpServerResponse");
            routingContext.next();
            return;
        }

        // make DataDome request
        DataDomeRequest dataDomeRequest = buildDataDomeRequest(routingContext, request);

        if (!dataDomeService.isRegexMatched(dataDomeRequest)) {
            logger.log(Level.FINE, "DataDome regex miss");
            routingContext.next();
            return;
        }

        DataDomeResponse dataDomeResponse = null;
        long startTime = System.currentTimeMillis();
        try {
            dataDomeResponse = dataDomeService.validateRequest(dataDomeRequest);
        } catch (IOException e) {
            logger.log(Level.SEVERE, "DataDome exception on request validation:", e);
        }
        long timeSpent = System.currentTimeMillis() - startTime;
        logger.log(Level.INFO, "DataDome request/response time in milliseconds: {0}", timeSpent);

        // store time spent (ms) to call datadome api as request attribute
        request.formAttributes().add("datadome.spent_time", Long.toString(timeSpent));

        if (dataDomeResponse == null) {
            routingContext.next();
            return;
        }

        for (Map.Entry<String, String> header : dataDomeResponse.getRequestHeaders().entrySet()) {
            request.headers().add(header.getKey(), header.getValue());
        }

        final HttpServerResponse response = routingContext.response();
        for (Map.Entry<String, String> header : dataDomeResponse.getResponseHeaders().entrySet()) {
            response.headers().add(header.getKey(), header.getValue());
        }

        // block if 403 and show captcha
        if (dataDomeResponse.shouldBeBlocked()) {
            response.setStatusCode(dataDomeResponse.getStatusCode());
            response.putHeader(CONTENT_TYPE_HEADER, "text/html");
            response.end(dataDomeResponse.getResponseBody());
            return; // break filter chain
        }

        routingContext.next();
    }

    protected static DataDomeRequest buildDataDomeRequest(RoutingContext routingContext, HttpServerRequest request) {
        DataDomeRequest.Builder requestBuilder = DataDomeRequest.newBuilder();

        requestBuilder.setUserAgent(request.getHeader(USER_AGENT_HEADER));
        requestBuilder.setIp(request.remoteAddress().host());
        requestBuilder.setPort(Integer.toString(request.remoteAddress().port()));
        requestBuilder.setClientID(getCookie(routingContext, "datadome"));
        requestBuilder.setHost(request.getHeader("Host"));
        requestBuilder.setReferer(request.getHeader(REFERER_HEADER));

        requestBuilder.setUri(request.uri());
        StringBuilder uriSB = new StringBuilder();
        uriSB.append(request.uri());
        if (request.query() != null && request.query().length() > 0) {
            uriSB.append("?");
            uriSB.append(request.query());
        }
        requestBuilder.setRequest(uriSB.toString());

        requestBuilder.setProtocol(request.scheme());
        requestBuilder.setMethod(request.method().name());
        requestBuilder.setCookiesLen(Integer.toString(getHeaderLen(request, "Cookie")));

        // Java 9 has `Instant.now()` with up to nanoseconds resolution but here we should support an old one
        requestBuilder.setTimeRequest(Long.toString(System.currentTimeMillis()) + "000");

        requestBuilder.setServerHostname(request.getHeader("Host"));
        requestBuilder.setPostParamLen(request.getHeader("Content-Length"));
        requestBuilder.setForwaredForIP(request.getHeader("X-Forwarded-For"));

        // we haven't any guarantee about headers order :(
        StringBuilder sbHeaders = new StringBuilder();
        if (request.headers() != null) {
            for (String name : request.headers().names()) {
                if (sbHeaders.length() > 0) {
                    sbHeaders.append(",");
                }
                sbHeaders.append(name.toLowerCase());
            }
        }
        requestBuilder.setHeadersList(sbHeaders.toString());

        requestBuilder.setAuthorizationLen(Integer.toString(getHeaderLen(request, "Authorization")));
        requestBuilder.setxRequestedWith(request.getHeader("X-Requested-With"));
        requestBuilder.setOrigin(request.getHeader("Origin"));
        requestBuilder.setConnection(request.getHeader("Connection"));
        requestBuilder.setPragma(request.getHeader("Pragma"));
        requestBuilder.setCacheControl(request.getHeader("Cache-Control"));
        requestBuilder.setContentType(request.getHeader(CONTENT_TYPE_HEADER));
        requestBuilder.setFrom(request.getHeader("From"));
        requestBuilder.setxRealIP(request.getHeader("X-Real-IP"));
        requestBuilder.setVia(request.getHeader("Via"));
        requestBuilder.setTrueClientIP(request.getHeader("True-Client-IP"));
        requestBuilder.setAccept(request.getHeader("Accept"));
        requestBuilder.setAcceptCharset(request.getHeader("Accept-Charset"));
        requestBuilder.setAcceptEncoding(request.getHeader("Accept-Encoding"));
        requestBuilder.setAcceptLanguage(request.getHeader("Accept-Language"));

        return requestBuilder.build();
    }

    private static String getCookie(RoutingContext routingContext, String name) {
        io.vertx.ext.web.Cookie cookie = routingContext.getCookie(name);

        return cookie == null ? null : cookie.getValue();
    }

    private static int getHeaderLen(HttpServerRequest request, String name) {
        String header = request.getHeader(name);

        return header == null ? 0 : header.length();
    }

    protected DataDomeRouteHandler(DataDomeService dataDomeService) {
        this.dataDomeService = dataDomeService;
        DataDomeEnvironment.setServerInfo("vertex-web/3.5.3");
    }
}
