package co.datadome.api.servlet;

import co.datadome.api.common.DataDomeEnvironment;
import co.datadome.api.common.DataDomeRequest;
import co.datadome.api.common.DataDomeResponse;
import co.datadome.api.common.DataDomeService;

import javax.servlet.*;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Collections;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;


public class DataDomeFilter implements Filter {

    public static final String USER_AGENT_HEADER = "User-Agent";

    private static final Logger logger = Logger.getLogger(DataDomeFilter.class.getCanonicalName());
    public static final String CONTENT_TYPE_HEADER = "Content-Type";
    public static final String REFERER_HEADER = "Referer";

    private DataDomeService dataDomeService = null;

    private static final boolean VALID_CONFIGURATION_STATE = true;

    public void init(FilterConfig filterConfig) throws ServletException {
        String apiKey = getStringParam(filterConfig, "datadome.apikey", null);
        String apiHost = getStringParam(filterConfig, "datadome.hostname", DataDomeService.DEFAULT_API_HOST);
        boolean apiSSL = getBooleanParam(filterConfig, "datadome.ssl", DataDomeService.DEFAULT_API_SSL);
        String regex = getStringParam(filterConfig, "datadome.regex", DataDomeService.DEFAULT_REGEX);
        String exclusionRegex = getStringParam(filterConfig, "datadome.exclusion_regex", DataDomeService.DEFAULT_EXCLUSION_REGEX);

        int connectTimeout = getIntegerParam(filterConfig, "datadome.connection_timeout", DataDomeService.DEFAULT_CONNECT_TIMEOUT);
        int readTimeout = getIntegerParam(filterConfig, "datadome.read_timeout", DataDomeService.DEFAULT_READ_TIMEOUT);
        int maxConnections = getIntegerParam(filterConfig, "datadome.max_connections", DataDomeService.DEFAULT_MAX_TOTAL_CONNECTIONS);

        String proxyServer = getStringParam(filterConfig,"datadome.proxy_server", "");
        int proxyPort= getIntegerParam(filterConfig,"datadome.proxy_port", 0);
        boolean proxySSL = getBooleanParam(filterConfig, "datadome.proxy_ssl", false);

        dataDomeService = new DataDomeService(apiKey, apiHost, apiSSL, regex, exclusionRegex, proxyServer, proxyPort, proxySSL, connectTimeout, readTimeout, maxConnections);

        DataDomeEnvironment.setServerInfo(filterConfig.getServletContext().getServerInfo());
    }

    protected void init(DataDomeService dataDomeService, String serverInfo) {
        this.dataDomeService = dataDomeService;
        DataDomeEnvironment.setServerInfo(serverInfo);
    }

    public void doFilter(final ServletRequest req, final ServletResponse resp, final FilterChain chain)
        throws IOException, ServletException {

        if (!VALID_CONFIGURATION_STATE) {
            logger.log(Level.WARNING, "Invalid DataDome configuration");
            chain.doFilter(req, resp);
            return;
        }

        if (!(req instanceof HttpServletRequest)) {
            logger.log(Level.WARNING, "DataDome can process only HttpServletRequest");
            chain.doFilter(req, resp);
            return;
        }

        // make DataDome request
        DataDomeRequest dataDomeRequest = buildDataDomeRequest((HttpServletRequest) req);

        if (!dataDomeService.isRegexMatched(dataDomeRequest)) {
            logger.log(Level.FINE, "DataDome regex miss");
            chain.doFilter(req, resp);
            return;
        }

        // using customer request to be able to add customer headers
        EnrichedHttpServletRequest request = new EnrichedHttpServletRequest((HttpServletRequest) req);

        HttpServletResponse response = (HttpServletResponse) resp;

        long startTime = System.currentTimeMillis();
      
        DataDomeResponse dataDomeResponse = dataDomeService.validateRequest(dataDomeRequest);
        long elapsedTime = System.currentTimeMillis() - startTime;
        logger.log(Level.INFO, "DataDome request/response time in milliseconds: {0}", elapsedTime);

        long timeSpent = System.currentTimeMillis() - startTime;

        // store time spent (ms) to call datadome api as request attribute
        request.setAttribute("datadome.spent_time", timeSpent);

        if (dataDomeResponse == null) {
            chain.doFilter(req, resp);
            return;
        }

        for (Map.Entry<String, String> header : dataDomeResponse.getRequestHeaders().entrySet()) {
            request.addHeader(header.getKey(), header.getValue());
        }

        for (Map.Entry<String, String> header : dataDomeResponse.getResponseHeaders().entrySet()) {
            response.addHeader(header.getKey(), header.getValue());
        }

        // block if 403 and show captcha
        if (dataDomeResponse.shouldBeBlocked()) {
            response.setStatus(dataDomeResponse.getStatusCode());
            resp.getWriter().write(dataDomeResponse.getResponseBody()); // showing captcha
            return; // break filter chain
        }

        chain.doFilter(request, response);
    }

    public void destroy() {
        dataDomeService = null;
    }

    static DataDomeRequest buildDataDomeRequest(final HttpServletRequest request) {
        DataDomeRequest.Builder requestBuilder = DataDomeRequest.newBuilder();

        requestBuilder.setUserAgent(request.getHeader(USER_AGENT_HEADER));
        requestBuilder.setIp(request.getRemoteAddr());
        requestBuilder.setPort(Integer.toString(request.getRemotePort()));
        requestBuilder.setClientID(getCookie(request, "datadome"));
        requestBuilder.setHost(request.getHeader("Host"));
        requestBuilder.setReferer(request.getHeader(REFERER_HEADER));

        requestBuilder.setUri(request.getRequestURI());
        StringBuilder uriSB = new StringBuilder();
        uriSB.append(request.getRequestURI());
        if (request.getQueryString() != null && request.getQueryString().length() > 0) {
            uriSB.append("?");
            uriSB.append(request.getQueryString());
        }
        requestBuilder.setRequest(uriSB.toString());

        requestBuilder.setProtocol(request.getScheme());
        requestBuilder.setMethod(request.getMethod());
        requestBuilder.setCookiesLen(Integer.toString(getHeaderLen(request, "Cookie")));

        // Java 9 has `Instant.now()` with up to nanoseconds resolution but here we should support an old one
        requestBuilder.setTimeRequest(Long.toString(System.currentTimeMillis()) + "000");

        requestBuilder.setServerHostname(request.getHeader("Host"));
        requestBuilder.setPostParamLen(request.getHeader("Content-Length"));
        requestBuilder.setForwaredForIP(request.getHeader("X-Forwarded-For"));

        // we haven't any guarantee about headers order :(
        StringBuilder sbHeaders = new StringBuilder();
        if (request.getHeaderNames() != null) {
            for (String name : Collections.list(request.getHeaderNames())) {
                if (sbHeaders.length() > 0) {
                    sbHeaders.append(",");
                }
                sbHeaders.append(name.toLowerCase());
            }
        }
        requestBuilder.setHeadersList(sbHeaders.toString());

        requestBuilder.setAuthorizationLen(Integer.toString(getHeaderLen(request, "Authorization")));
        requestBuilder.setxRequestedWith(request.getHeader("X-Requested-With"));
        requestBuilder.setOrigin(request.getHeader("Origin"));
        requestBuilder.setConnection(request.getHeader("Connection"));
        requestBuilder.setPragma(request.getHeader("Pragma"));
        requestBuilder.setCacheControl(request.getHeader("Cache-Control"));
        requestBuilder.setContentType(request.getHeader(CONTENT_TYPE_HEADER));
        requestBuilder.setFrom(request.getHeader("From"));
        requestBuilder.setxRealIP(request.getHeader("X-Real-IP"));
        requestBuilder.setVia(request.getHeader("Via"));
        requestBuilder.setTrueClientIP(request.getHeader("True-Client-IP"));
        requestBuilder.setAccept(request.getHeader("Accept"));
        requestBuilder.setAcceptCharset(request.getHeader("Accept-Charset"));
        requestBuilder.setAcceptEncoding(request.getHeader("Accept-Encoding"));
        requestBuilder.setAcceptLanguage(request.getHeader("Accept-Language"));

        return requestBuilder.build();
    }

    private static String getCookie(HttpServletRequest request, String name) {
        Cookie[] cookies = request.getCookies();

        if (cookies != null) {
            for (Cookie cookie : cookies) {
                if (cookie.getName().equals(name)) {
                    return cookie.getValue();
                }
            }
        }

        return null;
    }

    private static int getHeaderLen(HttpServletRequest request, String name) {
        String cookie = request.getHeader(name);

        if (cookie == null) {
            return 0;
        }

        return cookie.length();
    }

    private static Pattern varRegex = Pattern.compile("(\\$\\{(/?[^\\}]+)\\})");

    private static String getStringParam(FilterConfig filterConfig, String name, String defaultValue) throws UnavailableException {
        String rawValue = filterConfig.getInitParameter(name);
        if (rawValue == null) {
            if (defaultValue == null) {
                throw new UnavailableException("Missed DataDome filter init-param: " + name);
            }
            return defaultValue;
        }

        Matcher m = varRegex.matcher(rawValue);

        while (m.find()) {
            String wrappedEnvName = m.group(1);
            String envName = m.group(2);
            String envValue = System.getenv(envName);
            if (envValue == null) {
                throw new UnavailableException("Undefined variable " + envName);
            } else {
                rawValue = rawValue.replace(wrappedEnvName, envValue);
            }
        }

        return rawValue;
    }

    private boolean getBooleanParam(FilterConfig filterConfig, String name, Boolean defaultValue) throws UnavailableException {
        return Boolean.valueOf(getStringParam(filterConfig, name, defaultValue.toString()));
    }

    private int getIntegerParam(FilterConfig filterConfig, String name, Integer defaultValue) throws UnavailableException {
        String param = getStringParam(filterConfig, name, defaultValue.toString());
        try {
            return Integer.valueOf(param);
        } catch (NumberFormatException e) {
            throw new UnavailableException("DataDome filter init-param: " + name + " should be valid integer but it was: " + param);
        }
    }
}
