package co.datadome.api.common;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHost;
import org.apache.http.NameValuePair;
import org.apache.http.client.config.CookieSpecs;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;

import java.io.IOException;
import java.io.InputStream;
import java.net.SocketTimeoutException;
import java.util.*;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;

@java.lang.SuppressWarnings("squid:S00107")
public class DataDomeService {

    private static final Logger logger = Logger.getLogger(DataDomeService.class.getCanonicalName());

    public static final String DEFAULT_API_HOST = "api.datadome.co";

    public static final Boolean DEFAULT_API_SSL = true;

    public static final Boolean DEFAULT_PROXY_SSL = false;

    public static final int DEFAULT_CONNECT_TIMEOUT = 150; // ms;

    public static final int DEFAULT_READ_TIMEOUT = 50; // ms;

    public static final int DEFAULT_MAX_TOTAL_CONNECTIONS = 100;

    public static final String DEFAULT_REGEX = "";

    public static final String DEFAULT_EXCLUSION_REGEX = "\\.(js|css|jpg|jpeg|png|ico|gif|tiff|svg|woff|woff2|ttf|eot|mp4|otf)$";

    // special header to confirm datadome backend status
    private static final String X_DATADOME_RESPONSE_HEADER = "X-DataDomeResponse";

    // contain the name of headers that must be added to the request
    private static final String X_DATADOME_REQUEST_HEADERS = "X-DataDome-request-headers";

    // contain the name of headers that must be added to the final client response
    private static final String X_DATADOME_RESPONSE_HEADERS = "X-DataDome-headers";

    private final String apiKey;

    private final String apiURL;

    private final RequestConfig requestConfig;
    private final CloseableHttpClient httpClient;

    private final Pattern regex;
    private final Pattern exclusionRegex;

    public DataDomeService(String apiKey, String apiHost, boolean ssl, String regex, String exclusionRegex,
                           String proxyServer, int proxyPort, boolean proxySSL,
                           int connectTimeout, int readTimeout, int maxTotalConnections) {
        this(apiKey, apiHost, ssl, regex, exclusionRegex, connectTimeout, readTimeout,
                buildHttpClient(maxTotalConnections, proxyServer, proxyPort, proxySSL));
    }

    protected DataDomeService(String apiKey, String apiHost, boolean ssl, String regex, String exclusionRegex,
                              int connectTimeout, int readTimeout, CloseableHttpClient httpClient) {
        this.apiKey = apiKey;
        this.apiURL = (ssl ? "https://" : "http://") + apiHost + "/validate-request/";

        this.httpClient = httpClient;

        this.requestConfig = RequestConfig.custom()
                // the socket timeout (SO_TIMEOUT) in milliseconds
                .setSocketTimeout(connectTimeout)
                // the timeout in milliseconds until a connection is
                // established.
                .setConnectTimeout(connectTimeout)
                // the timeout in milliseconds used when requesting a connection
                // from the connection pool.
                .setConnectionRequestTimeout(connectTimeout)
                .setSocketTimeout(readTimeout)
                .setCookieSpec(CookieSpecs.IGNORE_COOKIES)
                .build();

        this.regex = nullOrEmpty(regex) ? null : Pattern.compile(regex);

        this.exclusionRegex = nullOrEmpty(exclusionRegex) ? null : Pattern.compile(exclusionRegex);
    }

    private static boolean nullOrEmpty(String value) {
        return value == null || value.length() == 0;
    }

    private static CloseableHttpClient buildHttpClient(int maxTotalConnections,
                                                       String proxyServer,
                                                       int proxyPort,
                                                       boolean proxySSL) {
        PoolingHttpClientConnectionManager poolingHttpClientConnectionManager;

        poolingHttpClientConnectionManager = new PoolingHttpClientConnectionManager();
        poolingHttpClientConnectionManager.setMaxTotal(maxTotalConnections);
        poolingHttpClientConnectionManager.setDefaultMaxPerRoute(maxTotalConnections);

        HttpClientBuilder httpClientBuilder = HttpClients.custom()
                .useSystemProperties() // grab system settings like proxy
                .setConnectionManager(poolingHttpClientConnectionManager);

        if (proxyServer != null && proxyServer.length() > 0 && proxyPort > 0) {
            httpClientBuilder.setProxy(new HttpHost(proxyServer, proxyPort, proxySSL ? "https" : "http"));
        }

        return httpClientBuilder.build();
    }

    public boolean isRegexMatched(DataDomeRequest request) {
        if (request.getUri() == null) {
            return false;
        }

        if (exclusionRegex != null) {
            if (exclusionRegex.matcher(request.getUri()).find()) {
                return false;
            }
        }

        if (regex != null) {
            return regex.matcher(request.getUri()).find();
        }

        return true;
    }

    private String limitingFromEndForURLEncoder(String s, int limit) {
        int new_start = s.length() - 1;
        do {
            char c = s.charAt(new_start);
            new_start--;
            limit--;
            if (!Character.isLetterOrDigit(c) && c != '-' && c != '_' && c != '.' && c != '~' && c != ' ') {
                limit -= 2;
            }
        } while (new_start >= 0 && limit > 0);

        return s.substring(new_start + 1, s.length());
    }

    private String limitingForURLEncoder(String s, int limit) {
        int new_length = 0;
        for (int i = 0; i < s.length() && limit != 0; i++) {
            char c = s.charAt(i);
            if (Character.isLetterOrDigit(c) || c == '-' || c == '_' || c == '.' || c == '~') {
                limit--;
                new_length++;
            } else if (c == ' ') {
                limit--;
                new_length++;
            } else {
                if (limit > 0 && limit < 3) {
                    break;
                }
                new_length++;
                limit -= 3;
            }
        }

        return s.substring(0, new_length);
    }

    private void addParam(List<NameValuePair> postData, String name, String value, int limit, boolean from_end) {
        if (value != null) {
            if (from_end) {
                postData.add(new BasicNameValuePair(name, limitingFromEndForURLEncoder(value, limit)));
            } else {
                postData.add(new BasicNameValuePair(name, limitingForURLEncoder(value, limit)));
            }
        }
    }

    private List<NameValuePair> buildPostParams(DataDomeRequest request) {
        List<NameValuePair> postData = new ArrayList<NameValuePair>(30);

        addParam(postData, "Key", apiKey, -1, false);
        addParam(postData, "UserAgent", request.getUserAgent(), 768, false);
        addParam(postData, "IP", request.getIp(), -1, false);
        addParam(postData, "Port", request.getPort(), -1, false);
        addParam(postData, "ClientID", request.getClientID(), 128, false);
        addParam(postData, "Host", request.getHost(), 512, false);
        addParam(postData, "Referer", request.getReferer(), 1024, false);
        addParam(postData, "Request", request.getRequest(), 2048, false);
        addParam(postData, "Protocol", request.getProtocol(), -1, false);
        addParam(postData, "Method", request.getMethod(), -1, false);
        addParam(postData, "CookiesLen", request.getCookiesLen(), -1, false);
        addParam(postData, "TimeRequest", request.getTimeRequest(), -1, false);
        addParam(postData, "ServerHostname", request.getServerHostname(), -1, false);
        addParam(postData, "RequestModuleName", DataDomeEnvironment.getModuleName(), -1, false);
        addParam(postData, "ModuleVersion", DataDomeEnvironment.getModuleVersion(), -1, false);
        addParam(postData, "PostParamLen", request.getPostParamLen(), -1, false);
        addParam(postData, "ServerName", DataDomeEnvironment.getServerName(), 512, false);
        addParam(postData, "XForwaredForIP", request.getxForwaredForIP(), 512, true);
        addParam(postData, "HeadersList", request.getHeadersList(), 512, false);
        addParam(postData, "AuthorizationLen", request.getAuthorizationLen(), -1, false);
        addParam(postData, "X-Requested-With", request.getxRequestedWith(), 128, false);
        addParam(postData, "Origin", request.getOrigin(), 512, false);
        addParam(postData, "Connection", request.getConnection(), 128, false);
        addParam(postData, "Pragma", request.getPragma(), 128, false);
        addParam(postData, "CacheControl", request.getCacheControl(), 128, false);
        addParam(postData, "ContentType", request.getContentType(), 128, false);
        addParam(postData, "From", request.getFrom(), 128, false);
        addParam(postData, "X-Real-IP", request.getxRealIP(), 128, false);
        addParam(postData, "Via", request.getVia(), 128, false);
        addParam(postData, "TrueClientIP", request.getTrueClientIP(), 128, false);
        addParam(postData, "Accept", request.getAccept(), 512, false);
        addParam(postData, "AcceptCharset", request.getAcceptCharset(), 128, false);
        addParam(postData, "AcceptEncoding", request.getAcceptEncoding(), 128, false);
        addParam(postData, "AcceptLanguage", request.getAcceptLanguage(), 256, false);

        return postData;
    }

    public DataDomeResponse validateRequest(DataDomeRequest request) throws IOException {
        DataDomeResponse dataDomeResponse = null;

        HttpPost httpPost = null;
        long startTime = System.currentTimeMillis();
        CloseableHttpResponse httpResponse = null;
        try {
            httpPost = new HttpPost(apiURL);
            httpPost.setConfig(requestConfig);
            httpPost.setEntity(new UrlEncodedFormEntity(buildPostParams(request)));

            httpResponse = httpClient.execute(httpPost);

            DataDomeResponse.Builder builder = DataDomeResponse.newBuilder();

            int statusCode = httpResponse.getStatusLine().getStatusCode();
            builder.setStatusCode(statusCode);

            HttpEntity entity = httpResponse.getEntity();
            String responseBody = convertStreamToString(entity.getContent());

            EntityUtils.consume(entity);

            builder.setResponseBody(responseBody);

            if (isConfirmedStatus(statusCode, httpResponse.getFirstHeader(X_DATADOME_RESPONSE_HEADER))) {

                Set<String> requestHeaderNames = parseDataDomeResponseHeaders(httpResponse.getFirstHeader(X_DATADOME_REQUEST_HEADERS));
                Set<String> responseHeaderNames = parseDataDomeResponseHeaders(httpResponse.getFirstHeader(X_DATADOME_RESPONSE_HEADERS));

                Map<String, String> requestHeaders = new HashMap<String, String>();
                Map<String, String> responseHeaders = new HashMap<String, String>();

                // headers that must be added to the request
                for (Header header : httpResponse.getAllHeaders()) {
                    if (requestHeaderNames.contains(header.getName())) {
                        requestHeaders.put(header.getName(), header.getValue());
                    }
                    if (responseHeaderNames.contains(header.getName())
                            || (statusCode == 301 || statusCode == 302) && header.getName().equals("Location")) {
                        responseHeaders.put(header.getName(), header.getValue());
                    }
                }

                builder.setRequestHeaders(requestHeaders);
                builder.setResponseHeaders(responseHeaders);

                dataDomeResponse = builder.build();
            } else {
                logger.log(Level.WARNING, "Invalid response from DataDome");
            }
        } catch (IOException e) {
            httpPost.abort();
            long timeSpent = System.currentTimeMillis() - startTime;
            if (e instanceof SocketTimeoutException) { // we expect timeout errors here

                // if timeout occurs log datadome api request/response time in ms
                logger.log(Level.WARNING,
                        "DataDome time out happened: " + e.getMessage()
                                + ". Spent time (ms): " + timeSpent);

            } else { // in case of other errors log stack trace
                logger.log(Level.SEVERE, "Problem when connecting with DataDome servers" + ". Spent time (ms): " + timeSpent, e);
            }

        } finally {

            if (httpResponse != null) {
                httpResponse.close();
            } else if (httpPost != null) {
                httpPost.releaseConnection();
            }

        }

        return dataDomeResponse;
    }

    private static String convertStreamToString(InputStream inputStream) throws IOException {
        if (inputStream == null) {
            return "";
        }
        Scanner s = new Scanner(inputStream).useDelimiter("\\A");
        return s.hasNext() ? s.next() : "";
    }

    private boolean isConfirmedStatus(Integer StatusCode, Header header) {

        int statusInHeader;
        if (header != null) {
            try {
                statusInHeader = Integer.valueOf(header.getValue());
                return (StatusCode == statusInHeader);
            } catch (NumberFormatException e) {
                return false;
            }
        }

        return false;
    }

    private Set<String> parseDataDomeResponseHeaders(Header header) {
        if (header == null) {
            return Collections.emptySet();
        }

        Set<String> res = new HashSet<String>();
        res.addAll(Arrays.asList(header.getValue().split(" ")));
        return res;
    }
}
