package co.datadome.api.common;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.CookieSpecs;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;

import java.io.IOException;
import java.io.InputStream;
import java.util.*;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;

public class DataDomeService {

    private static final Logger logger = Logger.getLogger(DataDomeService.class.getSimpleName());

    public static final String DEFAULT_API_HOST = "api.datadome.co";

    public static final Boolean DEFAULT_API_SSL = true;

    public static final int DEFAULT_CONNECT_TIMEOUT = 150; // ms;

    public static final int DEFAULT_READ_TIMEOUT = 50; // ms;

    public static final int DEFAULT_MAX_TOTAL_CONNECTIONS = 100;

    public static final String DEFAULT_REGEX = "";

    public static final String DEFAULT_EXCLUSION_REGEX = "\\.(js|css|jpg|jpeg|png|ico|gif|tiff|svg|woff|woff2|ttf|eot|mp4|otf)$";

    // special header to confirm datadome backend status
    private static String X_DATADOME_RESPONSE_HEADER = "X-DataDomeResponse";

    // contain the name of headers that must be added to the request
    private static String X_DATADOME_REQUEST_HEADERS = "X-DataDome-request-headers";

    // contain the name of headers that must be added to the final client response
    private static String X_DATADOME_RESPONSE_HEADERS = "X-DataDome-headers";

    private String apiKey;

    private String apiURL;

    private RequestConfig requestConfig;
    private HttpClient httpClient = null;

    private Pattern regex;
    private Pattern exclusionRegex;

    public DataDomeService(String apiKey, String apiHost, boolean ssl, String regex, String exclusionRegex,
                           int connectTimeout, int readTimeout, int maxTotalConnections) {
        this.apiKey = apiKey;
        this.apiURL = (ssl ? "https://" : "http://") + apiHost + "/validate-request/";

        PoolingHttpClientConnectionManager poolingHttpClientConnectionManager;

        poolingHttpClientConnectionManager = new PoolingHttpClientConnectionManager();
        poolingHttpClientConnectionManager.setMaxTotal(maxTotalConnections);
        poolingHttpClientConnectionManager.setDefaultMaxPerRoute(maxTotalConnections);

        httpClient = HttpClients.custom()
                .setConnectionManager(poolingHttpClientConnectionManager)
                .build();

        requestConfig = RequestConfig.custom()
                // the socket timeout (SO_TIMEOUT) in milliseconds
                .setSocketTimeout(connectTimeout)
                // the timeout in milliseconds until a connection is
                // established.
                .setConnectTimeout(connectTimeout)
                // the timeout in milliseconds used when requesting a connection
                // from the connection pool.
                .setConnectionRequestTimeout(connectTimeout)
                .setSocketTimeout(readTimeout)
                .setCookieSpec(CookieSpecs.IGNORE_COOKIES)
                .build();

        if (regex != null && regex.length() > 0) {
            this.regex = Pattern.compile(regex);
        }

        if (exclusionRegex != null && exclusionRegex.length() > 0) {
            this.exclusionRegex = Pattern.compile(exclusionRegex);
        }
    }

    public boolean isRegexMatched(DataDomeRequest request) {
        if (request.getUri() == null) {
            return false;
        }

        if (exclusionRegex != null) {
            if (exclusionRegex.matcher(request.getUri()).find()) {
                return false;
            }
        }

        if (regex != null) {
            return regex.matcher(request.getUri()).find();
        }

        return true;
    }

    private String limitingForURLEncoder(String s, int limit) {
        int new_length = 0;
        for (int i = 0; i < s.length() && limit != 0; i++) {
            char c = s.charAt(i);
            if (Character.isLetterOrDigit(c) || c == '-' || c == '_' || c == '.' || c == '~') {
                limit--;
                new_length++;
            } else if (c == ' ') {
                limit--;
                new_length++;
            } else {
                if (limit > 0 && limit < 3) {
                    break;
                }
                new_length++;
                limit -= 3;
            }
        }

        return s.substring(0, new_length);
    }

    private void addParam(List<NameValuePair> postData, String name, String value, Integer limit) {
        if (value != null) {
            postData.add(new BasicNameValuePair(name, limitingForURLEncoder(value, limit)));
        }
    }

    private List<NameValuePair> buildPostParams(DataDomeRequest request) {
        List<NameValuePair> postData = new ArrayList<NameValuePair>(30);

        addParam(postData, "Key", apiKey, -1);
        addParam(postData, "UserAgent", request.getUserAgent(), 768);
        addParam(postData, "IP", request.getIp(), -1);
        addParam(postData, "Port", request.getPort(), -1);
        addParam(postData, "ClientID", request.getClientID(), 128);
        addParam(postData, "Host", request.getHost(), 512);
        addParam(postData, "Referer", request.getReferer(), 1024);
        addParam(postData, "Request", request.getRequest(), 2048);
        addParam(postData, "Protocol", request.getProtocol(), -1);
        addParam(postData, "Method", request.getMethod(), -1);
        addParam(postData, "CookiesLen", request.getCookiesLen(), -1);
        addParam(postData, "TimeRequest", request.getTimeRequest(), -1);
        addParam(postData, "ServerHostname", request.getServerHostname(), -1);
        addParam(postData, "RequestModuleName", DataDomeEnvironment.getModuleName(), -1);
        addParam(postData, "ModuleVersion", DataDomeEnvironment.getModuleVersion(), -1);
        addParam(postData, "PostParamLen", request.getPostParamLen(), -1);
        addParam(postData, "ServerName", DataDomeEnvironment.getServerName(), 512);
        addParam(postData, "XForwaredForIP", request.getxForwaredForIP(), 512);
        addParam(postData, "HeadersList", request.getHeadersList(), 512);
        addParam(postData, "AuthorizationLen", request.getAuthorizationLen(), -1);
        addParam(postData, "X-Requested-With", request.getxRequestedWith(), 128);
        addParam(postData, "Origin", request.getOrigin(), 512);
        addParam(postData, "Connection", request.getConnection(), 128);
        addParam(postData, "Pragma", request.getPragma(), 128);
        addParam(postData, "CacheControl", request.getCacheControl(), 128);
        addParam(postData, "Accept", request.getAccept(), 512);
        addParam(postData, "AcceptCharset", request.getAcceptCharset(), 128);
        addParam(postData, "AcceptEncoding", request.getAcceptEncoding(), 128);
        addParam(postData, "AcceptLanguage", request.getAcceptLanguage(), 256);

        return postData;
    }

    public DataDomeResponse validateRequest(DataDomeRequest request) {
        DataDomeResponse dataDomeResponse = null;

        HttpPost httpPost = null;

        try {
            httpPost = new HttpPost(apiURL);
            httpPost.setConfig(requestConfig);
            httpPost.setEntity(new UrlEncodedFormEntity(buildPostParams(request)));

            HttpResponse httpResponse = httpClient.execute(httpPost);

            DataDomeResponse.Builder builder = DataDomeResponse.newBuilder();

            int statusCode = httpResponse.getStatusLine().getStatusCode();
            builder.setStatusCode(statusCode);

            HttpEntity entity = httpResponse.getEntity();
            String responseBody = convertStreamToString(entity.getContent());

            EntityUtils.consume(entity);

            builder.setResponseBody(responseBody);

            if (isConfirmedStatus(statusCode, httpResponse.getFirstHeader(X_DATADOME_RESPONSE_HEADER))) {

                Set<String> requestHeaderNames = parseDataDomeResponseHeaders(httpResponse.getFirstHeader(X_DATADOME_REQUEST_HEADERS));
                Set<String> responseHeaderNames = parseDataDomeResponseHeaders(httpResponse.getFirstHeader(X_DATADOME_RESPONSE_HEADERS));

                Map<String, String> requestHeaders = new HashMap<String, String>();
                Map<String, String> responseHeaders = new HashMap<String, String>();

                // headers that must be added to the request
                for (Header header : httpResponse.getAllHeaders()) {
                    if (requestHeaderNames.contains(header.getName())) {
                        requestHeaders.put(header.getName(), header.getValue());
                    }
                    if (responseHeaderNames.contains(header.getName())
                            || (statusCode == 301 || statusCode == 302) && header.getName().equals("Location")) {
                        responseHeaders.put(header.getName(), header.getValue());
                    }
                }

                builder.setRequestHeaders(requestHeaders);
                builder.setResponseHeaders(responseHeaders);

                dataDomeResponse = builder.build();
            } else {
                logger.log(Level.WARNING, "Invalid response from DataDome");
            }
        } catch (IOException e) {
            httpPost.abort();
            logger.log(Level.SEVERE, "Problem when connecting with DataDome servers", e);
        } finally {
            if (httpPost != null) {
                httpPost.releaseConnection();
            }
        }

        return dataDomeResponse;
    }

    private static String convertStreamToString(InputStream inputStream) throws IOException {
        if (inputStream == null) {
            return "";
        }
        Scanner s = new Scanner(inputStream).useDelimiter("\\A");
        return s.hasNext() ? s.next() : "";
    }

    private boolean isConfirmedStatus(Integer StatusCode, Header header) {

        int statusInHeader;
        if (header != null) {
            try {
                statusInHeader = Integer.valueOf(header.getValue());
                return (StatusCode == statusInHeader);
            } catch (NumberFormatException e) {
                return false;
            }
        }

        return false;
    }

    private Set<String> parseDataDomeResponseHeaders(Header header) {
        if (header == null) {
            return Collections.emptySet();
        }

        Set<String> res = new HashSet<String>();
        res.addAll(Arrays.asList(header.getValue().split(" ")));
        return res;
    }
}
