
package co.datadome.api.common;

import java.util.*;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Contains details about the DataDomeResponse
 *
 * <p>
 * Example:
 * <pre>
 * {@code
 * DataDomeResponse.Builder builder = DataDomeResponse.newBuilder();
 * builder.setStatusCode(200);
 * builder.responseBody(...);
 * ...
 * DataDomeResponse dataDomeResponse = builder.build();
 * }
 * </pre>
 *
 */
public class DataDomeResponse {

    private static final Logger logger = Logger.getLogger(DataDomeResponse.class.getSimpleName());

    private final int statusCode;

    private final String responseBody;

    private final Map<String, String> requestHeaders;

    private final Map<String, String> responseHeaders;

    public int getStatusCode() {
        return statusCode;
    }

    public String getResponseBody() {
        return responseBody;
    }

    public Map<String, String> getRequestHeaders() {
        return requestHeaders;
    }

    public Map<String, String> getResponseHeaders() {
        return responseHeaders;
    }

    DataDomeResponse(int statusCode, String responseBody, Map<String, String> requestHeaders, Map<String, String> responseHeaders) {
        this.statusCode = statusCode;
        this.responseBody = responseBody;
        this.requestHeaders = requestHeaders;
        this.responseHeaders = responseHeaders;
    }

    public boolean shouldBeBlocked() {
        switch (getStatusCode()) {
            case 301:
            case 302:
            case 401:
            case 403:
                return true;

            case 200:
                return false;

            default:
                logger.log(Level.WARNING, "Unexpected status from DataDome API server: " + getStatusCode());
                return false;
        }
    }

    static Builder newBuilder() {
        return new Builder();
    }

    public static class Builder {

        private int statusCode;

        private String responseBody;

        private Map<String,String> requestHeaders;

        private Map<String,String> responseHeaders;

        public Builder setStatusCode(int statusCode) {
            this.statusCode = statusCode;
            return this;
        }

        public Builder setResponseBody(String responseBody) {
            this.responseBody = responseBody;
            return this;
        }

        public Builder setRequestHeaders(Map<String, String> requestHeaders) {
            this.requestHeaders = requestHeaders;
            return this;
        }

        public Builder setResponseHeaders(Map<String, String> responseHeaders) {
            this.responseHeaders = responseHeaders;
            return this;
        }

        public DataDomeResponse build() {
            return new DataDomeResponse(statusCode, responseBody, requestHeaders, responseHeaders);
        }
    }
}
