package co.datadome.api.common;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHost;
import org.apache.http.NameValuePair;
import org.apache.http.client.config.CookieSpecs;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.impl.conn.SystemDefaultDnsResolver;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;

import java.io.IOException;
import java.io.InputStream;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.util.*;
import java.util.logging.Level;
import java.util.logging.Logger;

import static java.lang.Integer.parseInt;

@java.lang.SuppressWarnings("squid:S00107")
public class DataDomeService {

    private static final Logger logger = Logger.getLogger(DataDomeService.class.getCanonicalName());

    public static final String DEFAULT_API_HOST = "api.datadome.co";

    public static final Boolean DEFAULT_API_SSL = true;

    public static final int DEFAULT_CONNECT_TIMEOUT = 150; // in ms

    public static final int DEFAULT_READ_TIMEOUT = 50; // in ms

    public static final int DEFAULT_MAX_TOTAL_CONNECTIONS = 100;

    public static final String DEFAULT_REGEX = "";

    public static final String DEFAULT_EXCLUSION_REGEX = "(?i)\\.(7z|avi|bmp|bz2|css|csv|doc|docx|eot|flac|flv|gif|gz|ico|jpeg|jpg|js|less|mka|mkv|mov|mp3|mp4|mpeg|mpg|odt|otf|ogg|ogm|opus|pdf|png|ppt|pptx|rar|rtf|svg|svgz|swf|tar|tbz|tgz|ttf|txt|txz|wav|webm|webp|woff|woff2|xls|xlsx|xml|xz|zip)$";

    // special header to confirm datadome backend status
    protected static final String X_DATADOME_RESPONSE_HEADER = "X-DataDomeResponse";

    // contain the name of headers that must be added to the request
    protected static final String X_DATADOME_REQUEST_HEADERS = "X-DataDome-request-headers";

    // contain the name of headers that must be added to the final client response
    protected static final String X_DATADOME_RESPONSE_HEADERS = "X-DataDome-headers";

    private static final int DEFAULT_REFRESH_IN = 5 * 60 * 1000; // in ms, 5 minutes like another our modules

    private final String apiKey;

    private final String apiURL;

    private final RequestConfig requestConfig;
    private final CloseableHttpClient httpClient;

    private static final int URL_ENCODE_CHAR_SIZE = 3;

    public DataDomeService(String apiKey, String apiHost, boolean ssl, String proxyServer, int proxyPort, boolean proxySSL,
                           int connectTimeout, int readTimeout, int maxTotalConnections) throws UnknownHostException {
        this(apiKey, apiHost, ssl, connectTimeout, readTimeout,
                buildHttpClient(apiHost, maxTotalConnections, proxyServer, proxyPort, proxySSL));
    }

    protected DataDomeService(String apiKey, String apiHost, boolean ssl,
                              int connectTimeout, int readTimeout, CloseableHttpClient httpClient) {
        this.apiKey = apiKey;
        this.apiURL = (ssl ? "https://" : "http://") + apiHost + "/validate-request/";

        this.httpClient = httpClient;

        this.requestConfig = RequestConfig.custom()
                // the socket timeout (SO_TIMEOUT) in milliseconds
                .setSocketTimeout(connectTimeout)
                // the timeout in milliseconds until a connection is established.
                .setConnectTimeout(connectTimeout)
                // the timeout in milliseconds used when requesting a connection from the connection pool.
                .setConnectionRequestTimeout(connectTimeout)
                .setSocketTimeout(readTimeout)
                .setCookieSpec(CookieSpecs.IGNORE_COOKIES)
                .build();
    }

    private static Registry<ConnectionSocketFactory> getDefaultRegistry() {
        return RegistryBuilder.<ConnectionSocketFactory>create()
                .register("http", PlainConnectionSocketFactory.getSocketFactory())
                .register("https", SSLConnectionSocketFactory.getSocketFactory())
                .build();
    }

    @SuppressWarnings("squid:S2095")
    private static CloseableHttpClient buildHttpClient(String apiHost, int maxTotalConnections,
                                                       String proxyServer,
                                                       int proxyPort,
                                                       boolean proxySSL) throws UnknownHostException {
        DataDomeResolver dataDomeResolver =
                new DataDomeResolver(apiHost, DEFAULT_REFRESH_IN, SystemDefaultDnsResolver.INSTANCE);
        PoolingHttpClientConnectionManager poolingHttpClientConnectionManager =
                new PoolingHttpClientConnectionManager(getDefaultRegistry(), dataDomeResolver);
        poolingHttpClientConnectionManager.setMaxTotal(maxTotalConnections);
        poolingHttpClientConnectionManager.setDefaultMaxPerRoute(maxTotalConnections);

        HttpClientBuilder httpClientBuilder = HttpClients.custom()
                .useSystemProperties() // grab system settings like proxy
                .setConnectionManager(poolingHttpClientConnectionManager);

        if (proxyServer != null && proxyServer.length() > 0 && proxyPort > 0) {
            httpClientBuilder.setProxy(new HttpHost(proxyServer, proxyPort, proxySSL ? "https" : "http"));
        }

        return httpClientBuilder.build();
    }

    protected static String truncateFromEndStringBaseOnUrlEncodedSize(String s, int limit) {
        if (limit < 0) {
            return s;
        }
        int newStart = s.length() - 1;
        for (int i = newStart; i >= 0 && limit != 0; i--) {
            int encodedCharSize = urlEncodedCharSize(s.charAt(i));

            if (encodedCharSize == URL_ENCODE_CHAR_SIZE && limit < URL_ENCODE_CHAR_SIZE) {
                break;
            }
            newStart--;
            limit = limit - encodedCharSize;
        }

        return s.substring(newStart + 1);
    }

    protected static String truncateStringBaseOnUrlEncodedSize(String s, int limit) {
        if (limit < 0) {
            return s;
        }
        int newLength = 0;
        for (int i = 0; i < s.length() && limit != 0; i++) {
            int encodedCharSize = urlEncodedCharSize(s.charAt(i));

            if (encodedCharSize == URL_ENCODE_CHAR_SIZE && limit < URL_ENCODE_CHAR_SIZE) {
                break;
            }
            newLength++;
            limit = limit - encodedCharSize;
        }

        return s.substring(0, newLength);
    }

    protected static int urlEncodedCharSize(char c) {
        if (Character.isLetterOrDigit(c) || c == '-' || c == '_' || c == '.' || c == '~' || c == ' ') {
            return 1;
        }
        return URL_ENCODE_CHAR_SIZE;
    }

    protected static void addParam(List<NameValuePair> postData, String name, String value, int limit, boolean fromEnd) {
        if (value != null) {
            if (fromEnd) {
                postData.add(new BasicNameValuePair(name, truncateFromEndStringBaseOnUrlEncodedSize(value, limit)));
            } else {
                postData.add(new BasicNameValuePair(name, truncateStringBaseOnUrlEncodedSize(value, limit)));
            }
        }
    }

    private static List<NameValuePair> buildPostParams(String apiKey, DataDomeRequest request) {
        List<NameValuePair> postData = new ArrayList<NameValuePair>(30);

        addParam(postData, "Key", apiKey, -1, false);
        addParam(postData, "UserAgent", request.getUserAgent(), 768, false);
        addParam(postData, "IP", request.getIp(), -1, false);
        addParam(postData, "Port", request.getPort(), -1, false);
        addParam(postData, "ClientID", request.getClientID(), 128, false);
        addParam(postData, "Host", request.getHost(), 512, false);
        addParam(postData, "Referer", request.getReferer(), 1024, false);
        addParam(postData, "Request", request.getRequest(), 2048, false);
        addParam(postData, "Protocol", request.getProtocol(), -1, false);
        addParam(postData, "Method", request.getMethod(), -1, false);
        addParam(postData, "CookiesLen", request.getCookiesLen(), -1, false);
        addParam(postData, "TimeRequest", request.getTimeRequest(), -1, false);
        addParam(postData, "ServerHostname", request.getServerHostname(), -1, false);
        addParam(postData, "RequestModuleName", DataDomeEnvironment.getModuleName(), -1, false);
        addParam(postData, "ModuleVersion", DataDomeEnvironment.getModuleVersion(), -1, false);
        addParam(postData, "PostParamLen", request.getPostParamLen(), -1, false);
        addParam(postData, "ServerName", DataDomeEnvironment.getServerName(), 512, false);
        addParam(postData, "XForwaredForIP", request.getxForwardedForIP(), 512, true);
        addParam(postData, "HeadersList", request.getHeadersList(), 512, false);
        addParam(postData, "AuthorizationLen", request.getAuthorizationLen(), -1, false);
        addParam(postData, "X-Requested-With", request.getxRequestedWith(), 128, false);
        addParam(postData, "Origin", request.getOrigin(), 512, false);
        addParam(postData, "Connection", request.getConnection(), 128, false);
        addParam(postData, "Pragma", request.getPragma(), 128, false);
        addParam(postData, "CacheControl", request.getCacheControl(), 128, false);
        addParam(postData, "ContentType", request.getContentType(), 128, false);
        addParam(postData, "From", request.getFrom(), 128, false);
        addParam(postData, "X-Real-IP", request.getxRealIP(), 128, false);
        addParam(postData, "Via", request.getVia(), 128, false);
        addParam(postData, "TrueClientIP", request.getTrueClientIP(), 128, false);
        addParam(postData, "Accept", request.getAccept(), 512, false);
        addParam(postData, "AcceptCharset", request.getAcceptCharset(), 128, false);
        addParam(postData, "AcceptEncoding", request.getAcceptEncoding(), 128, false);
        addParam(postData, "AcceptLanguage", request.getAcceptLanguage(), 256, false);

        return postData;
    }

    protected HttpPost createHttpPost(DataDomeRequest request) throws IOException {
        HttpPost httpPost = new HttpPost(apiURL);
        httpPost.setConfig(requestConfig);
        httpPost.setEntity(new UrlEncodedFormEntity(buildPostParams(apiKey, request)));
        return httpPost;
    }

    @SuppressWarnings({"squid:S3776", "squid:S1181"})
    public DataDomeResponse validateRequest(DataDomeRequest request) {
        HttpPost httpPost = null;
        long startTime = System.currentTimeMillis();
        CloseableHttpResponse httpResponse = null;
        try {
            httpPost = createHttpPost(request);

            httpResponse = httpClient.execute(httpPost);

            DataDomeResponse.Builder responseBuilder = DataDomeResponse.builder();

            int statusCode = httpResponse.getStatusLine().getStatusCode();
            responseBuilder.setStatusCode(statusCode);

            HttpEntity entity = httpResponse.getEntity();
            String responseBody = convertStreamToString(entity.getContent());

            EntityUtils.consume(entity);

            responseBuilder.setResponseBody(responseBody);

            if (!isConfirmedStatus(statusCode, httpResponse.getFirstHeader(X_DATADOME_RESPONSE_HEADER))) {
                logger.log(Level.WARNING, "Invalid response from DataDome");
                return null;
            }

            Set<String> requestHeaderNames = parseDataDomeResponseHeaders(httpResponse.getFirstHeader(X_DATADOME_REQUEST_HEADERS));
            Set<String> responseHeaderNames = parseDataDomeResponseHeaders(httpResponse.getFirstHeader(X_DATADOME_RESPONSE_HEADERS));

            Map<String, String> requestHeaders = new HashMap<String, String>();
            Map<String, String> responseHeaders = new HashMap<String, String>();

            // headers that must be added to the request
            for (Header header : httpResponse.getAllHeaders()) {
                if (requestHeaderNames.contains(header.getName())) {
                    requestHeaders.put(header.getName(), header.getValue());
                }
                if (responseHeaderNames.contains(header.getName())
                        || (statusCode == 301 || statusCode == 302) && header.getName().equals("Location")) {
                    responseHeaders.put(header.getName(), header.getValue());
                }
            }

            responseBuilder.setRequestHeaders(requestHeaders);
            responseBuilder.setResponseHeaders(responseHeaders);

            return responseBuilder.build();
        } catch (SocketTimeoutException e) {
            if (httpPost != null) {
                httpPost.abort();
            }
            long timeSpent = System.currentTimeMillis() - startTime;
            logger.log(Level.WARNING, "DataDome time out happened: " + e.getMessage() + ". Spent time (ms): " + timeSpent);
        } catch (Throwable e) {
            if (httpPost != null) {
                httpPost.abort();
            }
            long timeSpent = System.currentTimeMillis() - startTime;
            logger.log(Level.SEVERE, "Problem when connecting with DataDome servers: " + e.getMessage() + ". Spent time (ms): " + timeSpent, e);
        } finally {
            if (httpResponse != null) {
                try {
                    httpResponse.close();
                } catch (IOException ignored) {
                    // we can't do anything with it
                }
            }
            if (httpPost != null) {
                httpPost.releaseConnection();
            }
        }

        return null;
    }

    private static String convertStreamToString(InputStream inputStream) {
        if (inputStream == null) {
            return "";
        }
        Scanner s = new Scanner(inputStream).useDelimiter("\\A");
        return s.hasNext() ? s.next() : "";
    }

    protected static boolean isConfirmedStatus(int statusCode, Header header) {
        if (header == null) {
            return false;
        }

        try {
            return (statusCode == parseInt(header.getValue()));
        } catch (NumberFormatException e) {
            return false;
        }
    }

    private Set<String> parseDataDomeResponseHeaders(Header header) {
        if (header == null) {
            return Collections.emptySet();
        }

        Set<String> res = new HashSet<String>();
        res.addAll(Arrays.asList(header.getValue().split(" ")));
        return res;
    }
}
