/*
 * Copyright © 2014 Cask Data, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package co.cask.tigon.logging;

import co.cask.tigon.conf.CConfiguration;
import org.apache.hadoop.fs.FileSystem;

import java.io.IOException;

/**
 * Configuration for logging.
 */
public class LogConfiguration {

  private final String account;
  private final String application;
  private final String flow;
  private final String prefix;
  private final String path;
  private final FileSystem fs;
  private final long threshold;
  private final int instances;


  public static final String CFG_ROLL_THRESHOLD = "logfile.roll.size";
  public static final String CFG_ROLL_INSTANCES = "logfile.roll.instances";

  public static final long DEFAULT_ROLL_THRESHOLD = 4 * 1024 * 1024; // 4MB
  public static final int  DEFAULT_ROLL_INSTANCES = 5; // .log, log.1, ... log.4

  public LogConfiguration(FileSystem fs,
                          CConfiguration config,
                          String pathPrefix,
                          String tag) throws IOException {
    // parse the log tag
    String[] splits = tag.split(":");
    if (splits.length < 3) {
      throw new IOException("Invalid log tag '" + tag + "'");
    }
    String account = splits[0], app = splits[1], flow = splits[2];

    this.fs = fs;
    this.account = account;
    this.application = app;
    this.flow = flow;
    this.prefix = pathPrefix;
    this.path = String.format("%s/%s/%s/", pathPrefix, application, flow);

    this.threshold = config.getLong(CFG_ROLL_THRESHOLD, DEFAULT_ROLL_THRESHOLD);
    this.instances = config.getInt(CFG_ROLL_INSTANCES, DEFAULT_ROLL_INSTANCES);
  }

  public FileSystem getFileSystem() {
    return fs;
  }

  public String getLogFileName() {
    return "flow.log";
  }

  public String getLogFilePath() {
    return this.path;
  }

  public long getSizeThreshold() {
    return this.threshold;
  }

  public int getMaxInstances() {
    return this.instances;
  }
}
