/*
 * Copyright © 2017 Cask Data, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package co.cask.microservice.api;

import co.cask.microservice.annotation.PublicEvolving;

/**
 * This class defines the microservice interface that users would implement.
 *
 * Following is an example on how this class can be used.
 *
 * <pre>
 * <code>
 *   {@literal @}Plugin(type = Microservice.TYPE)
 *   {@literal @}Name("IdentityMicroservice")
 *   {@literal @}Description("This is a sample microservice")
 *   public final class IdentityMicroservice extends AbstractMicroservice {
 *      public static final Logger LOG = LoggerFactory.getLogger(IdentityMicroservice.class);
 *
 *      {@literal @}Override
 *      public void process(byte[] input, EventContext context, Emitter emitter) throws MicroserviceException {
 *        // Send the received data to output channel
 *        emitter.emit(input);
 *      }
 *   }
 * </code>
 * </pre>
 *
 * @see MicroserviceDefinition
 * @see MicroserviceContext
 * @see MicroserviceConfiguration
 * @since 1.0
 */
@PublicEvolving
public abstract class AbstractMicroservice implements Microservice {
  private MicroserviceContext context;

  /**
   * {@link AbstractMicroservice#start(MicroserviceContext)} is invoked just before starting the microservice.
   * This is invoked at runtime of the microservice. This method is added to support resource initialization
   * or creation.
   *
   * @param context Runtime context includes the configuration.
   * @throws MicroserviceException thrown if there is issue with initialization of microservice.
   * @see AbstractMicroservice#stop()
   * @see MicroserviceContext
   */
  public void start(MicroserviceContext context) throws MicroserviceException {
    this.context = context;
  }

  @Override
  public void consume(byte[] input, EventContext context) throws MicroserviceException {
    // no-op
  }

  @Override
  public void process(byte[] input, EventContext context, Emitter emitter) throws MicroserviceException {
    // no-op
  }

  @Override
  public void produce(Emitter emitter) throws MicroserviceException {
    // no-op
  }

  /**
   * {@link AbstractMicroservice#stop()} invoked when the microservice is being shutdown.
   *
   * @throws MicroserviceException  thrown if there is issue with stopping of microservice.
   */
  public void stop() throws MicroserviceException {
    // No-op
  }

  /**
   * @return {@link MicroserviceContext} object associated with this deployment and run.
   */
  protected MicroserviceContext getContext() {
    return context;
  }
}
