/*
 * Copyright © 2018 Cask Data, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package co.cask.hydrator.format.output;

import co.cask.cdap.api.data.format.StructuredRecord;
import co.cask.cdap.api.data.schema.Schema;
import co.cask.cdap.api.dataset.lib.KeyValue;
import co.cask.hydrator.format.StructuredToAvroTransformer;
import org.apache.avro.generic.GenericRecord;
import org.apache.parquet.avro.AvroParquetOutputFormat;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

/**
 * Defines logic for reading and writing Parquet files.
 */
public class ParquetOutputFormatter implements FileOutputFormatter<Void, GenericRecord> {
  private final StructuredToAvroTransformer recordTransformer;
  private final Schema schema;

  public ParquetOutputFormatter(Schema schema) {
    this.schema = schema;
    recordTransformer = new StructuredToAvroTransformer(schema);
  }

  @Override
  public KeyValue<Void, GenericRecord> transform(StructuredRecord record) throws IOException {
    return new KeyValue<>(null, recordTransformer.transform(record));
  }

  @Override
  public String getFormatClassName() {
    return AvroParquetOutputFormat.class.getName();
  }

  @Override
  public Map<String, String> getFormatConfig() {
    Map<String, String> conf = new HashMap<>();
    conf.put("parquet.avro.schema", schema.toString());
    return conf;
  }
}
