/*
 * Decompiled with CFR 0.152.
 */
package co.cask.common.security.zookeeper;

import co.cask.common.io.Codec;
import co.cask.common.security.zookeeper.ResourceListener;
import co.cask.common.security.zookeeper.ZKExtOperations;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.cache.AbstractLoadingCache;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.apache.twill.common.Threads;
import org.apache.twill.zookeeper.NodeChildren;
import org.apache.twill.zookeeper.NodeData;
import org.apache.twill.zookeeper.OperationFuture;
import org.apache.twill.zookeeper.ZKClient;
import org.apache.twill.zookeeper.ZKOperations;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.data.ACL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SharedResourceCache<T>
extends AbstractLoadingCache<String, T> {
    private static final String ZNODE_PATH_SEP = "/";
    private static final int MAX_RETRIES = 3;
    private static final Logger LOG = LoggerFactory.getLogger(SharedResourceCache.class);
    private final List<ACL> znodeACL;
    private final ZKClient zookeeper;
    private final Codec<T> codec;
    private final String parentZnode;
    private ZKWatcher watcher;
    private Map<String, T> resources;
    private ListenerManager listeners;

    public SharedResourceCache(ZKClient zookeeper, Codec<T> codec, String parentZnode, List<ACL> znodeACL) {
        this.zookeeper = zookeeper;
        this.codec = codec;
        this.parentZnode = parentZnode;
        this.znodeACL = znodeACL;
        this.listeners = new ListenerManager();
    }

    public void init() throws InterruptedException {
        this.watcher = new ZKWatcher();
        try {
            LOG.info("Initializing SharedResourceCache.  Checking for parent znode {}", (Object)this.parentZnode);
            if (this.zookeeper.exists(this.parentZnode).get() == null) {
                ZKOperations.ignoreError((OperationFuture)this.zookeeper.create(this.parentZnode, null, CreateMode.PERSISTENT), KeeperException.NodeExistsException.class, null).get();
            }
        }
        catch (ExecutionException ee) {
            throw Throwables.propagate((Throwable)ee.getCause());
        }
        this.resources = this.reloadAll();
        this.listeners.notifyUpdate();
    }

    private Map<String, T> reloadAll() {
        final ConcurrentMap loaded = Maps.newConcurrentMap();
        ZKOperations.watchChildren((ZKClient)this.zookeeper, (String)this.parentZnode, (ZKOperations.ChildrenCallback)new ZKOperations.ChildrenCallback(){

            public void updated(NodeChildren nodeChildren) {
                LOG.info("Listing existing children for node {}", (Object)SharedResourceCache.this.parentZnode);
                List children = nodeChildren.getChildren();
                for (String child : children) {
                    OperationFuture dataFuture = SharedResourceCache.this.zookeeper.getData(SharedResourceCache.this.joinZNode(SharedResourceCache.this.parentZnode, child), (Watcher)SharedResourceCache.this.watcher);
                    final String nodeName = SharedResourceCache.this.getZNode(dataFuture.getRequestPath());
                    Futures.addCallback((ListenableFuture)dataFuture, (FutureCallback)new FutureCallback<NodeData>(){

                        public void onSuccess(NodeData result) {
                            LOG.debug("Got data for child {}", (Object)nodeName);
                            try {
                                Object resource = SharedResourceCache.this.codec.decode(result.getData());
                                loaded.put(nodeName, resource);
                                SharedResourceCache.this.listeners.notifyResourceUpdate(nodeName, resource);
                            }
                            catch (IOException ioe) {
                                throw Throwables.propagate((Throwable)ioe);
                            }
                        }

                        public void onFailure(Throwable t) {
                            LOG.error("Failed to get data for child node {}", (Object)nodeName, (Object)t);
                            SharedResourceCache.this.listeners.notifyError(nodeName, t);
                        }
                    });
                    LOG.debug("Added future for {}", (Object)child);
                }
            }
        });
        return loaded;
    }

    public void addListener(ResourceListener<T> listener) {
        this.listeners.add(listener);
    }

    public boolean removeListener(ResourceListener<T> listener) {
        return this.listeners.remove(listener);
    }

    public T get(String key) {
        if (key == null) {
            throw new NullPointerException("Key cannot be null.");
        }
        return this.resources.get(key);
    }

    public T getIfPresent(Object key) {
        Preconditions.checkArgument((boolean)(key instanceof String), (Object)"Key must be a String.");
        return this.get((String)key);
    }

    public void put(final String name, final T instance) {
        final String znode = this.joinZNode(this.parentZnode, name);
        try {
            byte[] encoded = this.codec.encode(instance);
            LOG.debug("Setting value for node {}", (Object)znode);
            ListenableFuture<String> future = ZKExtOperations.createOrSet(this.zookeeper, znode, encoded, znode, 3, this.znodeACL);
            Futures.addCallback(future, (FutureCallback)new FutureCallback<String>(){

                public void onSuccess(String result) {
                    LOG.debug("Created or set node {}", (Object)znode);
                    SharedResourceCache.this.resources.put(name, instance);
                }

                public void onFailure(Throwable t) {
                    LOG.error("Failed to set value for node {}", (Object)znode, (Object)t);
                    SharedResourceCache.this.listeners.notifyError(name, t);
                }
            });
        }
        catch (IOException ioe) {
            throw Throwables.propagate((Throwable)ioe);
        }
    }

    public void remove(Object key) {
        if (key == null) {
            throw new NullPointerException("Key cannot be null.");
        }
        final String name = key.toString();
        final String znode = this.joinZNode(this.parentZnode, name);
        OperationFuture future = this.zookeeper.delete(znode);
        Futures.addCallback((ListenableFuture)future, (FutureCallback)new FutureCallback<String>(){

            public void onSuccess(String result) {
                LOG.debug("Removed value for node {}", (Object)znode);
                SharedResourceCache.this.resources.remove(name);
            }

            public void onFailure(Throwable t) {
                LOG.error("Failed to remove znode {}", (Object)znode, (Object)t);
                SharedResourceCache.this.listeners.notifyError(name, t);
            }
        });
    }

    public Iterable<T> getResources() {
        return this.resources.values();
    }

    public long size() {
        return this.resources.size();
    }

    public void putAll(Map<? extends String, ? extends T> map) {
        for (Map.Entry<String, T> entry : map.entrySet()) {
            this.put(entry.getKey(), entry.getValue());
        }
    }

    public boolean equals(Object object) {
        if (!(object instanceof SharedResourceCache)) {
            return false;
        }
        SharedResourceCache other = (SharedResourceCache)((Object)object);
        return this.parentZnode.equals(other.parentZnode) && ((Object)this.resources).equals(other.resources);
    }

    private String joinZNode(String parent, String name) {
        if (parent.endsWith(ZNODE_PATH_SEP)) {
            return parent + name;
        }
        return parent + ZNODE_PATH_SEP + name;
    }

    private String getZNode(String path) {
        return path.substring(path.lastIndexOf(ZNODE_PATH_SEP) + 1);
    }

    private void notifyCreated(final String path) {
        LOG.debug("Got created event on {}", (Object)path);
        final String name = this.getZNode(path);
        this.getResource(path, new FutureCallback<T>(){

            public void onSuccess(T result) {
                SharedResourceCache.this.resources.put(name, result);
                SharedResourceCache.this.listeners.notifyResourceUpdate(name, result);
            }

            public void onFailure(Throwable t) {
                LOG.error("Failed updating resource for created znode {}", (Object)path, (Object)t);
                SharedResourceCache.this.listeners.notifyError(name, t);
            }
        });
    }

    private void notifyDeleted(String path) {
        LOG.debug("Got deleted event on {}", (Object)path);
        String name = this.getZNode(path);
        this.resources.remove(name);
        this.listeners.notifyDelete(name);
    }

    private void notifyChildrenChanged(String path) {
        LOG.debug("Got childrenChanged event on {}", (Object)path);
        if (!path.equals(this.parentZnode)) {
            LOG.warn("Ignoring children change on znode {}", (Object)path);
            return;
        }
        this.resources = this.reloadAll();
        this.listeners.notifyUpdate();
    }

    private void notifyDataChanged(final String path) {
        LOG.debug("Got dataChanged event on {}", (Object)path);
        final String name = this.getZNode(path);
        this.getResource(path, new FutureCallback<T>(){

            public void onSuccess(T result) {
                SharedResourceCache.this.resources.put(name, result);
                SharedResourceCache.this.listeners.notifyResourceUpdate(name, result);
            }

            public void onFailure(Throwable t) {
                LOG.error("Failed updating resource for data change on znode {}", (Object)path, (Object)t);
                SharedResourceCache.this.listeners.notifyError(name, t);
            }
        });
    }

    private void getResource(String path, final FutureCallback<T> resourceCallback) {
        OperationFuture future = this.zookeeper.getData(path, (Watcher)this.watcher);
        Futures.addCallback((ListenableFuture)future, (FutureCallback)new FutureCallback<NodeData>(){

            public void onSuccess(NodeData result) {
                Object resource = null;
                try {
                    resource = SharedResourceCache.this.codec.decode(result.getData());
                    resourceCallback.onSuccess(resource);
                }
                catch (IOException ioe) {
                    resourceCallback.onFailure((Throwable)ioe);
                }
            }

            public void onFailure(Throwable t) {
                resourceCallback.onFailure(t);
            }
        });
    }

    private class ListenerManager {
        private final Set<ResourceListener<T>> listeners = Sets.newCopyOnWriteArraySet();
        private ExecutorService listenerExecutor = Executors.newSingleThreadExecutor(Threads.createDaemonThreadFactory((String)"SharedResourceCache-listener-%d"));

        private ListenerManager() {
        }

        private void add(ResourceListener<T> listener) {
            this.listeners.add(listener);
        }

        private boolean remove(ResourceListener<T> listener) {
            return this.listeners.remove(listener);
        }

        private void notifyUpdate() {
            this.listenerExecutor.submit(new Runnable(){

                @Override
                public void run() {
                    for (ResourceListener listener : ListenerManager.this.listeners) {
                        try {
                            listener.onUpdate();
                        }
                        catch (Throwable t) {
                            LOG.error("Exception notifying listener {}", (Object)listener, (Object)t);
                            Throwables.propagateIfInstanceOf((Throwable)t, Error.class);
                        }
                    }
                }
            });
        }

        private void notifyResourceUpdate(final String name, final T resource) {
            this.listenerExecutor.submit(new Runnable(){

                @Override
                public void run() {
                    for (ResourceListener listener : ListenerManager.this.listeners) {
                        try {
                            listener.onResourceUpdate(name, resource);
                        }
                        catch (Throwable t) {
                            LOG.error("Exception notifying listener {}", (Object)listener, (Object)t);
                            Throwables.propagateIfInstanceOf((Throwable)t, Error.class);
                        }
                    }
                }
            });
        }

        private void notifyDelete(final String name) {
            this.listenerExecutor.submit(new Runnable(){

                @Override
                public void run() {
                    for (ResourceListener listener : ListenerManager.this.listeners) {
                        try {
                            listener.onResourceDelete(name);
                        }
                        catch (Throwable t) {
                            LOG.error("Exception notifying listener {}", (Object)listener, (Object)t);
                            Throwables.propagateIfInstanceOf((Throwable)t, Error.class);
                        }
                    }
                }
            });
        }

        private void notifyError(final String name, final Throwable throwable) {
            this.listenerExecutor.submit(new Runnable(){

                @Override
                public void run() {
                    for (ResourceListener listener : ListenerManager.this.listeners) {
                        try {
                            listener.onError(name, throwable);
                        }
                        catch (Throwable t) {
                            LOG.error("Exception notifying listener {}", (Object)listener, (Object)t);
                            Throwables.propagateIfInstanceOf((Throwable)t, Error.class);
                        }
                    }
                }
            });
        }
    }

    private class ZKWatcher
    implements Watcher {
        private ZKWatcher() {
        }

        public void process(WatchedEvent event) {
            LOG.debug("Watcher got event {}", (Object)event);
            switch (event.getType()) {
                case None: {
                    break;
                }
                case NodeCreated: {
                    SharedResourceCache.this.notifyCreated(event.getPath());
                    break;
                }
                case NodeDeleted: {
                    SharedResourceCache.this.notifyDeleted(event.getPath());
                    break;
                }
                case NodeChildrenChanged: {
                    SharedResourceCache.this.notifyChildrenChanged(event.getPath());
                    break;
                }
                case NodeDataChanged: {
                    SharedResourceCache.this.notifyDataChanged(event.getPath());
                }
            }
        }
    }
}

