package co.bittub.ares.cache.redis;

import co.bittub.prime.cache.CacheItem;
import co.bittub.prime.cache.redis.ListCacheHandler;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.lettuce.core.RedisClient;
import io.lettuce.core.api.StatefulRedisConnection;
import io.lettuce.core.api.sync.RedisCommands;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;

@Service
public class RedisListCacheHandler extends RedisAbstractCacheHandler implements ListCacheHandler {
    private static final Logger logger = LoggerFactory.getLogger(RedisListCacheHandler.class);
    private final ObjectMapper om;

    {
        om = new ObjectMapper();
        om.findAndRegisterModules();
    }

    @Autowired
    public RedisListCacheHandler(RedisClient client) {
        super(client);
    }

    @Override
    public long pushLeft(String key, Object... values) {
        return execute(commands -> commands.lpush(key, serialize(values)));
    }

    @Override
    public long pushRight(String key, Object... values) {
        return execute(commands -> commands.rpush(key, serialize(values)));
    }

    @Override
    public CacheItem popLeft(String key) {
        return execute(commands -> CacheItem.newInstance(key, commands.lpop(key)));
    }

    @Override
    public CacheItem popRight(String key) {
        return execute(commands -> CacheItem.newInstance(key, commands.rpop(key)));
    }

    @Override
    public List<CacheItem> range(String key, long start, long stop) {
        return execute(commands -> {
            List<String> values = commands.lrange(key, start, stop);
            List<CacheItem> items = new ArrayList<>();
            for (String value : values) {
                items.add(CacheItem.newInstance(key, value));
            }

            return items;
        });
    }

    @Override
    public long size(String key) {
        return execute(commands -> commands.llen(key));
    }

    private <R> R execute(Function<RedisCommands<String, String>, R> executor) {
        StatefulRedisConnection<String, String> connection = client.connect();
        RedisCommands<String, String> commands = connection.sync();
        R result = executor.apply(commands);
        connection.close();
        return result;
    }

    private String serialize(Object value) {
        if (value instanceof String) {
            return (String) value;
        } else {
            try {
                return om.writeValueAsString(value);
            } catch (JsonProcessingException e) {
                logger.error("Unable to serialize object {}. Exception: {}", value, e.getMessage());
            }

            return null;
        }
    }

    private String[] serialize(Object... values) {
        String[] result = new String[values.length];
        for (int i = 0; i < values.length; i++) {
            result[i] = serialize(values[i]);
        }
        return result;
    }
}
