package co.bittub.ares.cache.redis;

import co.bittub.prime.cache.CacheHandler;
import co.bittub.prime.cache.CacheItem;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.lettuce.core.RedisClient;
import io.lettuce.core.api.StatefulRedisConnection;
import io.lettuce.core.api.sync.RedisCommands;
import lombok.Getter;
import lombok.Setter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Getter
@Setter
@Service
public class RedisCacheHandler extends RedisAbstractCacheHandler implements CacheHandler {
    private static final Logger logger = LoggerFactory.getLogger(RedisCacheHandler.class);
    private final ObjectMapper om;

    {
        om = new ObjectMapper();
        om.findAndRegisterModules();
    }

    @Autowired
    public RedisCacheHandler(RedisClient client) {
        super(client);
    }

    @Override
    public CacheItem get(String key) {
        CacheItem item = CacheItem.newInstance(key);
        try {
            StatefulRedisConnection<String, String> connection = getConnection();
            RedisCommands<String, String> commands = connection.sync();
            item.setValue(commands.get(key));
            connection.close();
        } catch (Exception e) {
            logger.error("Unable to get cache of key {}. Exception: {}", key, e.getMessage());
        }

        return item;
    }

    @Override
    public int put(CacheItem... items) {
        int numberOfPutItems = 0;
        for (CacheItem item : items) {
            if (item.getValue() == null) {
                continue;
            }
            if (put(item)) {
                numberOfPutItems++;
            }
        }

        return numberOfPutItems;
    }

    @Override
    public int delete(String... keys) {
        int numberOfDeletedItems = 0;
        StatefulRedisConnection<String, String> connection = getConnection();
        RedisCommands<String, String> commands = connection.sync();
        for (String key : keys) {
            try {
                commands.del(key);
                numberOfDeletedItems++;
            } catch (Exception e) {
                logger.error("Unable to delete cache of key {}. Exception: {}", key, e.getMessage());
            }
        }
        connection.close();

        return numberOfDeletedItems;
    }

    @Override
    public Boolean has(String key) {
        StatefulRedisConnection<String, String> connection = getConnection();
        RedisCommands<String, String> commands = connection.sync();
        boolean ok = commands.exists(key) > 0;
        connection.close();

        return ok;
    }

    private boolean put(CacheItem item) {
        try {
            StatefulRedisConnection<String, String> connection = getConnection();
            RedisCommands<String, String> commands = connection.sync();

            if (item.getValue() instanceof String) {
                commands.set(item.getKey(), (String) item.getValue());
            } else {
                commands.set(item.getKey(), om.writeValueAsString(item.getValue()));
            }

            if (item.getExpiration() != null) {
                commands.expire(item.getKey(), item.getExpiration());
            }

            connection.close();
            return true;
        } catch (Exception e) {
            logger.error("Unable to put cache of key {}. Exception: {}", item.getKey(), e.getMessage());
        }

        return false;
    }
}