package co.ambisafe.keyserver.service;

import co.ambisafe.keyserver.exception.JsonParseException;
import co.ambisafe.keyserver.exception.ParsingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

/**
 * Can be used by all classes within the application
 */
public class JsonUtils {
    private final Logger LOG = LoggerFactory.getLogger(JsonUtils.class);

    public void validateJson(JsonNode json) {
        if (json.isMissingNode())
            throw new JsonParseException("Content must not be null");
    }

    public void validateJson(JsonNode json, String path) {
        if (json.isMissingNode())
            throw new JsonParseException("`" + path + "` must not be null");
    }

    public void checkNotEmpty(String value) {
        if (value.equals(""))
            throw new JsonParseException("Content must not be empty");
    }

    public void checkNotEmpty(String value, String errorMsg) {
        if (value.equals(""))
            throw new JsonParseException("Content must not be empty. " + errorMsg);
    }

    public List<String> getListFromJsonNode(JsonNode json, String listNodeName) {
        JsonNode node = json.path(listNodeName);
        validateJson(node, listNodeName);
        return StreamSupport.stream(Spliterators.spliteratorUnknownSize(node.elements(), Spliterator.ORDERED), false).map(JsonNode::asText).collect(Collectors.toList());
    }

    public Map<String, String> getMap(JsonNode json, String nodeName) {
        JsonNode map = json.path(nodeName);
        validateJson(map);

        ObjectMapper mapper = new ObjectMapper();
        TypeReference<HashMap<String, String>> typeRef = new TypeReference<HashMap<String, String>>() {
        };

        try {
            return mapper.readValue(mapper.writeValueAsString(map), typeRef);
        } catch (IOException e) {
            throw new ParsingException("Can't parse map from " + nodeName);
        }
    }
}
