package co.ambisafe.keyserver.schema;

import co.ambisafe.keyserver.dto.internal.AccountWrapper;
import co.ambisafe.keyserver.dto.internal.ContainerWrapper;
import co.ambisafe.keyserver.dto.internal.TransactionResponse;
import co.ambisafe.keyserver.dto.request.AccountRequest;
import co.ambisafe.keyserver.dto.request.SignTransactionRequest;
import co.ambisafe.keyserver.dto.request.TransactionRequest;
import co.ambisafe.keyserver.exception.BuildTransactionException;
import co.ambisafe.keyserver.exception.NotSignedException;
import co.ambisafe.keyserver.exception.SuperNodeException;
import org.springframework.core.env.Environment;

import java.util.List;
import java.util.Properties;

public interface SecuritySchema {
    /**
     * Get security schema unique id
     * @return schema name
     */
    public String getSchemaName();

    /**
     * Check payload used to create containers.
     * @param request request data
     * @param accountWrapper account data
     */
    public void beforeCreateContainers(AccountRequest request, AccountWrapper accountWrapper);

    /**
     * Creating container. All created containers add to accountWrapper.
     * @param request request data
     * @param accountWrapper account data
     */
    public void createContainers(AccountRequest request, AccountWrapper accountWrapper);

    /**
     * Validate created containers.
     * @param request request data
     * @param accountWrapper account data
     */
    public void afterCreateContainers(AccountRequest request, AccountWrapper accountWrapper);

    /**
     * Validate payload used to update containers.
     * @param request request data
     * @param accountWrapper account data
     */
    public void beforeUpdateContainers(AccountRequest request, AccountWrapper accountWrapper);

    /**
     * Update containers.
     * @param request request data
     * @param accountWrapper account data
     */
    public void updateContainers(AccountRequest request, AccountWrapper accountWrapper);

    /**
     * Validate updated containers.
     * @param request request data
     * @param accountWrapper account data
     */
    public void afterUpdateContainers(AccountRequest request, AccountWrapper accountWrapper);

    /**
     * Validate payload to create transaction.
     * @param request request data
     * @param accountWrapper account data
     * @throws BuildTransactionException if transaction request is invalid
     */
    public void beforeBuildTransaction(TransactionRequest request, AccountWrapper accountWrapper) throws BuildTransactionException;

    /**
     * Building transaction.
     * @param request request data
     * @param accountWrapper account data
     * @return generated transaction
     * @throws BuildTransactionException if request is invalid
     * @throws SuperNodeException if transport or super node fails
     */
    public TransactionResponse buildTransaction(TransactionRequest request, AccountWrapper accountWrapper) throws BuildTransactionException, SuperNodeException;

    /**
     * Validating created transaction.
     * @param request request data
     * @param accountWrapper account data
     * @param transaction transaction
     */
    public void afterBuildTransaction(TransactionRequest request, AccountWrapper accountWrapper, TransactionResponse transaction);

    /**
     * Validating request to create recovery transaction.
     * @param request request data
     * @param from wallet address
     * @param to wallet address
     */
    public void beforeBuildRecoveryTransaction(TransactionRequest request, AccountWrapper from, AccountWrapper to);

    /**
     * Creating recovery transaction.
     * @param request request data
     * @param from wallet address
     * @param to wallet address
     * @return Transaction
     */
    public TransactionResponse buildRecoveryTransaction(TransactionRequest request, AccountWrapper from, AccountWrapper to);

    /**
     * Validating recovery transaction.
     * @param response Transaction
     * @param request request data
     * @param from wallet address
     * @param to wallet address
     */
    public void afterBuildRecoveryTransaction(TransactionResponse response, TransactionRequest request, AccountWrapper from, AccountWrapper to);

    /**
     * Validating payload to sign transaction.
     * @param request request data
     * @param accountWrapper account data
     */
    public void beforeSignTransaction(SignTransactionRequest request, AccountWrapper accountWrapper);

    /**
     * Sign transaction.
     * @param request request data
     * @param accountWrapper account data
     * @return signed transaction
     */
    public TransactionResponse signTransaction(SignTransactionRequest request, AccountWrapper accountWrapper);

    /**
     * Validating signing transaction.
     * @param response transaction
     * @param request request data
     * @param accountWrapper account data
     */
    public void afterSignTransaction(TransactionResponse response, SignTransactionRequest request, AccountWrapper accountWrapper);

    /**
     * Validating request to get account containers.
     * @param request account request
     * @param accountWrapper account data
     */
    public void beforeGetAccountContainers(AccountRequest request, AccountWrapper accountWrapper);

    /**
     * Fitlering and returing account containers.
     * @param request request data
     * @param accountWrapper account data
     * @return account containers that will be returned to client
     */
    public List<ContainerWrapper> getAccountContainers(AccountRequest request, AccountWrapper accountWrapper);

    /**
     * Validate containers that will be returned.
     * @param containers account containers
     * @param request request data
     * @param accountWrapper account data
     */
    public void afterGetAccountContainers(List<ContainerWrapper> containers, AccountRequest request, AccountWrapper accountWrapper);
}