package co.ambisafe.keyserver.asset;

import co.ambisafe.keyserver.dto.internal.KeyPair;
import co.ambisafe.keyserver.dto.internal.TransactionResponse;
import com.google.common.collect.ImmutableList;

/**
 * Assets implementing an interface can perform operations with non-multisig addresses
 */
public interface NonMultisigOperations {

    /**
     * Builds non-signed transaction using UTXOs of <code>key</code>
     *
     * @param key         public key, which UTXOs would be spent
     * @param destination destination address
     * @param amount      amount
     * @return POJO which will be derialized to JSON
     */
    TransactionResponse buildTransaction(final byte[] key, final String destination, final String amount);

    /**
     * Returns non-signed transaction spending all tokens to <code>destination</code> address
     *
     * @param key         public key, which UTXOs would be spent
     * @param destination destination address
     * @return POJO which will be derialized to JSON
     */
    TransactionResponse buildEmptyAddressTransaction(final byte[] key, final String destination);

    /**
     * Insert signatures. No need to run verification afterwords as it verifies the transaction.
     *
     * @param rawTransactionHex non-signed raw transaction
     * @param signatures        array of signatures. One signature per input.
     * @param key               singer's public key
     * @return                  transaction hex
     */
    String insertSignaturesInNonMultisigTransaction(final String rawTransactionHex, ImmutableList<String> signatures,
                                                    final byte[] key);

    /**
     * Creates Key pair from WIF. @see <a href="https://en.bitcoin.it/wiki/Wallet_import_format">Wallet import format</a>
     *
     * @param wif string represesntation of private key
     * @return private and public keys
     */
    KeyPair getKeyPair(String wif);
}
