package co.ambisafe.keyserver.asset;

import co.ambisafe.keyserver.dto.internal.CryptoCurrencyBalance;
import co.ambisafe.keyserver.dto.internal.KeyPair;
import co.ambisafe.keyserver.dto.internal.TransactionResponse;
import co.ambisafe.keyserver.json.JsonData;
import co.ambisafe.keyserver.supernode.SuperNode;
import com.fasterxml.jackson.databind.JsonNode;

import java.util.List;
import java.util.Map;
import java.util.Set;

public interface CryptoAsset extends AssetFamily {
    /**
     * Returns non-signed transaction and all related data needed to sign the transaction. Bitcoin implementation
     * includes calculated fee.
     *
     * @param threshold   m in m-of-n multisig.
     * @param keys        n keys in m-of-n multisig
     * @param destination destination address
     * @param amount      transferred amount
     * @return TransactionResponse
     */
    TransactionResponse buildTransaction(int threshold, List<byte[]> keys, String destination, String amount);

    /**
     * Returns non-signed transaction and all related data needed to sign the transaction. Bitcoin implementation
     * includes calculated fee.
     *
     * @param threshold   m in m-of-n multisig.
     * @param keys        n keys in m-of-n multisig
     * @param destination destination address
     * @param amount      transferred amount
     * @param jsonNode    Request json payload
     * @return TransactionResponse
     */
    TransactionResponse buildTransaction(int threshold, List<byte[]> keys, String destination, String amount, JsonNode jsonNode);

    /**
     * Returns non-signed transaction spending all tokens to <code>destination</code> address
     *
     * @param threshold   m in m-of-n multisig.
     * @param keys        n keys in m-of-n multisig
     * @param destination destination address
     * @return TransactionResponse
     */
    TransactionResponse buildEmptyAddressTransaction(int threshold, List<byte[]> keys, String destination);

    /**
     * Signs the signature hashes <code>sighashes</code> by <code>privateKey</code>
     *
     * @param privateKey to sign transaction
     * @param sighashes  which will be signed
     * @return list of signed signature hashes
     */
    List<String> sign(byte[] privateKey, List<String> sighashes);

    /**
     * Inserts signatures returned by {@link #sign(byte[], List)} in <code>rawTransactionHex</code>
     *
     * @param rawTransactionHex deserialized to String transaction
     * @param signatures        returned value of {@link #sign(byte[], List)}
     * @param key               public key needed to determine where to insert the signatures in multisig address
     * @return raw transaction with inserted signatures
     */
    String insertSignatures(String rawTransactionHex, List<String> signatures, byte[] key);

    /**
     * Returns balance of the address
     *
     * @param address used to get balance for
     * @return CryptoCurrencyBalance
     */
    CryptoCurrencyBalance getBalance(String address);

    /**
     * Crypto assets based on asymmetric cryptography. Method returns random key pair.
     *
     * @return DTO object with public and private key
     */
    KeyPair getRandomKeyPair();

    /**
     * Returns unique symbol to determine the asset.
     *
     * @return CryptoCurrencyBalance
     */
    String getSymbol();

    /**
     * Used to execute non-defined in interface methods. It accepts action name and then dispatch it to particular
     * method of the interface implementation class.
     *
     * @param actionName name of the action specified in the request
     * @param arguments  arguments of which will be performed to particular method arguments.
     * @param publicKeys public keys fetched from the account and sorted by role.
     * @return POJO which will be deserialized to JSON
     */
    JsonData dispatchAction(final String actionName, Map<String, String> arguments, List<byte[]> publicKeys);

    /**
     * Verifies if signatures are sighahsed signed by provided key.
     *
     * @param sighashes  non-signed hashes
     * @param signatures signed hashes
     * @param publicKey  key to verify signatures against sighashes
     * @return true if signature verified successfully
     */
    boolean verifySignatures(List<String> sighashes, List<String> signatures, byte[] publicKey);

    /**
     * Verifies SigScript of all inputs is valid and fully signed.
     * Throws exception if transaction is signed not correctly.
     *
     * @param rawTxHex signed raw transaction
     */
    void verifySignatures(String rawTxHex);

    /**
     * Shows if notifications about incoming transactions are enabled or disabled.
     *
     * @return boolean
     */
    boolean transactionsNotifications();

    /**
     * Returns interval in milliseconds of the job to check incoming transactions
     *
     * @return time interval as int
     */
    int getNewBlockCheckInterval();

    /**
     * Used in transaction notifications functionality.
     * Method accepts raw transaction and finds if any of <code>addresses</code> exists among destination addresses
     *
     * @param addresses      set of addresses to look up
     * @param rawTransaction Raw transaction
     * @return Returns list of hashmap defined by you that will be send to postBackUrl by notificationService
     */
    List<Map<String, Object>> getNotificationsForAccountAddresses(Set<String> addresses, String rawTransaction);

    /**
     * Used for recovery transactions. Implements different validations for addresses and transaction. E.g.:
     * transaction addresses are match with sender and destination. There are no other addresses in the transaction etc.
     * Throws exception if validation fails.
     *
     * @param rawTransaction raw transaction
     * @param sender         sender address
     * @param destination    destination address
     */
    void validateRecoveryTxAddresses(String rawTransaction, String sender, String destination);

    /**
     * Used in transaction notification functionality. Returns URL where callback must be send.
     *
     * @return URL where to send notifications
     */
    String getPostbackUrl();

    /**
     * Used to get the supernode whenever connection to crypto network is needed
     *
     * @return supernode plugin which is currently used by the asset
     */
    SuperNode getSuperNode();
}