package co.abit.prime.cache;

import co.abit.prime.cache.exception.UnsupportedClassException;
import co.abit.prime.cache.item.SimpleCacheItem;
import com.fasterxml.jackson.databind.ObjectMapper;

import java.io.IOException;
import java.nio.charset.Charset;
import java.time.Duration;
import java.time.Instant;

public interface CacheItem {
    String CHARSET_UTF8 = "UTF-8";
    String DEFAULT_CHARSET = CHARSET_UTF8;

    /**
     * Returns key of item
     *
     * @return a string represents for key
     */
    String getKey();

    /**
     * Sets key of item
     *
     * @param key a string represents for key
     */
    void setKey(String key);

    /**
     * Returns value of item
     *
     * @return an object represents for item's value
     */
    Object getValue();

    /**
     * Get value as an instance
     * @param className a string represents for class's name
     * @param <T> object type to return
     * @return an instance of T
     *
     * @throws UnsupportedClassException if value cannot convert to proposed class
     */
    default <T> T getValue(Class<T> className) {
        Object value = getValue();
        if (value == null || !(value instanceof String)) {
            return null;
        }

        try {
            ObjectMapper om = new ObjectMapper();
            om.findAndRegisterModules();
            return om.readValue(((String) value).getBytes(Charset.forName(DEFAULT_CHARSET)), className);
        } catch (IOException e) {
            throw new UnsupportedClassException(e);
        }
    }

    /**
     * Sets value of item
     *
     * @param value an object represents for item's value
     */
    void setValue(Object value);

    /**
     * Returns time which item will be expired
     *
     * @return a long value represents for expiration time
     */
    Long getExpiration();

    /**
     * Sets time which item will be expired
     *
     * @param expiration a long value represents for expiration time
     */
    void setExpiration(Long expiration);

    /**
     * Sets period of time which items will be expired
     *
     * @param duration a long value represents for period time (in milliseconds)
     */
    default void expireAfter(Long duration) {
        setExpiration(Instant.now().toEpochMilli() + duration);
    }

    /**
     * Sets period of time which items will be expired
     *
     * @see Duration
     * @param duration a Duration object represents for period time
     */
    default void expireAfter(Duration duration) {
        expireAfter(duration.toMillis());
    }

    /**
     * Creates and returns a new instance of CacheItem with key
     * @param key a string represents for cache's key
     * @return an instance of CacheItem
     */
    static CacheItem newInstance(String key) {
        return SimpleCacheItem.builder()
                .key(key)
                .build();
    }

    /**
     * Creates and returns a new instance of CacheItem with key
     * @param key a string represents for cache's key
     * @param value an object represents of value
     * @return an instance of CacheItem
     */
    static CacheItem newInstance(String key, Object value) {
        return SimpleCacheItem.builder()
                .key(key)
                .value(value)
                .build();
    }

    /**
     * Creates and returns a new instance of CacheItem with key
     * @param key a string represents for cache's key
     * @param value an object represents of value
     * @param expiration a Long value represents of expiration
     * @return an instance of CacheItem
     */
    static CacheItem newInstance(String key, Object value, Long expiration) {
        return SimpleCacheItem.builder()
                .key(key)
                .value(value)
                .expiration(expiration)
                .build();
    }
}
