/*
 * Copyright 2015 YAN KA KA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cn.com.utils.data.utils;

import android.content.ContentValues;
import android.content.Context;
import android.content.pm.PackageManager;
import android.content.res.AssetManager;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteStatement;
import android.text.TextUtils;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Set;

import cn.com.utils.data.config.DataBaseConfig;
import cn.com.utils.data.exceptions.NewClassException;

public class DataUtils {

    private static final String TAG = "DataUtils";

    private static final String DATE_TIME = DataBaseConfig.DATE_TIME;

    private DataUtils() {

        throw new NewClassException(NewClassException.NEW_CLASS_IS_PROHIBIT);
    }

    public static String getCreateTableSQL(String tableName, boolean autoIncrementId, String keyName, String valueName) {

        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append("create table %s ( ");
        if (autoIncrementId) {
            stringBuilder.append("_id integer primary key autoincrement , ");
        }
        stringBuilder.append("%s text unique on conflict replace , ");
        stringBuilder.append("%s text ,");
        stringBuilder.append(DATE_TIME + " text )");
        String sql = String.format(stringBuilder.toString(), tableName, keyName, valueName);
        DataLog.e(TAG, "create table sql fun1:" + sql);
        return sql;
    }

    public static String getCreateBlobTableSQL(String tableName, boolean autoIncrementId, String keyName, String valueName) {

        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append("create table %s ( ");
        if (autoIncrementId) {
            stringBuilder.append("_id integer primary key autoincrement , ");
        }
        stringBuilder.append("%s text unique on conflict replace , ");
        stringBuilder.append("%s blob ,");
        stringBuilder.append(DATE_TIME + " text )");
        String sql = String.format(stringBuilder.toString(), tableName, keyName, valueName);
        DataLog.e(TAG, "create table sql fun2:" + sql);
        return sql;
    }

    public static String getCreateTableSQL(String tableName, boolean autoIncrementId, ContentValues contentValues) {

        StringBuilder stringBuilder = new StringBuilder();
        if (contentValues == null) {
            return "";
        }

        stringBuilder.append("create table %s (");
        if (autoIncrementId) {
            stringBuilder.append("_id integer primary key autoincrement , ");
        }
        if (!contentValues.containsKey("datetime")) {
            stringBuilder.append(DATE_TIME + " text , ");
        }
        Set<String> keySet = contentValues.keySet();
        boolean isAppend = false;
        for (String key : keySet) {
            if (isAppend) {
                stringBuilder.append(",");
            }
            isAppend = true;
            stringBuilder.append(key).append(" text ");
        }
        stringBuilder.append(")");
        String sql = String.format(stringBuilder.toString(), tableName);
        DataLog.e(TAG, "create table sql fun3:" + sql);
        return sql;
    }


    public static String addColumnSQL(String tableName, String columnName) {

        String sqlString = "alter table %s add column %s text ";
        String sql = String.format(sqlString, tableName, columnName);
        DataLog.e(TAG, "update table sql:" + sql);
        return sql;
    }

    public static String getCreateIndexSQL(String indexName, String tableName, String columnName) {

        String sql = String.format("create index %s on %s (%s);", indexName, tableName, columnName);
        DataLog.e(TAG, "create index sql:" + sql);
        return sql;
    }

    public static String getStringColumnValue(String columnName, Cursor cursor) {

        try {
            return cursor.getString(cursor.getColumnIndex(columnName));
        } catch (Exception e) {
            e.printStackTrace();
        }
        return "";
    }

    public static int getIntColumnValue(String columnName, Cursor cursor) {

        try {
            return cursor.getInt(cursor.getColumnIndex(columnName));
        } catch (Exception e) {
            e.printStackTrace();
        }
        return -1;
    }

    public static void closeDatabase(SQLiteDatabase sqLiteDatabase) {

        if (sqLiteDatabase != null && sqLiteDatabase.isOpen()) {
            sqLiteDatabase.close();
        }
    }


    public static void closeCursor(Cursor cursor) {

        if (cursor != null && !cursor.isClosed()) {
            cursor.close();
        }
    }

    public static void closeSQLiteStatement(SQLiteStatement sqLiteStatement) {

        if (sqLiteStatement != null) {
            sqLiteStatement.close();
        }
    }

    public static List<String> findAllTableNames(SQLiteDatabase sqLiteDatabase) {

        List<String> tableNames = new ArrayList<String>();
        Cursor cursor = null;
        try {
            cursor = sqLiteDatabase.rawQuery("select * from sqlite_master where type = ?", new String[]{"table"});
            if (cursor.moveToFirst()) {
                do {
                    String tableName = cursor.getString(cursor.getColumnIndexOrThrow("tbl_name"));
                    if (!tableNames.contains(tableName)) {
                        tableNames.add(tableName);
                    }
                } while (cursor.moveToNext());
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            closeCursor(cursor);
        }
        DataLog.e(TAG, sqLiteDatabase.getPath() + tableNames.toString());
        return tableNames;
    }

    public static List<String> findAllTableColumns(String tableName, SQLiteDatabase sqLiteDatabase) {

        List<String> tableColumns = new ArrayList<String>();
        if (TextUtils.isEmpty(tableName)) {
            return tableColumns;
        }
        Cursor cursor = null;
        try {
            String checkingColumnSQL = "pragma table_info(" + tableName + ")";
            cursor = sqLiteDatabase.rawQuery(checkingColumnSQL, null);
            if (cursor.moveToFirst()) {
                do {
                    String columnName = cursor.getString(cursor.getColumnIndexOrThrow("name"));
                    if (!tableColumns.contains(columnName)) {
                        tableColumns.add(columnName);
                    }
                } while (cursor.moveToNext());
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            closeCursor(cursor);
        }
        DataLog.e(TAG, tableName + ": " + tableColumns.toString());
        return tableColumns;
    }

    public static String getCurrentDate() {

        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss", Locale.CHINA);
        return simpleDateFormat.format(new Date());
    }

    public static String getPackageName(Context context) {

        String authorities = "com.android";
        try {
            if (context == null) {
                return authorities;
            }
            PackageManager packageManager = context.getPackageManager();
            authorities = packageManager.getPackageInfo(context.getPackageName(), PackageManager.GET_ACTIVITIES).packageName;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return authorities;
    }

    public static boolean isBasicType(Object field) {

        return field instanceof String || field instanceof Integer || field instanceof Long || field instanceof Double || field instanceof Float || field instanceof Boolean;
    }

    public static SQLiteDatabase openAssetsDatabase(Context context, String assetsPath) {

        String newDatabaseName;
        if (assetsPath.contains("/")) {
            int start = assetsPath.lastIndexOf("/");
            newDatabaseName = assetsPath.substring(start);
        } else {
            newDatabaseName = assetsPath;
        }
        String filePath = context.getFilesDir() + "/" + newDatabaseName;
        try {
            File file = new File(filePath);
            if (file.exists()) {
                return SQLiteDatabase.openOrCreateDatabase(file, null);
            }
            AssetManager assetManager = context.getAssets();
            InputStream inputStream = assetManager.open(assetsPath);
            FileOutputStream fileOutputStream = new FileOutputStream(file);
            byte[] buffer = new byte[1024];
            int count;
            while ((count = inputStream.read(buffer)) > 0) {
                fileOutputStream.write(buffer, 0, count);
            }
            fileOutputStream.flush();
            fileOutputStream.close();
            inputStream.close();
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        }
        return openAssetsDatabase(context, assetsPath);
    }
}