/*
 * Copyright 2015 YAN KA KA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cn.com.utils.data.provider;

import android.content.ContentValues;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.net.Uri;
import android.text.TextUtils;

import cn.com.utils.data.config.DataGlobalConfig;
import cn.com.utils.data.database.DataGlobalHelper;
import cn.com.utils.data.utils.DataLog;
import cn.com.utils.data.utils.DataUtils;

public class DataGlobalProvider extends DataBaseProvider implements DataGlobalConfig {

    private static final String TAG = "DataGlobalProvider";

    private DataGlobalHelper mDataGlobalHelper = null;

    @Override
    public boolean onCreate() {

        if (mDataGlobalHelper == null) {
            mDataGlobalHelper = DataGlobalHelper.getInstance(getContext());
            DataLog.e(TAG, "create DataGlobalHelper");
        }
        return true;
    }

    @Override
    public Cursor query(Uri uri, String[] projection, String selection, String[] selectionArgs, String sortOrder) {

        DataArguments dataArguments = new DataArguments(uri, projection, selection, selectionArgs, sortOrder);
        Cursor cursor = null;
        if (!isValidTable(dataArguments.mTableName)) {
            return null;
        }
        if (mDataGlobalHelper.isValidTable(dataArguments.mTableName)) {
            cursor = query(getContext(), mDataGlobalHelper, dataArguments);
        }
        return cursor;
    }

    @Override
    public String getType(Uri uri) {

        DataArguments args = new DataArguments(uri);
        if (TextUtils.isEmpty(args.mSelection)) {
            return "vnd.android.cursor.dir/" + args.mTableName;
        } else {
            return "vnd.android.cursor.item/" + args.mTableName;
        }
    }

    @Override
    public Uri insert(Uri uri, ContentValues values) {

        values.put(DATE_TIME, DataUtils.getCurrentDate());
        DataArguments dataArguments = new DataArguments(uri, values);
        if (!isValidTable(dataArguments.mTableName)) {
            DataLog.e(TAG, dataArguments.mTableName + " table is not exist");
            mDataGlobalHelper.createDatabaseTable(dataArguments.mTableName, false, dataArguments.mContentValues);
        }
        if (isValidTable(dataArguments.mTableName)) {
            DataLog.e(TAG, dataArguments.mTableName + " table is exist,check the column");
            checkColumn(mDataGlobalHelper, dataArguments);
        }
        if (mDataGlobalHelper.isValidTable(dataArguments.mTableName)) {
            return insert(getContext(), mDataGlobalHelper, dataArguments);
        }
        return null;
    }

    @Override
    public int delete(Uri uri, String selection, String[] selectionArgs) {

        DataArguments dataArguments = new DataArguments(uri, selection, selectionArgs);
        int item = 0;
        if (!isValidTable(dataArguments.mTableName)) {
            return item;
        }
        if (mDataGlobalHelper.isValidTable(dataArguments.mTableName)) {
            return delete(getContext(), mDataGlobalHelper, dataArguments);
        }
        return item;
    }

    @Override
    public int update(Uri uri, ContentValues values, String selection, String[] selectionArgs) {

        values.put(DATE_TIME, DataUtils.getCurrentDate());
        DataArguments dataArguments = new DataArguments(uri, values, selection, selectionArgs);
        int item = 0;
        if (!isValidTable(dataArguments.mTableName)) {
            return item;
        }
        if (mDataGlobalHelper.isValidTable(dataArguments.mTableName)) {
            return update(getContext(), mDataGlobalHelper, dataArguments);
        }
        return item;
    }

    @Override
    public Uri getUriFor(SQLiteDatabase sqLiteDatabase, Uri uri, ContentValues contentValues) {

        if (uri.getPathSegments().size() != 1) {
            throw new IllegalArgumentException("Invalid URI: " + uri);
        }
        String tableName = uri.getPathSegments().get(0);
        if (GLOBAL_KEY_VAL_TABLE_NAME.equals(tableName)) {
            String name = contentValues.getAsString(KEY_NAME);
            return Uri.withAppendedPath(uri, name);
        } else {
            return uri;
        }
    }

    @Override
    public boolean isValidTable(String tableName) {

        return (mDataGlobalHelper.isValidTable(tableName));
    }
}
