/*
 * Copyright 2015 YAN KA KA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cn.com.utils.data.provider;

import android.content.ContentProvider;
import android.content.ContentResolver;
import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteQueryBuilder;
import android.net.Uri;

import cn.com.utils.data.database.DataBaseHelper;
import cn.com.utils.data.utils.DataLog;

public abstract class DataBaseProvider extends ContentProvider {

    private static final String TAG = "DataBaseProvider";

    private ContentResolver mContentResolver = null;

    public Cursor query(Context context, DataBaseHelper dataBaseHelper, DataArguments dataArguments) {

        return query(context, dataBaseHelper.getReadableDatabase(), dataArguments);
    }

    public Cursor query(Context context, SQLiteDatabase sqliteDatabase, DataArguments dataArguments) {

        Cursor cursor = null;
        if (sqliteDatabase == null || dataArguments == null) {
            return null;
        }
        try {
            SQLiteQueryBuilder sqLiteQueryBuilder = new SQLiteQueryBuilder();
            sqLiteQueryBuilder.setTables(dataArguments.mTableName);
            cursor = sqLiteQueryBuilder.query(sqliteDatabase, dataArguments.mProjection, dataArguments.mSelection, dataArguments.mSelectionArgs, null, null, dataArguments.mSortOrder);
            if (mContentResolver == null) {
                mContentResolver = context.getApplicationContext().getContentResolver();
            }
           // cursor.setNotificationUri(mContentResolver, dataArguments.mUri);  //don't notify when query data
        } catch (Exception e) {
            e.printStackTrace();
        }
        return cursor;
    }

    public Uri insert(Context context, DataBaseHelper dataBaseHelper, DataArguments dataArguments) {

        return insert(context, dataBaseHelper.getWritableDatabase(), dataArguments);
    }

    public Uri insert(Context context, SQLiteDatabase sqliteDatabase, DataArguments dataArguments) {

        Uri uri = null;
        if (sqliteDatabase == null || dataArguments == null) {
            return null;
        }
        try {
            long rowId = sqliteDatabase.insert(dataArguments.mTableName, null, dataArguments.mContentValues);
            if (rowId <= 0) {
                return null;
            }
            uri = getUriFor(sqliteDatabase, dataArguments.mUri, dataArguments.mContentValues);
            if (mContentResolver == null) {
                mContentResolver = context.getApplicationContext().getContentResolver();
            }
            if (uri != null) {
                mContentResolver.notifyChange(uri, null);
                DataLog.e(TAG, "insert data,notifyChange uri:" + uri);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return uri;
    }

    public void checkColumn(DataBaseHelper dataBaseHelper, DataArguments dataArguments) {

        DataLog.e(TAG, "checkColumn.............." + dataArguments.mContentValues.keySet().toString());
        try {
            ContentValues contentValues = dataArguments.mContentValues;
            for (String keyName : contentValues.keySet()) {
                boolean isValidColumn = dataBaseHelper.isValidColumn(dataArguments.mTableName, keyName);
                DataLog.e(TAG, keyName + " column in table " + dataArguments.mTableName + " valid is " + isValidColumn);
                if (!isValidColumn) {
                    dataBaseHelper.addDatabaseColumn(dataArguments.mTableName, keyName);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public int delete(Context context, DataBaseHelper dataBaseHelper, DataArguments dataArguments) {

        return delete(context, dataBaseHelper.getWritableDatabase(), dataArguments);
    }

    public int delete(Context context, SQLiteDatabase sqliteDatabase, DataArguments dataArguments) {

		int item = 0;
		if (sqliteDatabase == null || dataArguments == null) {
			return item;
		}
		Uri uri = null;
		try {
			item = sqliteDatabase.delete(dataArguments.mTableName, dataArguments.mSelection, dataArguments.mSelectionArgs);

			if (mContentResolver == null) {
				mContentResolver = context.getApplicationContext().getContentResolver();
			}
			uri = getUriFor(sqliteDatabase, dataArguments.mUri, dataArguments.mContentValues);
			if (uri != null) {
				mContentResolver.notifyChange(uri, null);
				DataLog.e(TAG, "delete data,notifyChange uri:" + uri);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		return item;
	}

    public int update(Context context, DataBaseHelper dataBaseHelper, DataArguments dataArguments) {

        int item = 0;
        if (dataBaseHelper == null || dataArguments == null) {
            return item;
        }
        Uri uri;
        try {
            item = dataBaseHelper.getWritableDatabase().update(dataArguments.mTableName, dataArguments.mContentValues, dataArguments.mSelection, dataArguments.mSelectionArgs);
            uri = getUriFor(dataBaseHelper.getReadableDatabase(), dataArguments.mUri, dataArguments.mContentValues);
            if (mContentResolver == null) {
                mContentResolver = context.getApplicationContext().getContentResolver();
            }
            if (uri != null) {
                mContentResolver.notifyChange(uri, null);
                DataLog.e(TAG, "update data,notifyChange uri:" + uri);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return item;
    }

    public int update(Context context, SQLiteDatabase sqliteDatabase, DataArguments dataArguments) {

        int item = 0;
        if (sqliteDatabase == null || dataArguments == null) {
            return item;
        }
        Uri uri;
        try {
            item = sqliteDatabase.update(dataArguments.mTableName, dataArguments.mContentValues, dataArguments.mSelection, dataArguments.mSelectionArgs);
            uri = getUriFor(sqliteDatabase, dataArguments.mUri, dataArguments.mContentValues);
            if (mContentResolver == null) {
                mContentResolver = context.getApplicationContext().getContentResolver();
            }
            if (uri != null) {
                mContentResolver.notifyChange(uri, null);
                DataLog.e(TAG, "update data,notifyChange uri:" + uri);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return item;
    }

    public abstract Uri getUriFor(SQLiteDatabase sqLiteDatabase, Uri uri, ContentValues contentValues);

    public abstract boolean isValidTable(String tableName);

}
