/*
 * Copyright 2015 YAN KA KA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cn.com.utils.data.manager;

import android.content.ContentProviderOperation;
import android.content.ContentProviderResult;
import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.net.Uri;
import android.text.TextUtils;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import cn.com.utils.data.config.DataGlobalConfig;
import cn.com.utils.data.config.DataMemoryConfig;
import cn.com.utils.data.utils.DataLog;
import cn.com.utils.data.utils.DataUtils;
import cn.com.utils.data.utils.ReflectionUtil;

public class DataManager implements DataGlobalConfig, DataMemoryConfig {

    private static final String TAG = "DataManager";

    private static volatile DataManager mDataManager = null;

    private Context mContext = null;

    private Uri GLOBAL_URI = Uri.parse("content://com.android.data.global.provider/");

    private Uri MEMORY_URI = Uri.parse("content://com.android.data.memory.provider/");

    private DataManager(Context context) {

        mContext = context.getApplicationContext();
        GLOBAL_URI = Uri.parse(String.format("content://%s.data.global.provider/", DataUtils.getPackageName(mContext)));
        MEMORY_URI = Uri.parse(String.format("content://%s.data.memory.provider/", DataUtils.getPackageName(mContext)));
    }

    public void setGlobalUri(String uri) {

        GLOBAL_URI = Uri.parse(String.format("content://%s/", uri));
    }

    public void setMemoryUri(String uri) {

        MEMORY_URI = Uri.parse(String.format("content://%s/", uri));
    }

    public static DataManager getInstance(Context context) {

        synchronized (DataManager.class) {
            if (mDataManager == null) {
                mDataManager = new DataManager(context.getApplicationContext());
            }
            return mDataManager;
        }
    }

    public static String getStringColumnValue(Cursor cursor, String columnName, String defaultValue) {

        if (cursor == null || cursor.getCount() < 0) {
            return defaultValue;
        }
        try {
            return cursor.getString(cursor.getColumnIndex(columnName));
        } catch (Exception e) {
            e.printStackTrace();
        }
        return defaultValue;
    }

    public void regGlobalValuesObserver(String tableName, DataObserver dataObserver, String field) {

        if (dataObserver == null || TextUtils.isEmpty(tableName) || TextUtils.isEmpty(field)) {
            return;
        }
        try {
            Uri tableUri = getUriFor(GLOBAL_URI, tableName);
            Uri fieldUri = getUriFor(tableUri, field);
            DataLog.e(TAG, "reg global observer uri:" + fieldUri);
            mContext.getContentResolver().registerContentObserver(fieldUri, true, dataObserver);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void regGlobalValuesObserver(String tableName, DataObserver dataObserver) {

        if (dataObserver == null || TextUtils.isEmpty(tableName)) {
            return;
        }
        try {
            Uri tableUri = getUriFor(GLOBAL_URI, tableName);
            DataLog.e(TAG, "reg global observer uri:" + tableUri);
            mContext.getContentResolver().registerContentObserver(tableUri, true, dataObserver);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void regGlobalValueObserver(DataObserver dataObserver, String field) {

        regGlobalValuesObserver(GLOBAL_KEY_VAL_TABLE_NAME, dataObserver, field);
    }

    public void regMemoryValuesObserver(String tableName, DataObserver dataObserver, String field) {

        if (dataObserver == null || TextUtils.isEmpty(tableName) || TextUtils.isEmpty(field)) {
            return;
        }
        try {
            Uri tableUri = getUriFor(MEMORY_URI, tableName);
            Uri fieldUri = getUriFor(tableUri, field);
            DataLog.e(TAG, "reg memory observer uri:" + fieldUri);
            mContext.getContentResolver().registerContentObserver(fieldUri, true, dataObserver);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void regMemoryValuesObserver(String tableName, DataObserver dataObserver) {

        if (dataObserver == null || TextUtils.isEmpty(tableName)) {
            return;
        }
        try {
            Uri tableUri = getUriFor(MEMORY_URI, tableName);
            DataLog.e(TAG, "reg memory observer uri:" + tableUri);
            mContext.getContentResolver().registerContentObserver(tableUri, true, dataObserver);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void regMemoryValueObserver(DataObserver dataObserver, String field) {

        regMemoryValuesObserver(MEMORY_KEY_VAL_TABLE_NAME, dataObserver, field);
    }

    public void unRegObserver(DataObserver dataObserver) {

        if (dataObserver == null) {
            return;
        }
        try {
            mContext.getContentResolver().unregisterContentObserver(dataObserver);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private Uri getUriFor(Uri baseUri, String pathSegment) {

        return Uri.withAppendedPath(baseUri, pathSegment);
    }

    public void notifyGlobalValueChange(String field) {

        if (TextUtils.isEmpty(field)) {
            return;
        }
        try {
            Uri baseUri = getUriFor(GLOBAL_URI, GLOBAL_KEY_VAL_TABLE_NAME);
            mContext.getContentResolver().notifyChange(Uri.withAppendedPath(baseUri, field), null);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void notifyMemoryValueChange(String field) {

        if (TextUtils.isEmpty(field)) {
            return;
        }
        try {
            Uri baseUri = getUriFor(MEMORY_URI, MEMORY_KEY_VAL_TABLE_NAME);
            mContext.getContentResolver().notifyChange(Uri.withAppendedPath(baseUri, field), null);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public SQLiteDatabase openAssetsDatabase(String dataBasePath) {

        if (TextUtils.isEmpty(dataBasePath) || !dataBasePath.endsWith("db")) {
            return null;
        }
        try {
            return DataUtils.openAssetsDatabase(mContext, dataBasePath);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    public void showLog(boolean show) {

        DataLog.showLogCat(show);
    }

    public ContentProviderResult[] applyGlobalBatch(ArrayList<ContentProviderOperation> contentProviderOperations) {

        ContentProviderResult[] contentProviderResults = null;
        try {
            contentProviderResults = mContext.getContentResolver().applyBatch(GLOBAL_URI.getAuthority(), contentProviderOperations);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return contentProviderResults;
    }

    public ContentProviderResult[] applyMemoryBatch(ArrayList<ContentProviderOperation> contentProviderOperations) {

        ContentProviderResult[] contentProviderResults = null;
        try {
            contentProviderResults = mContext.getContentResolver().applyBatch(MEMORY_URI.getAuthority(), contentProviderOperations);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return contentProviderResults;
    }

    public void setGlobalValues(String tableName, Cursor cursor) {

        if (TextUtils.isEmpty(tableName) || cursor == null || cursor.getCount() < 0) {
            return;
        }
        try {
            String[] columnNames = cursor.getColumnNames();
            for (cursor.moveToFirst(); !cursor.isAfterLast(); cursor.moveToNext()) {
                ContentValues contentValues = new ContentValues();
                for (String columnName : columnNames) {
                    String columnValue = getStringColumnValue(cursor, columnName, "");
                    contentValues.put(columnName, columnValue);
                }
                setGlobalValues(tableName, contentValues);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            cursor.close();
        }
    }

    public Uri setGlobalValues(String tableName, ContentValues contentValues) {

        Uri uri = null;
        if (TextUtils.isEmpty(tableName) || contentValues == null || contentValues.size() < 0) {
            return null;
        }
        try {
            Uri tableUri = getUriFor(GLOBAL_URI, tableName);
            DataLog.e(TAG, "setGlobalValues uri->" + tableUri);
            uri = mContext.getContentResolver().insert(tableUri, contentValues);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return uri;
    }

    public ContentProviderOperation applySetGlobalValuesBatch(String tableName, ContentValues contentValues) {

        ContentProviderOperation contentProviderOperation = null;
        if (TextUtils.isEmpty(tableName) || contentValues == null || contentValues.size() < 0) {
            return null;
        }
        try {
            Uri tableUri = getUriFor(GLOBAL_URI, tableName);
            ContentProviderOperation.Builder builder = ContentProviderOperation.newInsert(tableUri);
            builder.withValues(contentValues);
            builder.withYieldAllowed(true);
            contentProviderOperation = builder.build();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return contentProviderOperation;
    }

    public <T> ContentProviderOperation applySetGlobalValuesBatch(T model) {

        ContentProviderOperation contentProviderOperation = null;
        try {
            if (model == null) {
                return null;
            }
            Class modelClass = model.getClass();
            Field[] fields = modelClass.getDeclaredFields();
            ContentValues contentValues = new ContentValues();
            for (Field field : fields) {
                String fieldName = field.getName();
                Object fieldValue = ReflectionUtil.getValue(model, fieldName);
                if (DataUtils.isBasicType(fieldValue)) {
                    contentValues.put(fieldName, String.valueOf(fieldValue));
                }
            }
            if (contentValues.size() > 0) {
                return applySetGlobalValuesBatch(modelClass.getSimpleName(), contentValues);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return contentProviderOperation;
    }

    public <T> Uri setGlobalValues(T model) {

        try {
            if (model == null) {
                return null;
            }
            Class modelClass = model.getClass();
            Field[] fields = modelClass.getDeclaredFields();
            ContentValues contentValues = new ContentValues();
            for (Field field : fields) {
                String fieldName = field.getName();
                Object fieldValue = ReflectionUtil.getValue(model, fieldName);
                if (DataUtils.isBasicType(fieldValue)) {
                    contentValues.put(fieldName, String.valueOf(fieldValue));
                }
            }
            if (contentValues.size() > 0) {
                return setGlobalValues(modelClass.getSimpleName(), contentValues);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    public Uri setGlobalValue(ContentValues contentValues) {

        return setGlobalValues(GLOBAL_KEY_VAL_TABLE_NAME, contentValues);
    }

    public void setMemoryValues(String tableName, Cursor cursor) {

        if (TextUtils.isEmpty(tableName) || cursor == null || cursor.getCount() < 0) {
            return;
        }
        try {
            String[] columnNames = cursor.getColumnNames();
            for (cursor.moveToFirst(); !cursor.isAfterLast(); cursor.moveToNext()) {
                ContentValues contentValues = new ContentValues();
                for (String columnName : columnNames) {
                    String columnValue = getStringColumnValue(cursor, columnName, "");
                    contentValues.put(columnName, columnValue);
                }
                setMemoryValues(tableName, contentValues);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            cursor.close();
        }
    }

    public Uri setMemoryValues(String tableName, ContentValues contentValues) {

        Uri uri = null;
        if (TextUtils.isEmpty(tableName) || contentValues == null || contentValues.size() < 0) {
            return null;
        }
        try {
            Uri tableUri = getUriFor(MEMORY_URI, tableName);
            DataLog.e(TAG, "setMemoryValues uri->" + tableUri);
            uri = mContext.getContentResolver().insert(tableUri, contentValues);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return uri;
    }

    public ContentProviderOperation applySetMemoryValuesBatch(String tableName, ContentValues contentValues) {

        ContentProviderOperation contentProviderOperation = null;
        if (TextUtils.isEmpty(tableName) || contentValues == null || contentValues.size() < 0) {
            return null;
        }
        try {
            Uri tableUri = getUriFor(MEMORY_URI, tableName);
            ContentProviderOperation.Builder builder = ContentProviderOperation.newInsert(tableUri);
            builder.withValues(contentValues);
            builder.withYieldAllowed(true);
            contentProviderOperation = builder.build();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return contentProviderOperation;
    }

    public <T> ContentProviderOperation applySetMemoryValuesBatch(T model) {

        ContentProviderOperation contentProviderOperation = null;
        try {
            if (model == null) {
                return null;
            }
            Class modelClass = model.getClass();
            Field[] fields = modelClass.getDeclaredFields();
            ContentValues contentValues = new ContentValues();
            for (Field field : fields) {
                String fieldName = field.getName();
                Object fieldValue = ReflectionUtil.getValue(model, fieldName);
                if (DataUtils.isBasicType(fieldValue)) {
                    contentValues.put(fieldName, String.valueOf(fieldValue));
                }
            }
            if (contentValues.size() > 0) {
                return applySetMemoryValuesBatch(modelClass.getSimpleName(), contentValues);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return contentProviderOperation;
    }

    public <T> Uri setMemoryValues(T model) {

        try {
            if (model == null) {
                return null;
            }
            Class modelClass = model.getClass();
            Field[] fields = modelClass.getDeclaredFields();
            ContentValues contentValues = new ContentValues();
            for (Field field : fields) {
                String fieldName = field.getName();
                Object fieldValue = ReflectionUtil.getValue(model, fieldName);
                if (DataUtils.isBasicType(fieldValue)) {
                    contentValues.put(fieldName, String.valueOf(fieldValue));
                }
            }
            if (contentValues.size() > 0) {
                return setMemoryValues(modelClass.getSimpleName(), contentValues);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    public Uri setMemoryValue(ContentValues contentValues) {

        return setMemoryValues(MEMORY_KEY_VAL_TABLE_NAME, contentValues);
    }

    public int cleanMemoryValue() {

        return deleteMemoryValues(MEMORY_KEY_VAL_TABLE_NAME, null, null);
    }

    public int cleanGlobalValue() {

        return deleteGlobalValues(GLOBAL_KEY_VAL_TABLE_NAME, null, null);
    }

    public int deleteGlobalValues(String tableName, String where, String[] selectionArgs) {

        int item = 0;
        if (TextUtils.isEmpty(tableName)) {
            return item;
        }
        try {
            Uri tableUri = getUriFor(GLOBAL_URI, tableName);
            item = mContext.getContentResolver().delete(tableUri, where, selectionArgs);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return item;
    }

    public ContentProviderOperation applyDeleteGlobalValuesBatch(String tableName, String where, String[] selectionArgs) {

        ContentProviderOperation contentProviderOperation = null;
        if (TextUtils.isEmpty(tableName)) {
            return null;
        }
        try {
            Uri tableUri = getUriFor(GLOBAL_URI, tableName);
            ContentProviderOperation.Builder builder = ContentProviderOperation.newDelete(tableUri);
            builder.withSelection(where, selectionArgs);
            builder.withYieldAllowed(true);
            contentProviderOperation = builder.build();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return contentProviderOperation;
    }

    public <T> ContentProviderOperation applyDeleteGlobalValuesBatch(Class<T> modelClass, String where, String[] selectionArgs) {

        return applyDeleteGlobalValuesBatch(modelClass.getSimpleName(), where, selectionArgs);
    }

    public <T> int deleteGlobalValues(Class<T> modelClass, String where, String[] selectionArgs) {

        return deleteGlobalValues(modelClass.getSimpleName(), where, selectionArgs);
    }

    public int deleteGlobalValue(String where, String[] selectionArgs) {

        return deleteGlobalValues(GLOBAL_KEY_VAL_TABLE_NAME, where, selectionArgs);
    }

    public int deleteMemoryValues(String tableName, String where, String[] selectionArgs) {

        int item = 0;
        if (TextUtils.isEmpty(tableName)) {
            return item;
        }
        try {
            Uri tableUri = getUriFor(MEMORY_URI, tableName);
            item = mContext.getContentResolver().delete(tableUri, where, selectionArgs);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return item;
    }

    public ContentProviderOperation applyDeleteMemoryValuesBatch(String tableName, String where, String[] selectionArgs) {

        ContentProviderOperation contentProviderOperation = null;
        if (TextUtils.isEmpty(tableName)) {
            return null;
        }
        try {
            Uri tableUri = getUriFor(MEMORY_URI, tableName);
            ContentProviderOperation.Builder builder = ContentProviderOperation.newDelete(tableUri);
            builder.withSelection(where, selectionArgs);
            builder.withYieldAllowed(true);
            contentProviderOperation = builder.build();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return contentProviderOperation;
    }

    public <T> ContentProviderOperation applyDeleteMemoryValuesBatch(Class<T> modelClass, String where, String[] selectionArgs) {

        return applyDeleteMemoryValuesBatch(modelClass.getSimpleName(), where, selectionArgs);
    }

    public <T> int deleteMemoryValues(Class<T> modelClass, String where, String[] selectionArgs) {

        return deleteMemoryValues(modelClass.getSimpleName(), where, selectionArgs);
    }

    public int deleteMemoryValue(String where, String[] selectionArgs) {

        return deleteMemoryValues(MEMORY_KEY_VAL_TABLE_NAME, where, selectionArgs);
    }

    public Cursor queryGlobalValues(String tableName, String[] projection, String selection, String[] selectionArgs, String sortOrder) {

        Cursor cursor = null;
        if (TextUtils.isEmpty(tableName)) {
            return null;
        }
        try {
            Uri tableUri = getUriFor(GLOBAL_URI, tableName);
            DataLog.e(TAG, "query uri:" + tableUri.toString());
            cursor = mContext.getContentResolver().query(tableUri, projection, selection, selectionArgs, sortOrder);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return cursor;
    }

    public Cursor queryGlobalValue(String[] projection, String selection, String[] selectionArgs, String sortOrder) {

        return queryGlobalValues(GLOBAL_KEY_VAL_TABLE_NAME, projection, selection, selectionArgs, sortOrder);
    }

    public Cursor queryMemoryValues(String tableName, String[] projection, String selection, String[] selectionArgs, String sortOrder) {

        Cursor cursor = null;
        if (TextUtils.isEmpty(tableName)) {
            return null;
        }
        try {
            Uri tableUri = getUriFor(MEMORY_URI, tableName);
            cursor = mContext.getContentResolver().query(tableUri, projection, selection, selectionArgs, sortOrder);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return cursor;
    }

    public Cursor queryMemoryValue(String[] projection, String selection, String[] selectionArgs, String sortOrder) {

        return queryMemoryValues(MEMORY_KEY_VAL_TABLE_NAME, projection, selection, selectionArgs, sortOrder);
    }

    public List<Map<String, String>> getGlobalValues(String tableName, String[] projection, String selection, String[] selectionArgs, String sortOrder) {

        Cursor cursor = queryGlobalValues(tableName, projection, selection, selectionArgs, sortOrder);
        List<Map<String, String>> list = new ArrayList<Map<String, String>>();
        if (cursor == null) {
            return list;
        }
        try {
            String[] columnNames = cursor.getColumnNames();
            for (cursor.moveToFirst(); !cursor.isAfterLast(); cursor.moveToNext()) {
                Map<String, String> columnMap = new HashMap<String, String>();
                for (String columnName : columnNames) {
                    String columnValue = getStringColumnValue(cursor, columnName, "");
                    DataLog.e(TAG, "columnName =" + columnName + "    columnValue=" + columnValue);
                    columnMap.put(columnName, columnValue);
                }
                list.add(columnMap);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            cursor.close();
        }
        return list;
    }

    public <T> List<T> getGlobalValues(Class<T> modelClass, String[] projection, String selection, String[] selectionArgs, String sortOrder) {

        Cursor cursor = queryGlobalValues(modelClass.getSimpleName(), projection, selection, selectionArgs, sortOrder);
        List<T> list = new ArrayList<T>();
        if (cursor == null) {
            return list;
        }
        try {
            String[] columnNames = cursor.getColumnNames();
            for (cursor.moveToFirst(); !cursor.isAfterLast(); cursor.moveToNext()) {
                T model = modelClass.newInstance();
                for (String columnName : columnNames) {
                    String columnValue = getStringColumnValue(cursor, columnName, "");
                    ReflectionUtil.setValue(model, columnName, columnValue);
                }
                list.add(model);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            cursor.close();
        }
        return list;
    }

    public List<Map<String, String>> getMemoryValues(String tableName, String[] projection, String selection, String[] selectionArgs, String sortOrder) {

        Cursor cursor = queryMemoryValues(tableName, projection, selection, selectionArgs, sortOrder);
        List<Map<String, String>> list = new ArrayList<Map<String, String>>();
        if (cursor == null) {
            return list;
        }
        try {
            String[] columnNames = cursor.getColumnNames();
            for (cursor.moveToFirst(); !cursor.isAfterLast(); cursor.moveToNext()) {
                Map<String, String> columnMap = new HashMap<String, String>();
                for (String columnName : columnNames) {
                    String columnValue = getStringColumnValue(cursor, columnName, "");
                    columnMap.put(columnName, columnValue);
                }
                list.add(columnMap);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            cursor.close();
        }
        return list;
    }

    public <T> List<T> getMemoryValues(Class<T> modelClass, String[] projection, String selection, String[] selectionArgs, String sortOrder) {

        Cursor cursor = queryMemoryValues(modelClass.getSimpleName(), projection, selection, selectionArgs, sortOrder);
        List<T> list = new ArrayList<T>();
        if (cursor == null) {
            return list;
        }
        try {
            String[] columnNames = cursor.getColumnNames();
            for (cursor.moveToFirst(); !cursor.isAfterLast(); cursor.moveToNext()) {
                T model = modelClass.newInstance();
                for (String columnName : columnNames) {
                    String columnValue = getStringColumnValue(cursor, columnName, "");
                    ReflectionUtil.setValue(model, columnName, columnValue);
                }
                list.add(model);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            cursor.close();
        }
        return list;
    }

    public String getGlobalValue(String key, String columnName, String defaultColumnValue) {

        Cursor cursor = queryGlobalValue(null, KEY_NAME + "='" + key + "'", null, null);
        if (cursor == null || cursor.getCount() < 0) {
            return defaultColumnValue;
        }
        try {
            if (cursor.moveToFirst()) {
                return getStringColumnValue(cursor, columnName, defaultColumnValue);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            cursor.close();
        }
        return defaultColumnValue;
    }

    public String getGlobalValue(String key, String defaultValue) {

        return getGlobalValue(key, VALUE_NAME, defaultValue);
    }

    public String getMemoryValue(String key, String columnName, String defaultColumnValue) {

        Cursor cursor = queryMemoryValue(null, KEY_NAME + "='" + key + "'", null, null);
        if (cursor == null || cursor.getCount() < 0) {
            return defaultColumnValue;
        }
        try {
            if (cursor.moveToFirst()) {
                return getStringColumnValue(cursor, columnName, defaultColumnValue);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            cursor.close();
        }
        return defaultColumnValue;
    }

    public String getMemoryValue(String key, String defaultValue) {

        return getMemoryValue(key, VALUE_NAME, defaultValue);
    }

    public void setGlobalValue(String key, String value) {

        if (TextUtils.isEmpty(key) || TextUtils.isEmpty(value)) {
            return;
        }
        ContentValues contentValues = new ContentValues();
        contentValues.put(KEY_NAME, key);
        contentValues.put(VALUE_NAME, value);
        setGlobalValue(contentValues);
    }

    public void setMemoryValue(String key, String value) {

        if (TextUtils.isEmpty(key) || TextUtils.isEmpty(value)) {
            return;
        }
        ContentValues contentValues = new ContentValues();
        contentValues.put(KEY_NAME, key);
        contentValues.put(VALUE_NAME, value);
        setMemoryValue(contentValues);
    }

    public int updateGlobalValues(String tableName, ContentValues contentValues, String where, String[] selectionArgs) {

        int item = 0;
        if (TextUtils.isEmpty(tableName) || contentValues == null || contentValues.size() < 0) {
            return item;
        }
        try {
            Uri tableUri = getUriFor(GLOBAL_URI, tableName);
            item = mContext.getContentResolver().update(tableUri, contentValues, where, selectionArgs);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return item;
    }

    public ContentProviderOperation applyUpdateGlobalValuesBatch(String tableName, ContentValues contentValues, String where, String[] selectionArgs) {

        ContentProviderOperation contentProviderOperation = null;
        if (TextUtils.isEmpty(tableName) || contentValues == null || contentValues.size() < 0) {
            return null;
        }
        try {
            Uri tableUri = getUriFor(GLOBAL_URI, tableName);
            ContentProviderOperation.Builder builder = ContentProviderOperation.newUpdate(tableUri);
            builder.withSelection(where, selectionArgs);
            builder.withValues(contentValues);
            builder.withYieldAllowed(true);
            contentProviderOperation = builder.build();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return contentProviderOperation;
    }

    public <T> ContentProviderOperation applyUpdateGlobalValuesBatch(T model, String where, String[] selectionArgs) {

        ContentProviderOperation contentProviderOperation = null;
        try {
            if (model == null) {
                return contentProviderOperation;
            }
            Class modelClass = model.getClass();
            Field[] fields = modelClass.getDeclaredFields();
            ContentValues contentValues = new ContentValues();
            for (Field field : fields) {
                String fieldName = field.getName();
                Object fieldValue = ReflectionUtil.getValue(model, fieldName);
                if (DataUtils.isBasicType(fieldValue)) {
                    contentValues.put(fieldName, String.valueOf(fieldValue));
                }
            }
            if (contentValues.size() > 0) {
                return applyUpdateGlobalValuesBatch(modelClass.getSimpleName(), contentValues, where, selectionArgs);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        return contentProviderOperation;
    }

    public <T> int updateGlobalValues(T model, String where, String[] selectionArgs) {

        try {
            if (model == null) {
                return 0;
            }
            Class modelClass = model.getClass();
            Field[] fields = modelClass.getDeclaredFields();
            ContentValues contentValues = new ContentValues();
            for (Field field : fields) {
                String fieldName = field.getName();
                Object fieldValue = ReflectionUtil.getValue(model, fieldName);
                if (DataUtils.isBasicType(fieldValue)) {
                    contentValues.put(fieldName, String.valueOf(fieldValue));
                }
            }
            if (contentValues.size() > 0) {
                return updateGlobalValues(modelClass.getSimpleName(), contentValues, where, selectionArgs);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return 0;
    }

    public int updateGlobalValue(ContentValues contentValues, String where, String[] selectionArgs) {

        return updateGlobalValues(GLOBAL_KEY_VAL_TABLE_NAME, contentValues, where, selectionArgs);
    }

    public int updateMemoryValues(String tableName, ContentValues contentValues, String where, String[] selectionArgs) {

        int item = 0;
        if (TextUtils.isEmpty(tableName) || contentValues == null || contentValues.size() < 0) {
            return item;
        }
        try {
            Uri tableUri = getUriFor(MEMORY_URI, tableName);
            item = mContext.getContentResolver().update(tableUri, contentValues, where, selectionArgs);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return item;
    }

    public ContentProviderOperation applyUpdateMemoryValuesBatch(String tableName, ContentValues contentValues, String where, String[] selectionArgs) {

        ContentProviderOperation contentProviderOperation = null;
        if (TextUtils.isEmpty(tableName) || contentValues == null || contentValues.size() < 0) {
            return null;
        }
        try {
            Uri tableUri = getUriFor(MEMORY_URI, tableName);
            ContentProviderOperation.Builder builder = ContentProviderOperation.newUpdate(tableUri);
            builder.withSelection(where, selectionArgs);
            builder.withValues(contentValues);
            builder.withYieldAllowed(true);
            contentProviderOperation = builder.build();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return contentProviderOperation;
    }

    public <T> ContentProviderOperation applyUpdateMemoryValuesBatch(T model, String where, String[] selectionArgs) {

        ContentProviderOperation contentProviderOperation = null;
        try {
            if (model == null) {
                return contentProviderOperation;
            }
            Class modelClass = model.getClass();
            Field[] fields = modelClass.getDeclaredFields();
            ContentValues contentValues = new ContentValues();
            for (Field field : fields) {
                String fieldName = field.getName();
                Object fieldValue = ReflectionUtil.getValue(model, fieldName);
                if (DataUtils.isBasicType(fieldValue)) {
                    contentValues.put(fieldName, String.valueOf(fieldValue));
                }
            }
            if (contentValues.size() > 0) {
                return applyUpdateMemoryValuesBatch(modelClass.getSimpleName(), contentValues, where, selectionArgs);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return contentProviderOperation;
    }

    public <T> int updateMemoryValues(T model, String where, String[] selectionArgs) {

        try {
            if (model == null) {
                return 0;
            }
            Class modelClass = model.getClass();
            Field[] fields = modelClass.getDeclaredFields();
            ContentValues contentValues = new ContentValues();
            for (Field field : fields) {
                String fieldName = field.getName();
                Object fieldValue = ReflectionUtil.getValue(model, fieldName);
                if (DataUtils.isBasicType(fieldValue)) {
                    contentValues.put(fieldName, String.valueOf(fieldValue));
                }
            }
            if (contentValues.size() > 0) {
                return updateMemoryValues(modelClass.getSimpleName(), contentValues, where, selectionArgs);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return 0;
    }

    public int updateMemoryValue(ContentValues contentValues, String where, String[] selectionArgs) {

        return updateMemoryValues(MEMORY_KEY_VAL_TABLE_NAME, contentValues, where, selectionArgs);
    }

    public void setMemorySerializable(String key, Serializable serializableObject) {

        ByteArrayOutputStream byteArrayOutputStream = null;
        ObjectOutputStream objectOutputStream = null;
        try {
            if (TextUtils.isEmpty(key) || serializableObject == null) {
                return;
            }
            byteArrayOutputStream = new ByteArrayOutputStream();
            objectOutputStream = new ObjectOutputStream(byteArrayOutputStream);
            objectOutputStream.writeObject(serializableObject);
            objectOutputStream.flush();
            byte[] data = byteArrayOutputStream.toByteArray();
            if (data != null && data.length > 0) {
                ContentValues contentValues = new ContentValues();
                contentValues.put(KEY_NAME, key);
                contentValues.put(VALUE_NAME, data);
                setMemoryValues(MEMORY_SERIALIZABLE_TABLE_NAME, contentValues);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (byteArrayOutputStream != null) {
                try {
                    byteArrayOutputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
            if (objectOutputStream != null) {
                try {
                    objectOutputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    public Object getMemorySerializable(String key) {

        Object object = null;
        ByteArrayInputStream byteArrayInputStream = null;
        ObjectInputStream objectInputStream = null;
        try {
            Cursor cursor = queryMemoryValues(MEMORY_SERIALIZABLE_TABLE_NAME, null, KEY_NAME + "='" + key + "'", null, null);
            if (cursor == null || cursor.getCount() < 0) {
                return null;
            }
            byte[] data = null;
            if (cursor.moveToFirst()) {
                data = cursor.getBlob(cursor.getColumnIndex(VALUE_NAME));
            }
            if (data != null && data.length > 0) {
                byteArrayInputStream = new ByteArrayInputStream(data);
                objectInputStream = new ObjectInputStream(byteArrayInputStream);
                object = objectInputStream.readObject();
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (byteArrayInputStream != null) {
                try {
                    byteArrayInputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
            if (objectInputStream != null) {
                try {
                    objectInputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return object;
    }

    public void setGlobalSerializable(String key, Serializable serializableObject) {

        ByteArrayOutputStream byteArrayOutputStream = null;
        ObjectOutputStream objectOutputStream = null;
        try {
            if (TextUtils.isEmpty(key) || serializableObject == null) {
                return;
            }
            byteArrayOutputStream = new ByteArrayOutputStream();
            objectOutputStream = new ObjectOutputStream(byteArrayOutputStream);
            objectOutputStream.writeObject(serializableObject);
            objectOutputStream.flush();
            byte[] data = byteArrayOutputStream.toByteArray();
            if (data != null && data.length > 0) {
                ContentValues contentValues = new ContentValues();
                contentValues.put(KEY_NAME, key);
                contentValues.put(VALUE_NAME, data);
                setGlobalValues(GLOBAL_SERIALIZABLE_TABLE_NAME, contentValues);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (byteArrayOutputStream != null) {
                try {
                    byteArrayOutputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
            if (objectOutputStream != null) {
                try {
                    objectOutputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    public Object getGlobalSerializable(String key) {

        Object object = null;
        ByteArrayInputStream byteArrayInputStream = null;
        ObjectInputStream objectInputStream = null;
        try {
            Cursor cursor = queryGlobalValues(GLOBAL_SERIALIZABLE_TABLE_NAME, null, KEY_NAME + "='" + key + "'", null, null);
            if (cursor == null || cursor.getCount() < 0) {
                return null;
            }
            byte[] data = null;
            if (cursor.moveToFirst()) {
                data = cursor.getBlob(cursor.getColumnIndex(VALUE_NAME));
            }
            if (data != null && data.length > 0) {
                byteArrayInputStream = new ByteArrayInputStream(data);
                objectInputStream = new ObjectInputStream(byteArrayInputStream);
                object = objectInputStream.readObject();
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (byteArrayInputStream != null) {
                try {
                    byteArrayInputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
            if (objectInputStream != null) {
                try {
                    objectInputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return object;
    }

    public int keepGlobalValuesCount(String tableName, int count) {

        int item = 0;
        try {
            List<Map<String, String>> list = getGlobalValues(tableName, null, null, null, DATE_TIME + " desc");
            int items = count - 1;
            if (list != null && list.size() > items) {
                Map<String, String> map = list.get(items);
                if (map != null) {
                    String value = map.get(DATE_TIME);
                    if (!TextUtils.isEmpty(value)) {
                        DataLog.e(TAG, "delete global values where " + DATE_TIME + " > " + value);
                        item = deleteGlobalValues(tableName, DATE_TIME + ">?", new String[]{value});
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return item;
    }

    public <T> int keepGlobalValuesCount(T model, int count) {

        if (model == null) {
            return 0;
        }
        Class modelClass = model.getClass();
        return keepGlobalValuesCount(modelClass.getSimpleName(), count);
    }

    public int keepMemoryValuesCount(String tableName, int count) {

        int item = 0;
        try {
            List<Map<String, String>> list = getMemoryValues(tableName, null, null, null, DATE_TIME + " desc");
            int items = count - 1;
            if (list != null && list.size() > items) {
                Map<String, String> map = list.get(items);
                if (map != null) {
                    String value = map.get(DATE_TIME);
                    if (!TextUtils.isEmpty(value)) {
                        DataLog.e(TAG, "delete memory values where " + DATE_TIME + " > " + value);
                        item = deleteMemoryValues(tableName, DATE_TIME + ">?", new String[]{value});
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return item;
    }

    public <T> int keepMemoryValuesCount(T model, int count) {

        if (model == null) {
            return 0;
        }
        Class modelClass = model.getClass();
        return keepMemoryValuesCount(modelClass.getSimpleName(), count);
    }
}