/*
 * Copyright 2015 YAN KA KA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cn.com.utils.data.database;

import android.content.ContentValues;
import android.content.Context;

import cn.com.utils.data.utils.DataLog;
import cn.com.utils.data.utils.DataUtils;

import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteDatabase.CursorFactory;
import android.database.sqlite.SQLiteOpenHelper;
import android.database.sqlite.SQLiteStatement;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

public abstract class DataBaseHelper extends SQLiteOpenHelper {

    private static final String TAG = "DataBaseHelper";

    private static final int DEFAULT_DATABASE_VERSION = 1;

    private final Map<String, List<String>> mColumnNameMap = new HashMap<String, List<String>>();

    private List<String> mTableNameList = new ArrayList<String>();

    private final Map<String, String> mDefaultKeyValMap = new HashMap<String, String>();

    public DataBaseHelper(Context context) {

        super(context, null, null, DEFAULT_DATABASE_VERSION);
    }

    public DataBaseHelper(Context context, int version) {

        super(context, null, null, version);
    }

    public DataBaseHelper(Context context, String name, int version) {

        super(context, name, null, version);
    }

    public DataBaseHelper(Context context, String name, CursorFactory factory, int version) {

        super(context, name, factory, version, null);
    }

    public boolean isValidTable(String tableName) {

        if (mTableNameList.contains(tableName)) {
            return true;
        } else {
            List<String> tableNameList = DataUtils.findAllTableNames(getReadableDatabase());
            if (tableNameList.contains(tableName)) {
                mTableNameList = tableNameList;
                return true;
            } else {
                return false;
            }
        }
    }

    public boolean isValidColumn(String tableName, String columnName) {

        List<String> columnList = mColumnNameMap.get(tableName);
        if (columnList == null || columnList.isEmpty()) {
            columnList = DataUtils.findAllTableColumns(tableName, getReadableDatabase());
            if (columnList != null) {
                mColumnNameMap.put(tableName, columnList);
            }
        }

        if (columnList == null) {
            DataLog.e(TAG, "columnList == null ,return false");
            return false;
        } else if (columnList.contains(columnName)) {
            DataLog.e(TAG, "columnList.contains(columnName) ,return true");
            return true;
        } else {
            List<String> columnNameList = DataUtils.findAllTableColumns(tableName, getReadableDatabase());
            if (columnNameList == null || columnName.isEmpty()) {
                DataLog.e(TAG, "columnNameList == null || columnName.isEmpty() ,return false");
                return false;
            }
            if (columnNameList.contains(columnName)) {
                DataLog.e(TAG, "columnNameList.contains(tableName) ,return true");
                mColumnNameMap.put(tableName, columnNameList);
                return true;
            } else {
                DataLog.e(TAG, "else ,return false");
                return false;
            }
        }
    }

    public void createDatabaseTable(String tableName, boolean autoIncrementId, ContentValues contentValues) {

        if (tableName == null || tableName.length() < 1 || contentValues == null) {
            return;
        }
        try {
            getWritableDatabase().execSQL(DataUtils.getCreateTableSQL(tableName, autoIncrementId, contentValues));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void addDatabaseColumn(String tableName, String columnName) {

        if (tableName == null || tableName.length() < 1 || columnName == null || columnName.length() < 1) {
            return;
        }
        try {
            getWritableDatabase().execSQL(DataUtils.addColumnSQL(tableName, columnName));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void createDatabaseTable(SQLiteDatabase sqLiteDatabase, String tableName, String keyName, String valueName, String indexName) {

        DataLog.e(TAG, "prepare create database key val table ........");
        if (tableName == null || tableName.length() < 1 || keyName == null || keyName.length() < 1 || indexName == null || indexName.length() < 1) {
            DataLog.e(TAG, "create database key val table is return");
            return;
        }
        try {
            // create table
            sqLiteDatabase.execSQL(DataUtils.getCreateTableSQL(tableName, false, keyName, valueName));
            DataLog.e(TAG, "create key val table ->" + tableName);
            // create index
            sqLiteDatabase.execSQL(DataUtils.getCreateIndexSQL(indexName, tableName, valueName));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
    public void createDatabaseBlobTable(SQLiteDatabase sqLiteDatabase, String tableName, String keyName, String valueName, String indexName) {

        DataLog.e(TAG, "prepare create database key val table ........");
        if (tableName == null || tableName.length() < 1 || keyName == null || keyName.length() < 1 || indexName == null || indexName.length() < 1) {
            DataLog.e(TAG, "create database key val table is return");
            return;
        }
        try {
            // create table
            sqLiteDatabase.execSQL(DataUtils.getCreateBlobTableSQL(tableName, false, keyName, valueName));
            DataLog.e(TAG, "create key val table ->" + tableName);
            // create index
            sqLiteDatabase.execSQL(DataUtils.getCreateIndexSQL(indexName, tableName, valueName));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void addDefaultKeyVal(String key, String value) {

        if (mDefaultKeyValMap != null) {
            mDefaultKeyValMap.put(key, value);
            DataLog.e(TAG, "put default key->" + key + " val->" + value);
        }
    }

    public void loadDefaultKeyVal(SQLiteDatabase sqLiteDatabase, String tableName, String keyName, String valueName, String dateTime) {

        if (mDefaultKeyValMap == null) {
            return;
        }
        SQLiteStatement sqLiteStatement;
        try {
            sqLiteStatement = sqLiteDatabase.compileStatement(String.format("insert or ignore into %s(%s,%s,%s) values(?,?,?);", tableName, keyName, valueName, dateTime));
            Set<String> keySet = mDefaultKeyValMap.keySet();
            for (String keyStr : keySet) {
                loadDefaultKeyVal(sqLiteStatement, keyStr, mDefaultKeyValMap.get(keyStr), DataUtils.getCurrentDate());
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void loadDefaultKeyVal(SQLiteStatement sqLiteStatement, String key, String value, String dataTime) {

        if (sqLiteStatement != null) {
            sqLiteStatement.bindString(1, key);
            sqLiteStatement.bindString(2, value);
            sqLiteStatement.bindString(3, dataTime);
            sqLiteStatement.execute();
        }
    }
}
