package cn.com.startai.fssdk.download.dfs;


import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.ConnectException;
import java.net.InetSocketAddress;

import cn.com.startai.fssdk.BaseDownloader;
import cn.com.startai.fssdk.StartaiDownloaderManager;
import cn.com.startai.fssdk.db.FDBManager;
import cn.com.startai.fssdk.db.entity.DownloadBean;
import cn.com.startai.fssdk.event.FSEventDispatcher;
import cn.com.startai.fssdk.utils.FFileUtils;
import cn.com.startai.fssdk.utils.FLog;
import cn.com.startai.fssdk.FSDownloadCallback;
import cn.com.startai.fssdk.common.MyException;
import cn.com.startai.fssdk.fastdfs.ClientGlobal;
import cn.com.startai.fssdk.fastdfs.DownloadCallback;
import cn.com.startai.fssdk.fastdfs.FileInfo;
import cn.com.startai.fssdk.fastdfs.StorageClient1;
import cn.com.startai.fssdk.fastdfs.StorageServer;
import cn.com.startai.fssdk.fastdfs.TrackerClient;
import cn.com.startai.fssdk.fastdfs.TrackerGroup;
import cn.com.startai.fssdk.fastdfs.TrackerServer;
import cn.com.startai.fssdk.utils.FUtils;
import cn.com.startai.fssdk.utils.NetAddrUtil;


/**
 * 下载器
 */
public class StartaiDownloader extends BaseDownloader implements Runnable {

    private BufferedOutputStream out;
    private final DownloadBean downloadEntity;
    private FSDownloadCallback callback;
    private StorageClient1 client;

    private long max_cache_size = 1024 * 512; // 每当下载到达数 就更新进度
    private int currCacheSize; // 当前cache数

    private boolean flagIsDownloading = true;

    @Override
    public DownloadBean getDownloadBean() {
        return downloadEntity;
    }

    @Override
    public int getDownloadStatus() {
        return 0;
    }

    public void stopDownload() {
        FLog.d(TAG, "stopDownload");
        flagIsDownloading = false;
        close();
        onPause(downloadEntity);
    }


    @Override
    public void onWaiting(final DownloadBean downloadBean) {


        StartaiDownloaderManager.getInstance().runInTHread(new Runnable() {
            @Override
            public void run() {
                downloadBean.setStatus(3);
                FDBManager.getInstance().addOrUpdateDownloadBean(downloadBean);
                if (callback != null) {
                    callback.onWaiting(downloadBean);
                }else{
                    FSEventDispatcher.getInstance().onDownloadWaiting(downloadBean);
                }
            }
        });
    }

    @Override
    public void onPause(final DownloadBean downloadBean) {


        StartaiDownloaderManager.getInstance().runInTHread(new Runnable() {
            @Override
            public void run() {
                downloadBean.setStatus(0);
                FDBManager.getInstance().addOrUpdateDownloadBean(downloadBean);
                if (callback != null) {
                    callback.onPause(downloadBean);
                } else {
                    FSEventDispatcher.getInstance().onDownloadPause(downloadBean);
                }
                StartaiDownloaderManager.getInstance().toDownload();
            }
        });
    }


    @Override
    public void onSuccess(final DownloadBean entity) {

        StartaiDownloaderManager.getInstance().runInTHread(new Runnable() {

            @Override
            public void run() {
                entity.setStatus(2);
                FDBManager.getInstance().addOrUpdateDownloadBean(entity);
                FFileUtils.rename(entity.getLocalPath() + ".tmp", entity.getFileName());
                if (callback != null) {
                    callback.onSuccess(entity);
                } else {
                    FSEventDispatcher.getInstance().onDownloadSuccess(entity);
                }
                StartaiDownloaderManager.getInstance().toDownload();
            }
        });
    }


    @Override
    public synchronized void onFailue(final DownloadBean entity, final int errorCode) {

        StartaiDownloaderManager.getInstance().runInTHread(new Runnable() {
            @Override
            public void run() {

                entity.setStatus(4);
                FDBManager.getInstance().addOrUpdateDownloadBean(entity);
                if (callback != null) {
                    callback.onFailure(entity, errorCode);
                } else {
                    FSEventDispatcher.getInstance().onDownloadFailure(entity, errorCode);
                }
                StartaiDownloaderManager.getInstance().toDownload();
            }
        });


    }

    @Override
    public synchronized void onProgress(final DownloadBean entity) {
        FLog.d(TAG, "progress = " + entity.getAddedSize() + "/" + entity.getTotalSize());


        StartaiDownloaderManager.getInstance().runInTHread(new Runnable() {
            @Override
            public void run() {
                entity.setStatus(1);
                FDBManager.getInstance().addOrUpdateDownloadBean(entity);
                if (callback != null) {
                    callback.onProgress(entity);
                } else {
                    FSEventDispatcher.getInstance().onDownloadProgress(entity);
                }
            }
        });

    }

    @Override
    public synchronized void onStart(final DownloadBean entity) {

        StartaiDownloaderManager.getInstance().runInTHread(new Runnable() {
            @Override
            public void run() {
                entity.setStatus(1);
                FDBManager.getInstance().addOrUpdateDownloadBean(entity);
                if (callback != null) {

                    callback.onStart(entity);
                } else {
                    FSEventDispatcher.getInstance().onDownloadStart(entity);
                }
            }
        });

    }


    public StartaiDownloader(DownloadBean entity, FSDownloadCallback callback) {
        this.callback = callback;
        downloadEntity = entity;
    }

    @Override
    public void run() {
        //初始化下载参数
        initClientGlobal(NetAddrUtil.getIpFromUrl(downloadEntity.getUrl()), NetAddrUtil.getIpPortFromUrl(downloadEntity.getUrl()).port);

        //获取 下载对象
        if (client == null) {
            try {
                client = getStorageClient1();
            } catch (IOException e) {

                e.printStackTrace();
                onFailue(downloadEntity, ERROR_CODE_CONN_SERVER_FAILED);
                return;
            }
        }

        if (client == null) {
            onFailue(downloadEntity, ERROR_CODE_CONN_SERVER_FAILED);
            return;
        }
        FileInfo fileInfo = null;
        try {
            fileInfo = client.get_file_info1(downloadEntity.getFileId());
        } catch (Exception e) {
            onFailue(downloadEntity, ERROR_CODE_GET_FILEINFO_ERROR);
            return;
        }
        if (fileInfo == null || fileInfo.getFileSize() == 0) {
            onFailue(downloadEntity, ERROR_CODE_GET_FILEINFO_ERROR);
            return;
        }

        downloadEntity.setTotalSize(fileInfo.getFileSize());

        // 每当缓存下载量到 1%的时候 更新进度并清空缓存重新开始缓存


        if (downloadEntity.getTotalSize() < 1024 * 1024 * 2) {
            //如果是2M以下的文件直接下载 不回调进度
            max_cache_size = downloadEntity.getTotalSize();
        } else if (downloadEntity.getTotalSize() < 1024 * 1024 * 20) {
            //如果是20M以下 每10%回进度
            max_cache_size = (long) (downloadEntity.getTotalSize() * 1.0 / 10);
        } else {
            //如果是20M以上 每 1%回进度
            max_cache_size = (int) (downloadEntity.getTotalSize() * 1.0 / 100);
        }

        downloadEntity.setProgress((int) (downloadEntity.getAddedSize() * 100.0 / downloadEntity.getTotalSize()));

        // 开始下载
        if (downloadEntity.getAddedSize() > 0) {
            FLog.d(TAG, "断点续传... 已完成 " + downloadEntity.getProgress() + "%" + downloadEntity.getFileId());

        } else {
            FLog.d(TAG, "文件大小为0 开始下载..." + downloadEntity.getFileId());

        }
        currCacheSize = 0;


        //创建本地文件输出流
        try {
            File saveFileTemp = new File(downloadEntity.getLocalPath() + ".tmp");
            if (!saveFileTemp.exists()) {
                FUtils.mkDir(saveFileTemp.getAbsolutePath());
            }

            out = new BufferedOutputStream(new FileOutputStream(saveFileTemp, true));
        } catch (FileNotFoundException e) {
            e.printStackTrace();
            onFailue(downloadEntity, ERROR_CODE_LOCAL_PATH_NOT_EXIT);
            return;
        } catch (IOException e1) {
            e1.printStackTrace();
            onFailue(downloadEntity, ERROR_CODE_LOCAL_SAVE_FILE_EXCPETION);
            return;
        }


        try {

            int result = client.download_file1(downloadEntity.getFileId(), downloadEntity.getAddedSize(), 0, new DownloadCallback() {

                @Override
                public int recv(long file_size, byte[] data, int bytes) {

                    try {


                        out.write(data, 0, bytes);

                        downloadEntity.setAddedSize(downloadEntity.getAddedSize() + bytes);
                        currCacheSize += bytes;
                        // 计算下载进度
                        int progress = (int) (downloadEntity.getAddedSize() * 100.0 / downloadEntity.getTotalSize());

                        downloadEntity.setProgress(progress);


                        // 每当缓存下载量到 一定量的时候 更新进度并清空缓存重新开始缓存
                        if (currCacheSize >= max_cache_size && progress != 100) {
                            currCacheSize = 0;
                            // 更新进度
                            onProgress(downloadEntity);
                        }
                        if (downloadEntity.getAddedSize() == downloadEntity.getTotalSize()) {

                            if (out != null) {
                                try {
                                    out.flush();
                                    out.close();
                                } catch (IOException e) {
                                    // e.printStackTrace();
                                }
                            }

                            // 更新进度
                            onProgress(downloadEntity);
                            // 文件下载完成
                            onSuccess(downloadEntity);
                            return 0;
                        } else if (downloadEntity.getAddedSize() > downloadEntity.getTotalSize()) {
                            //文件下载异常
                            new File(downloadEntity.getLocalPath()).delete();
                            onFailue(downloadEntity, ERROR_CODE_FILE_TO_LARGE); //文件下载异常
                            return 1;
                        }

                    } catch (IOException e) {
                        e.printStackTrace();
                        onFailue(downloadEntity, ERROR_CODE_WRITE_FILE_FAILED);
                        return 1;
                    }
                    return 0;
                }
            });


        } catch (ConnectException e) {
            e.printStackTrace();
            onFailue(downloadEntity, ERROR_CODE_CONN_SERVER_FAILED);
        } catch (IOException e) {
            if (!flagIsDownloading) {
                //如果主动暂停也会触发此异常需要处理
                return;
            }
            e.printStackTrace();
            onFailue(downloadEntity, ERROR_CODE_NETWORK_UNVALIBLE);
        } catch (MyException e) {
            e.printStackTrace();
            onFailue(downloadEntity, ERROR_CODE_DOWNLOAD_EXCEPTION);
        } catch (Exception e) {
            e.printStackTrace();
            onFailue(downloadEntity, ERROR_CODE_UNKNOW_EXCEPTION);
        } finally {
            if (out != null) {
                try {
                    out.flush();
                    out.close();
                } catch (IOException e) {
                    //e.printStackTrace();
                }
            }
        }
    }


//--------------------------------------------------------------------------------------------------

    /**
     * 存储服务器地址
     */
    private InetSocketAddress StorageAddr;

    public InetSocketAddress getStorageAddr() {
        return StorageAddr;
    }

    public void setStorageAddr(InetSocketAddress storageAddr) {
        StorageAddr = storageAddr;
    }

    private TrackerGroup trackerGroup;

    public static final String TAG = "StartaiDownloader";

    public void initClientGlobal(String ip, int port) {
        ClientGlobal.g_connect_timeout = 15000;
        ClientGlobal.g_anti_steal_token = false;
        ClientGlobal.g_charset = "UTF-8";
        ClientGlobal.g_network_timeout = 30000;
        ClientGlobal.g_secret_key = "FastDFS1234567890";
        ClientGlobal.g_tracker_http_port = 3108;
        ClientGlobal.setG_anti_steal_token(true);
        // ClientGlobal.g_tracker_group = new TrackerGroup(new
        // InetSocketAddress[] { new InetSocketAddress(ip, port) });
        this.trackerGroup = new TrackerGroup(new InetSocketAddress[]{new InetSocketAddress(ip, port)});
    }

    private void close() {

        try {
            if (trackerServer != null) {
                trackerServer.close();
            }
            if (storageServer != null) {
                storageServer.close();
            }
        } catch (IOException e) {
            e.printStackTrace();
        }

    }

    private StorageServer storageServer;
    private TrackerServer trackerServer;

    /**
     * 获取存储客户端连接
     *
     * @return 存储客户端
     * @throws IOException
     */
    public StorageClient1 getStorageClient1() throws IOException {

        StorageClient1 client1 = null;

        TrackerClient tc = new TrackerClient(trackerGroup);
        // 建立tracker server 的连接
        if (tc == null) {
            return null;
        }
        trackerServer = tc.getConnection();

        // 建立存储服务器的连接
        storageServer = tc.getStoreStorage(trackerServer);

        if (storageServer != null) {
            // FLog.i(TAG, "storageServer IPS = " +
            // storageServer.getInetSocketAddress());
            setStorageAddr(storageServer.getInetSocketAddress());
        } else {
            return null;
        }

        // 建立存储客户端
        client1 = new StorageClient1(trackerServer, storageServer);

        return client1;
    }


}
