package cn.com.startai.fssdk;

import android.app.Application;
import android.os.Handler;
import android.os.HandlerThread;
import android.text.TextUtils;

import org.xutils.x;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.regex.Pattern;

import cn.com.startai.fssdk.db.FDBManager;
import cn.com.startai.fssdk.db.entity.DownloadBean;
import cn.com.startai.fssdk.download.dfs.StartaiDownloader;
import cn.com.startai.fssdk.download.http.StartaiHttpDownloader;
import cn.com.startai.fssdk.event.FSEventDispatcher;
import cn.com.startai.fssdk.utils.FFileUtils;
import cn.com.startai.fssdk.utils.FLog;
import cn.com.startai.fssdk.utils.FUtils;
import cn.com.startai.fssdk.utils.NetAddrUtil;

import static cn.com.startai.fssdk.BaseDownloader.ERROR_CODE_FILE_CREATE_FAILED;
import static cn.com.startai.fssdk.BaseDownloader.ERROR_CODE_URL_FORMAT_ERROR;


/**
 * Created by Robin on 2017/8/15.
 * 419109715@qq.com 彬影
 */

public class StartaiDownloaderManager {

    Handler mHandler;
    private Application context;
    private static volatile StartaiDownloaderManager singleton;

    HashMap<String, BaseDownloader> maps = new HashMap<>();
    HashMap<String, FSDownloadCallback> callbacks = new HashMap<>();
    private String TAG = StartaiDownloaderManager.class.getSimpleName();

    static ExecutorService executorService;
    private boolean isStopAll;


    private StartaiDownloaderManager() {

    }

    public void runInTHread(Runnable runnable) {
        mHandler.post(runnable);
    }

    public void init(Application context, DownloadInifParam param) {
        singleton.context = context;
        x.Ext.init(context);
        x.Ext.setDebug(true);
        if (param != null) {

            int maxThread = param.getMaxThread();
            if (maxThread > 3) {
                maxThread = 3;
            }
            if (maxThread < 1) {
                maxThread = 1;
            }
            int retryTimes = param.getRetryTimes();
            if (retryTimes > 2) {
                retryTimes = 2;
            }
            if (retryTimes < 0) {
                retryTimes = 0;
            }
            DownloadParam.maxThread = maxThread;
            DownloadParam.retryTimes = retryTimes;
        }

        executorService = Executors.newFixedThreadPool(DownloadParam.maxThread);
        HandlerThread ht = new HandlerThread(StartaiDownloaderManager.class.getSimpleName());
        ht.start();
        singleton.mHandler = new Handler(ht.getLooper());

        getFDBManager().resetDownloadDB();

    }

    public static StartaiDownloaderManager getInstance() {
        if (singleton == null) {
            synchronized (StartaiDownloaderManager.class) {
                if (singleton == null) {
                    singleton = new StartaiDownloaderManager();

                }
            }
        }

        return singleton;
    }


    /**
     * 暂停任务
     *
     * @param url 下载的url
     */
    public void stopDownloader(final String url) {

        BaseDownloader downloader = maps.remove(url);
        if (downloader != null) {
            DownloadBean downloadBean = downloader.getDownloadBean();
            downloadBean.setStatus(0);
            FDBManager.getInstance().addOrUpdateDownloadBean(downloadBean);

            downloader.stopDownload();
        }
    }

    /**
     * 清除下载任务
     *
     * @param url
     * @param isDeleteFile
     */
    public void clearDownloader(final String url, final boolean isDeleteFile) {

        stopDownloader(url);

        FDBManager.getInstance().deleteDownloadBeanByUrl(url);

        if (isDeleteFile) {
            DownloadBean downloadBeanByUrl = FDBManager.getInstance().getDownloadBeanByUrl(url);
            if (downloadBeanByUrl != null) {
                FFileUtils.deleteFile(downloadBeanByUrl.getLocalPath());
                FFileUtils.deleteFile(downloadBeanByUrl.getLocalPath() + ".tmp");
            }
        }
    }

    /**
     * 清除下载任务
     *
     * @param isDeleteFile
     */
    public void clearAllDownloader(final boolean isDeleteFile) {


        stopAllDownloaders();
        getFDBManager().deleteAllDownload();
        if (isDeleteFile) {
            ArrayList<DownloadBean> allDownloadBean = FDBManager.getInstance().getAllDownloadBean();
            for (DownloadBean downloadBean : allDownloadBean) {
                FFileUtils.deleteFile(downloadBean.getLocalPath());
                FFileUtils.deleteFile(downloadBean.getLocalPath() + ".tmp");
            }
        }
    }


    /**
     * 暂停所有下载任务
     */
    public void stopAllDownloaders() {

        isStopAll = true;
        Collection<BaseDownloader> values = maps.values();
        for (BaseDownloader value : values) {
            value.stopDownload();
        }
        maps.clear();

        getFDBManager().resetDownloadDB();


    }

    /**
     * 开启所有下载任务
     */
    public void startAllDownloader() {
        isStopAll = false;
        ArrayList<DownloadBean> allDownloadBean = getFDBManager().getAllUncompleteDownloadBean();

        for (DownloadBean downloadBean : allDownloadBean) {
            downloadBean.setStatus(3);
            getFDBManager().addOrUpdateDownloadBean(downloadBean);
            toDownload();
        }

    }

    /**
     * 开始下载
     *
     * @param downloadBean
     */
    public void startDownload(final DownloadBean downloadBean, final FSDownloadCallback callback) {

        runInTHread(new Runnable() {
            @Override
            public void run() {
                isStopAll = false;

                if (!isAvalibleUrl(downloadBean)) {
                    downloadBean.setStatus(4);
                    FDBManager.getInstance().addOrUpdateDownloadBean(downloadBean);
                    if (callback != null) {
                        callback.onFailure(downloadBean, ERROR_CODE_URL_FORMAT_ERROR);
                    } else {
                        FSEventDispatcher.getInstance().onDownloadFailure(downloadBean, ERROR_CODE_URL_FORMAT_ERROR);
                    }
                    return;
                }

                callbacks.put(downloadBean.getUrl(), callback);


                final DownloadBean downloadBeanByUrl = FDBManager.getInstance().getDownloadBeanByUrl(downloadBean.getUrl());
                if (downloadBeanByUrl == null) {
                    FLog.d(TAG, "没有下载过此文件");
                    FFileUtils.deleteFile(downloadBean.getLocalPath());
                    FFileUtils.deleteFile(downloadBean.getLocalPath() + ".tmp");


                    try {
                        FUtils.mkDir(downloadBean.getLocalPath() + ".tmp");
                    } catch (IOException e) {
                        e.printStackTrace();

                        downloadBean.setStatus(4);
                        getFDBManager().addOrUpdateDownloadBean(downloadBean);
                        if (callback != null) {

                            callback.onFailure(downloadBean, ERROR_CODE_FILE_CREATE_FAILED);
                        } else {
                            FSEventDispatcher.getInstance().onDownloadFailure(downloadBean, ERROR_CODE_FILE_CREATE_FAILED);
                        }

                        return;
                    }


                    downloadBean.setStatus(3);
                    downloadBean.setAddedSize(0);
                    downloadBean.setProgress(0);
                    downloadBean.setTotalSize(0);
                    FDBManager.getInstance().addOrUpdateDownloadBean(downloadBean);
                    if (callback != null) {

                        callback.onWaiting(downloadBean);
                    } else {
                        FSEventDispatcher.getInstance().onDownloadWaiting(downloadBean);
                    }
                } else {

                    //曾下载过此文件
                    //判断文件是否已经下载完成
                    File file = new File(downloadBeanByUrl.getLocalPath());
                    File fileTmp = new File(downloadBeanByUrl.getLocalPath() + ".tmp");
                    if (file.exists() && file.length() != 0 && file.length() == downloadBeanByUrl.getTotalSize()) {
                        FLog.d(TAG, "文件已经下载完成过");
                        //回调下载成功
                        downloadBeanByUrl.setStatus(2);
                        downloadBeanByUrl.setProgress(100);
                        FDBManager.getInstance().addOrUpdateDownloadBean(downloadBeanByUrl);

                        FFileUtils.deleteFile(fileTmp);
                        if (callback != null) {
                            callback.onSuccess(downloadBeanByUrl);
                        } else {
                            FSEventDispatcher.getInstance().onDownloadSuccess(downloadBeanByUrl);
                        }
                        return;
                    }

                    if (!fileTmp.exists() && !file.exists()) {

                        FLog.d(TAG, "曾经下载过此文件，但已经被删除掉了");
                        try {

                            downloadBeanByUrl.setTotalSize(0);
                            downloadBeanByUrl.setProgress(0);
                            downloadBeanByUrl.setAddedSize(0);
                            downloadBeanByUrl.setStatus(3);
                            FDBManager.getInstance().addOrUpdateDownloadBean(downloadBeanByUrl);

                            FUtils.mkDir(fileTmp.getAbsolutePath());

                        } catch (IOException e) {
                            e.printStackTrace();

                            downloadBeanByUrl.setStatus(4);
                            getFDBManager().addOrUpdateDownloadBean(downloadBeanByUrl);
                            if (callback != null) {

                                callback.onFailure(downloadBeanByUrl, ERROR_CODE_FILE_CREATE_FAILED);
                            } else {
                                FSEventDispatcher.getInstance().onDownloadFailure(downloadBeanByUrl, ERROR_CODE_FILE_CREATE_FAILED);
                            }

                            return;
                        }
                    }


                    FLog.d(TAG, "曾经下载过此文件，但文件还没有下载完全");
                    //文件是否正在下载中
                    if (downloadBeanByUrl.getStatus() == 1) {
                        //文件正在下载，不处理
                        FLog.d(TAG, "文件正在下载不做处理");
                        return;
                    }

                    //文件没有下载完成 继续断点续传
                    downloadBeanByUrl.setStatus(3);
                    downloadBeanByUrl.setAddedSize(fileTmp.length()); //已经下载的大小
                    FLog.d(TAG, "fileTmp.lenght = " + fileTmp.length());
                    FDBManager.getInstance().addOrUpdateDownloadBean(downloadBeanByUrl);
                    if (callback != null) {
                        callback.onWaiting(downloadBeanByUrl);
                    } else {
                        FSEventDispatcher.getInstance().onDownloadWaiting(downloadBeanByUrl);
                    }
                }

                toDownload();
            }
        });
    }

    public void toDownload() {
        if (isStopAll) {
            FLog.d(TAG, "已经暂停所有任务，不再请求下载");
            return;
        }
        FLog.d(TAG, "toDownload");
        long downloadingTaskSize = getDownloadingTaskSize();
        FLog.d(TAG, "当前下载队列个数  " + downloadingTaskSize + " maxThread = " + DownloadParam.maxThread);
        //从数据库存中取出一个等待下载的下载任务 并开始下载
        if (downloadingTaskSize < DownloadParam.maxThread) {
            final DownloadBean downloadBeanByStatus = FDBManager.getInstance().getDownloadBeanByStatus(3);
            if (downloadBeanByStatus == null) {
                FLog.d(TAG, "没有需要下载的任务了");
            } else {
                BaseDownloader downloader = null;
                FSDownloadCallback callback = callbacks.get(downloadBeanByStatus.getUrl());
                if ("tcp".equals(downloadBeanByStatus.getProtocol())) {
                    downloader = new StartaiDownloader(downloadBeanByStatus, callback);

                } else if ("http".equals(downloadBeanByStatus.getProtocol())) {

                    downloader = new StartaiHttpDownloader(downloadBeanByStatus, callback);

                } else {

                    downloadBeanByStatus.setStatus(4);
                    getFDBManager().addOrUpdateDownloadBean(downloadBeanByStatus);
                    if (callback != null) {

                        callback.onFailure(downloadBeanByStatus, BaseDownloader.ERROR_CODE_UNKNOW_DOWNLOAD_TYPE);
                    } else {
                        FSEventDispatcher.getInstance().onDownloadFailure(downloadBeanByStatus, BaseDownloader.ERROR_CODE_UNKNOW_DOWNLOAD_TYPE);
                    }

                    return;
                }
                downloadBeanByStatus.setStatus(1);
                FDBManager.getInstance().addOrUpdateDownloadBean(downloadBeanByStatus);
                if (callback != null) {
                    callback.onStart(downloadBeanByStatus);
                } else {
                    FSEventDispatcher.getInstance().onDownloadStart(downloadBeanByStatus);
                }
                //开始下载任务
                executorService.execute(downloader);
                FLog.d(TAG, "提交给线程池，开始下载");
                maps.put(downloadBeanByStatus.getUrl(), downloader);
            }
        } else {
            FLog.d(TAG, "下载队列已经超过最大限制，等待下载中。。。");
        }


    }

    public FDBManager getFDBManager() {
        return FDBManager.getInstance();
    }


    private boolean isAvalibleUrl(DownloadBean downloadBean) {

        if (downloadBean == null) {
            return false;
        }

        String url = downloadBean.getUrl();
        if (TextUtils.isEmpty(url)) {
            return false;
        }

        if (!isURL(url)) {
            return false;
        }
        if (url.startsWith("tcp")) {
            downloadBean.setProtocol("tcp");

        } else if (url.startsWith("http")) {
            downloadBean.setProtocol("http");
        } else {
            return false;
        }

        if (TextUtils.isEmpty(downloadBean.getLocalPath())) {

            if (TextUtils.isEmpty(downloadBean.getFileName())) {
                downloadBean.setFileName(FFileUtils.getFileName(downloadBean.getUrl()));
            }
            if (TextUtils.isEmpty(downloadBean.getExtName())) {
                downloadBean.setExtName(FFileUtils.getFileExtension(downloadBean.getFileName()));
            }
            downloadBean.setLocalPath(DownloadParam.defaultDir + "/" + downloadBean.getFileName());


        } else {
            downloadBean.setFileName(FFileUtils.getFileName(downloadBean.getLocalPath()));
            downloadBean.setExtName(FFileUtils.getFileExtension(downloadBean.getLocalPath()));
        }


        if ("tcp".equals(downloadBean.getProtocol())) {
            downloadBean.setFileId(NetAddrUtil.getFileIdFromUrl(downloadBean.getUrl()));
        }


        return true;
    }

    private static boolean isMatch(String regex, CharSequence input) {
        return input != null && input.length() > 0 && Pattern.matches(regex, input);
    }

    private static boolean isURL(CharSequence input) {
        return isMatch("[a-zA-z]+://[^\\s]*", input);
    }


    /**
     * 获取当前正在下载的任务个数
     */
    public long getDownloadingTaskSize() {


        return FDBManager.getInstance().getDownloadingCount();
    }

}
