package cn.com.startai.common.channel.mqtt.client;


import android.app.Application;
import android.net.NetworkInfo;

import org.eclipse.paho.client.mqttv3.IMqttActionListener;
import org.eclipse.paho.client.mqttv3.IMqttDeliveryToken;
import org.eclipse.paho.client.mqttv3.IMqttToken;
import org.eclipse.paho.client.mqttv3.MqttAsyncClient;
import org.eclipse.paho.client.mqttv3.MqttCallback;
import org.eclipse.paho.client.mqttv3.MqttConnectOptions;
import org.eclipse.paho.client.mqttv3.MqttException;
import org.eclipse.paho.client.mqttv3.MqttMessage;

import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import cn.com.startai.common.channel.CErrorCode;
import cn.com.startai.common.channel.CallbackManager;
import cn.com.startai.common.channel.ICallListener;
import cn.com.startai.common.channel.mqtt.IMqtt;
import cn.com.startai.common.channel.mqtt.MqttConnectState;
import cn.com.startai.common.channel.mqtt.MqttParam;
import cn.com.startai.common.channel.mqtt.event.HeartPingSender;
import cn.com.startai.common.channel.mqtt.event.IMqttHeartListener;
import cn.com.startai.common.channel.mqtt.event.IMqttListener;
import cn.com.startai.common.utils.CLog;
import cn.com.startai.common.utils.CStringUtils;
import cn.com.startai.common.utils.CThreadPoolUtils;
import cn.com.startai.common.utils.network.CNetworkManager;
import cn.com.startai.common.utils.network.INetworkListener;

import static cn.com.startai.common.CommonSDKInterface.TAG;

/**
 * Created by Robin on 2019/3/21.
 * 419109715@qq.com 彬影
 */
public class MqttImpl implements IMqtt, INetworkListener, CErrorCode {


    private ScheduledExecutorService singleConnectThreadPool;

    private MqttAsyncClient client;

    private IMqttListener mqttListener;
    private IMqttHeartListener heartListener;

    private MqttConnectState connectState;
    private int retryCount;

    private MqttParam mqttParam;
    private Application application;
    private MqttConnectOptions option;
    private HeartPingSender pingSender;
    private ScheduledExecutorService netThreadPool;
    private String clientId;
    private boolean isReleaseIntent = false;
    private ScheduledFuture<?> singleSschedule;


    @Override
    public void init(Application application, MqttParam mqttParam) {
        if (this.application != null) {
            return;
        }
        CLog.d(TAG, "MqttImpl.init()");
        this.heartListener = mqttParam.getHeartListener();
        this.mqttListener = mqttParam.getMqttListener();
        this.application = application;
        this.mqttParam = mqttParam;
        this.clientId = mqttParam.getClientId();
        this.isReleaseIntent = false;

        netThreadPool = CThreadPoolUtils.getInstance().getNetThreadPool();
        singleConnectThreadPool = CThreadPoolUtils.getInstance().getSingleConnectThreadPool();

        try {
            if (client == null) {
                if (heartListener != null) {
                    pingSender = new HeartPingSender(heartListener);
                    client = new MqttAsyncClient("ssl://thisUrlWillBeOverrideByOption.serverURIs", mqttParam.getClientId(), null, pingSender);
                } else {
                    client = new MqttAsyncClient("ssl://thisUrlWillBeOverrideByOption.serverURIs", mqttParam.getClientId(), null);
                }
                client.setCallback(mqttCallback);
            }

        } catch (MqttException e) {
            e.printStackTrace();
            connectState = MqttConnectState.DISCONNECTED;
            callbackDisConn(ERROR_CONN_PARAM);
            return;
        }

        connectWithDelay(0);

        CNetworkManager.getInstance().addNetworkListener(this);

    }


    private void connectWithDelay(long timeDelay) {
        CLog.d(TAG, "connectWithDelay " + timeDelay);
        if (isReleaseIntent) {
            CLog.e(TAG, "is in release ,stop reconnect 2");
            close();
            return;
        }
        if (timeDelay == 0 ) {
            CThreadPoolUtils.getInstance().cancelScheduledFuture(singleSschedule);
        }
        singleSschedule = singleConnectThreadPool.schedule(new Runnable() {
            @Override
            public void run() {
                if (isReleaseIntent) {
                    CLog.e(TAG, "is in release ,stop reconnect 2");
                    close();
                    return;
                }
                toConnect();
            }
        }, timeDelay, TimeUnit.MILLISECONDS);

    }


    private void toConnect() {
        if (connectState == MqttConnectState.CONNECTED || connectState == MqttConnectState.CONNECTING) {
            CLog.d(TAG, "connectState = " + connectState + "  , no need repeat connect");
            return;
        }

        CLog.d(TAG, "toConnect() client = " + client);
        connectState = MqttConnectState.CONNECTING;

        boolean availableParam = mqttParam.isAvailableParam();
        if (!availableParam) {
            CLog.e(TAG, "availableParam mqttParam = " + mqttParam.toString());
            connectState = MqttConnectState.DISCONNECTED;
            callbackDisConn(ERROR_CONN_PARAM);
            return;
        }
        if (!CNetworkManager.getInstance().isAvaliableNetwork(false)) {
            CLog.e(TAG, "network is unvaliable ");
            connectState = MqttConnectState.DISCONNECTED;
            callbackDisConn(ERROR_CONN_NET);
            return;
        }


        option = mqttParam.getConnectOptions(application);
        if (option == null) {
            connectState = MqttConnectState.DISCONNECTED;
            callbackDisConn(ERROR_CONN_CER);
            return;
        }

        if ((client != null && client.isConnected())) {
            connectState = MqttConnectState.CONNECTED;
            CLog.d(TAG, "client is connected , no need to repeat  connect");
            return;
        }


        if (client != null) {
            try {
                CLog.d(TAG, "connect host = " + Arrays.toString(option.getServerURIs()) + " clientid = " + clientId);
                client.connect(option).waitForCompletion();
                if (isReleaseIntent) {
                    CLog.d(TAG, "connect success isReleaseIntent = true 11 ");
                    close();
                    return;
                }
                CLog.e(TAG, "connect success host = " + getHost() + " clientid = " + client.getClientId());
                retryCount = 0;
                connectState = MqttConnectState.CONNECTED;
                callbackConn(getHost(), client.getClientId());
            } catch (MqttException exception) {
                CLog.file(exception);
                exception.printStackTrace();
                CLog.e(TAG, "connect failure  ");
                doWithConnectFailed(exception);
            }
        } else {
            CLog.e(TAG, "client = " + client);
        }
    }


    private void doWithConnectFailed(MqttException mqttException) {

        int reasonCode = mqttException.getReasonCode();
        if (reasonCode == 32100 || reasonCode == 32110 || reasonCode == 32102 || reasonCode == 32111) {
            CLog.e(TAG, "reasonCode = " + reasonCode);
            return;
        }
        connectState = MqttConnectState.CONNECTFAIL;
        mqttException.printStackTrace();
        CLog.e(TAG, "connect failed , will reconnecting " + retryCount++);
        if (reasonCode == 0) {
            String exceptionStr = mqttException.toString();
            if (exceptionStr.contains("SocketTimeoutException")) {
                callbackDisConn(ERROR_CONN_TIMEOUT);
            } else if (exceptionStr.contains("UnknownHostException")) {
                callbackDisConn(ERROR_CONN_SERVER);
            }
            connectWithDelay(1000);
        } else {
            callbackDisConn(ERROR_CONN_NET);
            //准备重连
            connectWithDelay(1000);
        }

    }

    @Override
    public void publish(final String topic, final byte[] data, final ICallListener onCallListener) {

        if (connectState == MqttConnectState.CONNECTED) {

            final MqttMessage mqttMessage = new MqttMessage(data);

            try {


                if (CStringUtils.isMessyCode(topic)) {
                    CLog.e(TAG, "public falied , topic contains messyCode , topic =  " + topic);
                    CallbackManager.callbackCallResult(false, onCallListener, ERROR_SEND_MESSYCODE_TOPIC);
                    return;
                }


                CLog.d(TAG, "publish before");
                client.publish(topic, mqttMessage, null, new IMqttActionListener() {
                    @Override
                    public void onSuccess(IMqttToken asyncActionToken) {

                        CLog.d(TAG, "publish : topic = " + topic + "\nqos = " + mqttMessage.getQos() + "\nretain = " + mqttMessage.isRetained() + "\nmsg = " + new String(mqttMessage.getPayload()));

                        //send success
                        CallbackManager.callbackCallResult(true, onCallListener, 0);
                    }

                    @Override
                    public void onFailure(IMqttToken asyncActionToken, Throwable exception) {
                        CLog.e(TAG, "public failed ");
                        CLog.file(exception);
                        exception.printStackTrace();
                        if (exception instanceof MqttException) {
                            MqttException e = (MqttException) exception;
                            int reasonCode = e.getReasonCode();
                            if (reasonCode == MqttException.REASON_CODE_MAX_INFLIGHT) {
                                CallbackManager.callbackCallResult(false, onCallListener, ERROR_SEND_TOO_FAST);
                            } else if (reasonCode == MqttException.REASON_CODE_CLIENT_DISCONNECTING
                                    || reasonCode == MqttException.REASON_CODE_CLIENT_NOT_CONNECTED
                            ) {
                                connectState = MqttConnectState.DISCONNECTED;
                                CallbackManager.callbackCallResult(false, onCallListener, ERROR_SEND_CLIENT_DISCONNECT);
                            } else {
                                CallbackManager.callbackCallResult(false, onCallListener, ERROR_SEND_NET);
                            }
                        } else {
                            CallbackManager.callbackCallResult(false, onCallListener, ERROR_SEND_NET);
                        }
                    }
                });

            } catch (MqttException e) {
                CLog.e(TAG, "public failed 2");
                CLog.file(e);
                e.printStackTrace();
                int reasonCode = e.getReasonCode();
                if (reasonCode == MqttException.REASON_CODE_MAX_INFLIGHT) {
                    CallbackManager.callbackCallResult(false, onCallListener, ERROR_SEND_TOO_FAST);
                } else if (reasonCode == MqttException.REASON_CODE_CLIENT_DISCONNECTING
                        || reasonCode == MqttException.REASON_CODE_CLIENT_NOT_CONNECTED
                ) {
                    connectState = MqttConnectState.DISCONNECTED;
                    CallbackManager.callbackCallResult(false, onCallListener, ERROR_SEND_CLIENT_DISCONNECT);
                    connectWithDelay(1000);
                } else {
                    CallbackManager.callbackCallResult(false, onCallListener, ERROR_SEND_NET);
                }
            }

        } else {
            CLog.e(TAG, "publish failed " + ERROR_SEND_CLIENT_DISCONNECT);
            CallbackManager.callbackCallResult(false, onCallListener, ERROR_SEND_CLIENT_DISCONNECT);
            connectWithDelay(1000);
        }


    }

    private void toSubscribe(boolean isSync, List<String> topics, final ICallListener callListener) {
        if (connectState == MqttConnectState.CONNECTED) {
            try {
                if (topics == null || topics.size() == 0) {
                    CLog.e(TAG, "subscribe failed,topics = null or topic.size = 0");
                    CallbackManager.callbackCallResult(false, callListener, ERROR_SUB_NULL_TOPIC);
                    return;
                }

                String[] topicArr = new String[topics.size()];

                topicArr = topics.toArray(topicArr);


                int[] qoss = new int[topics.size()];

                for (int i = 0; i < topicArr.length; i++) {

                    String finalTopic = topicArr[i];
                    if (CStringUtils.isMessyCode(finalTopic)) {
                        CLog.e(TAG, "subscribe failed, topic contains messyCode , topic =  " + finalTopic);
                        CallbackManager.callbackCallResult(false, callListener, ERROR_SUB_UNVALIABLE_TOPIC);
                        return;
                    }
                }

                for (int i : qoss) {
                    qoss[i] = 1;
                }

                if (isSync) {
                    client.subscribe(topicArr, qoss).waitForCompletion();
                    CLog.e(TAG, "sub topics = " + Arrays.toString(topicArr) + " success");
                    CallbackManager.callbackCallResult(true, callListener, 0);
                } else {
                    final String[] finalTopicArr = topicArr;
                    client.subscribe(topicArr, qoss, null, new IMqttActionListener() {
                        @Override
                        public void onSuccess(IMqttToken asyncActionToken) {
                            CLog.e(TAG, "sub topics = " + Arrays.toString(finalTopicArr) + " success");
                            CallbackManager.callbackCallResult(true, callListener, 0);
                        }

                        @Override
                        public void onFailure(IMqttToken asyncActionToken, Throwable exception) {
                            CallbackManager.callbackCallResult(false, callListener, UNKOWN);
                            exception.printStackTrace();
                        }
                    });
                }

            } catch (MqttException e) {
                e.printStackTrace();
                CLog.file(e);
                CLog.e(TAG, "sub topics = " + topics + " failed " + ERROR_SUB_CAN_NOT_SUB);
                CallbackManager.callbackCallResult(false, callListener, ERROR_SUB_CAN_NOT_SUB);
            }

        } else {
            CLog.e(TAG, "sub failed " + ERROR_SUB_NO_CONN);
            CallbackManager.callbackCallResult(false, callListener, ERROR_SUB_NO_CONN);
            connectWithDelay(1000);
        }
    }

    @Override
    public void subscribe(final List<String> topics, final ICallListener callListener) {
        toSubscribe(false, topics, callListener);
    }

    @Override
    public void subscribeSync(List<String> topics, ICallListener callListener) {
        toSubscribe(true, topics, callListener);
    }


    @Override
    public void unSubscribe(final List<String> topics, final ICallListener callListener) {

        if (connectState == MqttConnectState.CONNECTED) {
            try {
                if (topics == null) {
                    CLog.e(TAG, "topics must not be null");
                    CallbackManager.callbackCallResult(false, callListener, ERROR_UNSUB_NULL_TOPIC);
                    return;
                }

                for (int i = 0; i < topics.size(); i++) {

                    String finalTopic = topics.get(i);
                    if (CStringUtils.isMessyCode(finalTopic)) {
                        CLog.e(TAG, "unsubscribe failed, topic contains messyCode , topic =  " + finalTopic);
                        CallbackManager.callbackCallResult(false, callListener, ERROR_UNSUB_UNVALIABLE_TOPIC);
                        return;
                    }
                }

                if (client == null || !client.isConnected()) {
                    CLog.e(TAG, "sub failed " + ERROR_UNSUB_NO_CONN);
                    CallbackManager.callbackCallResult(false, callListener, ERROR_UNSUB_NO_CONN);
                    return;
                }

                String[] topicArr = new String[topics.size()];

                topicArr = topics.toArray(topicArr);

                client.unsubscribe(topicArr).waitForCompletion();

                CLog.d(TAG, "unSub topics = " + Arrays.toString(topicArr) + " success");
                CallbackManager.callbackCallResult(true, callListener, 0);
            } catch (MqttException e) {
                CLog.e(TAG, "unSub topics = " + topics + " failed 1 " + ERROR_UNSUB_CAN_NOT_UNSUB);
                CLog.file(e);
                e.printStackTrace();
                CallbackManager.callbackCallResult(false, callListener, ERROR_UNSUB_CAN_NOT_UNSUB);

            }

        } else {
            CLog.e(TAG, "sub failed " + ERROR_UNSUB_NO_CONN);
            CallbackManager.callbackCallResult(false, callListener, ERROR_UNSUB_NO_CONN);
            connectWithDelay(1000);
        }

    }

    @Override
    public MqttConnectState getMqttConnectState() {
        return connectState;
    }


    @Override
    public void release() {
        CLog.d(TAG, "MqttImpl release");
        CNetworkManager.getInstance().removeNetworkListener(this);

        if (pingSender != null) {
            pingSender.stop();
            pingSender = null;
        }

        singleSschedule.cancel(true);
        application = null;
        mqttParam = null;
        mqttListener = null;
        retryCount = 0;
        isReleaseIntent = true;
        close();
    }

    private void disconnect() {

        connectState = MqttConnectState.DISCONNECTING;
        if (client != null) {
            if (client.isConnected()) {
                try {
                    client.disconnect().waitForCompletion();
                    CLog.d(TAG, "disconnect");
                } catch (MqttException e) {
                    e.printStackTrace();
                }
            }
        }
        connectState = MqttConnectState.DISCONNECTED;
    }

    private void close() {


        CLog.d(TAG, "MqttImpl close");

        if (client != null) {
            CLog.d(TAG, "close() connectState = " + connectState);
            if (client.isConnected()) {
                try {
                    connectState = MqttConnectState.DISCONNECTING;
                    client.disconnect(1000).waitForCompletion();
                    CLog.e(TAG, "mqtt close.disconnect!! ");
                } catch (Exception e) {
                    e.printStackTrace();
                }
            } else if (connectState == MqttConnectState.CONNECTING) {
                try {
                    connectState = MqttConnectState.DISCONNECTING;
                    client.disconnectForcibly(1000, 1000);
                    CLog.e(TAG, "mqtt close.disconnectForcibly!! ");
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }

            try {
                client.close();
                CLog.e(TAG, "mqtt close!!  ");
                connectState = MqttConnectState.DISCONNECTED;
                client = null;
            } catch (Exception e) {
                e.printStackTrace();
            }

        }


    }

    @Override
    public String getClientId() {
        if (client == null) {
            return null;
        }
        return client.getClientId();
    }

    @Override
    public String getHost() {
        if (client == null) {
            return null;
        }
        return client.getCurrentServerURI();
    }


    @Override
    public void changeHost(final String[] hosts, final ICallListener callListener) {

        netThreadPool.execute(new Runnable() {
            @Override
            public void run() {


                CLog.e(TAG, "changeHost hosts = " + Arrays.toString(hosts));

                if (client == null || !client.isConnected()) {
                    CLog.e(TAG, "client = null or isDisconnect");
                    CallbackManager.callbackCallResult(false, callListener, ERROR_CHANGEHOST_DISCONNECT);
                    return;
                }

                if (hosts == null) {
                    CallbackManager.callbackCallResult(false, callListener, ERROR_CHANGEHOST_EMPTY);
                    return;
                }

                for (String host : hosts) {
                    try {
                        MqttConnectOptions.validateURI(host);
                    } catch (IllegalArgumentException e) {
                        e.printStackTrace();
                        CLog.e(TAG, "invalidate uri");
                        CallbackManager.callbackCallResult(false, callListener, ERROR_CHANGEHOST_INVALIDE);
                        return;
                    }
                }

                boolean needChangeNow = true;
                if (hosts[0].equals(getHost())) {
                    CLog.e(TAG, "The replaced node is the same as the currently connected node and will not be disconnected.");
                    needChangeNow = false;
                }

                if (mqttParam != null) {
                    mqttParam.setHosts(hosts);
                }

                if (needChangeNow) {
                    disconnect();
                    callbackDisConn(ERROR_LOST_CHANGE_HOST);
                    connectWithDelay(200);
                }
                CallbackManager.callbackCallResult(true, callListener, 0);
            }
        });
    }

    private MqttCallback mqttCallback = new MqttCallback() {

        @Override
        public void connectionLost(Throwable cause) {
            CLog.e(TAG, "connectionLost");
            cause.printStackTrace();
            connectState = MqttConnectState.DISCONNECTED;
            if (isReleaseIntent) {
                CLog.e(TAG, "is in release connectionLost");
                close();
                return;
            }

            if (cause instanceof MqttException) {
                MqttException mqttException = (MqttException) cause;
                int reasonCode = mqttException.getReasonCode();
                if (reasonCode == 32000) {//等待来自服务器的响应时超时
                    callbackDisConn(ERROR_LOST_TIMEOUT);
                } else {
                    callbackDisConn(ERROR_LOST_NET);
                }
            } else {
                callbackDisConn(ERROR_LOST_NET);
            }

            connectWithDelay(1000);
        }

        @Override
        public void messageArrived(final String topic, final MqttMessage message) {

            String msg = new String(message.getPayload());
            int qos = message.getQos();
            CLog.e(TAG, "onMqttMessageArrived : topic = " + topic + "\nqos =  " + qos + "\nmessage = " + msg);

            callbackMsgArrive(topic, message);

        }

        @Override
        public void deliveryComplete(IMqttDeliveryToken token) {
        }
    };


    @Override
    public void onWifiConnected() {
        connectWithDelay(100);
    }

    @Override
    public void onMobileConnected() {
        connectWithDelay(100);
    }

    @Override
    public void onEthernetConnected() {
        connectWithDelay(100);
    }

    @Override
    public void onUnkownNetwork() {
        CLog.e(TAG, "unkown network");
    }

    @Override
    public void onNetworkStateChange(String networkType, NetworkInfo.State state) {

    }

    private void callbackMsgArrive(final String topic, final MqttMessage message) {
        netThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                if (mqttListener != null) {
                    mqttListener.onMqttMessageArrived(topic, message.getPayload());
                }
            }
        });
    }

    private void callbackConn(final String host, final String clientId) {
        netThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                if (mqttListener != null) {
                    mqttListener.onMqttConnected(host, clientId);
                }
            }
        });

    }

    private void callbackDisConn(final int errCode) {
        netThreadPool.execute(new Runnable() {
            @Override
            public void run() {
                if (mqttListener != null) {
                    mqttListener.onMqttDisconnected(errCode);
                }
            }
        });

    }


    private MqttImpl() {
    }

    public static MqttImpl getInstance() {
        return SingleTonHoulder.singleTonInstance;
    }


    private static class SingleTonHoulder {
        private static final MqttImpl singleTonInstance = new MqttImpl();
    }

}
