package cn.com.itep.printer.usb;

import android.hardware.usb.UsbDevice;
import android.hardware.usb.UsbDeviceConnection;
import android.hardware.usb.UsbEndpoint;
import android.hardware.usb.UsbInterface;
import android.util.Log;

import java.util.ArrayList;
import java.util.List;

import cn.com.itep.printer.DeviceCode;
import cn.com.itep.printer.PrinterStatusListener;
import cn.com.itep.printer.PrinterType;
import cn.com.itep.printer.Utils;

/**
 * Created by 辉 on 2016/9/8.
 */
class UsbDev {

    private static final int mWriteTimeOut = 300000;
    public boolean mIsConnect = false;
    public UsbEndpoint mUsbOutputpoint = null;
    public UsbEndpoint mUsbInputPoint = null;
    private UsbInterface mUsbInterface = null;
    private UsbDevice mUsbDevice = null;
    private UsbDeviceConnection mUsbDeviceConnection = null;
    private PrinterStatusListener mPrinterStatusListener;

    public UsbInterface getUsbInterface() {
        return mUsbInterface;
    }

    public void setUsbInterface(UsbInterface usbInterface) {
        mUsbInterface = usbInterface;
    }

    public UsbDeviceConnection getUsbDeviceConnection() {
        return mUsbDeviceConnection;
    }

    public void setUsbDeviceConnection(UsbDeviceConnection usbDeviceConnection) {
        mUsbDeviceConnection = usbDeviceConnection;
    }

    public void setPrinterStatusListener(PrinterStatusListener printerStatusListener){
        this.mPrinterStatusListener = printerStatusListener;
    }

    public UsbDevice getUsbDevice() {
        return mUsbDevice;
    }

    public void setUsbDevice(UsbDevice usbDevice) {
        mUsbDevice = usbDevice;
    }

    @Override
    protected void finalize() throws Throwable {
        super.finalize();

        closeDevice();
    }

    public void closeDevice(){
        if (mUsbDeviceConnection != null && mUsbInterface != null) {
            mUsbDeviceConnection.releaseInterface(mUsbInterface);
        }

        if (mUsbDeviceConnection != null) {
            mUsbDeviceConnection.close();
        }
        mUsbDeviceConnection = null;
        mIsConnect = false;
        mUsbOutputpoint = null;
        mUsbInputPoint = null;
    }

    public boolean IsConnect(){
        return mIsConnect;
    }

    public String getUsbPrinterName(){
        if (IsConnect()){
            return mUsbDevice.getDeviceName();
        }

        return "";
    }

    /**
     *  获取打印机的状态
     * @return 24表示打印机就绪，48表示打印机缺纸
     */
    public byte getStatus(){
        if (mIsConnect){
            byte[] buffer = new byte[64];
            mUsbDeviceConnection.controlTransfer(0xa1, 0x01, 0x00, 0x00, buffer, 64, 5000);
            Log.d(UsbDev.class.getSimpleName(), buffer[0] + "");
            return buffer[0];
        }

        return -1;
    }

    /**
     *  向指定的端口输出数据
     * @param outBuff 传入缓存
     * @param outLength 传入缓存数据的长度
     * @return 返回实际写入数据的长度，返回-1表示设备端口并未打开，写入失败
     */
    public int writeDevice(byte[] outBuff, int outLength){

//        this.getStatus();

        int writeLength = -1;
        if (mIsConnect) {
            try {

                if (outLength <= Utils.PackageSize) {
                    writeLength = mUsbDeviceConnection.bulkTransfer(mUsbOutputpoint, outBuff, outLength, mWriteTimeOut);
                    if (writeLength < 0){
                        throw new Exception();
                    }
                }
                else{
                    int n = outLength / Utils.PackageSize;
                    int m = outLength % Utils.PackageSize;

                    byte[] buffer = new byte[Utils.PackageSize];

                    for (int i = 0; i < n; ++i)
                    {
                        for (int j = 0; j < Utils.PackageSize; ++j){
                            buffer[j] = outBuff[i * Utils.PackageSize + j];
                        }
                        writeLength = mUsbDeviceConnection.bulkTransfer(mUsbOutputpoint, buffer, Utils.PackageSize, mWriteTimeOut);

                        if (writeLength < 0){
                            throw new Exception();
                        }
                    }

                    for (int j = 0; j < m; ++j){
                        buffer[j] = outBuff[n * Utils.PackageSize + j];
                    }
                    writeLength = mUsbDeviceConnection.bulkTransfer(mUsbOutputpoint, buffer, m, mWriteTimeOut);
                    if (writeLength < 0){
                        throw new Exception();
                    }
                }
            }catch (Exception e){
                mIsConnect = false;
                e.printStackTrace();

                if (mPrinterStatusListener != null){
                    mPrinterStatusListener.disconnect(PrinterType.printToUsb, DeviceCode.ErrorDisconnect.getCode());
                }
            }
        }

        return writeLength;
    }

    /**
     *  读取指定端口的数据
     * @return 返回实际读取数据的长度
     * @throws Exception 抛出异常，如果传入的缓存为空的话
     */
    public byte[] readDevice(){

        byte[] inBuff = null;
        int length = -1;

        if (mIsConnect){
            List<Byte> listInBuff = new ArrayList<>();
            inBuff = new byte[1024];

            while (true) {
//                Log.d("UsbPrinter Serial", "" + mUsbDeviceConnection.getSerial());
//                Log.d("UsbPrinter Descriptors", new String(mUsbDeviceConnection.getRawDescriptors()));

                length = mUsbDeviceConnection.bulkTransfer(mUsbInputPoint, inBuff, inBuff.length, 300);

                if (length <= 0){
                    inBuff = null;
                    break;
                }

                for (int i = 0; i < length; ++i){
                    listInBuff.add(inBuff[i]);
                }
            }

            inBuff = new byte[listInBuff.size()];
            for (int i = 0; i < listInBuff.size(); ++i){
                inBuff[i] = listInBuff.get(i);
            }
        }

        if (mIsConnect && length < 0 && null == mUsbDeviceConnection.getSerial()){
            mIsConnect = false;

//            Log.d("UsbPrinter Serial", "" + mUsbDeviceConnection.getSerial());

            if (mPrinterStatusListener != null){
                mPrinterStatusListener.disconnect(PrinterType.printToUsb, DeviceCode.ErrorDisconnect.getCode());
            }
        }

        return inBuff;
    }

}
