package cn.com.broadlink.tool.libs.common;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.ActivityManager;
import android.app.Application;
import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.graphics.drawable.Drawable;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

public final class BLAppUtils {

	@SuppressLint("StaticFieldLeak")
	private static Application sApplication;

	private static final ActivityLifecycleImpl ACTIVITY_LIFECYCLE = new ActivityLifecycleImpl();

	private BLAppUtils() {
		throw new UnsupportedOperationException("u can't instantiate me...");
	}

	/**
	 * Init utils.
	 * <p>Init it in the class of Application.</p>
	 *
	 * @param context context
	 */
	public static void init(final Context context) {
		init((Application) context.getApplicationContext());
	}

	/**
	 * Init utils.
	 * <p>Init it in the class of Application.</p>
	 *
	 * @param app application
	 */
	public static void init(final Application app) {
		if (sApplication == null) {
			BLAppUtils.sApplication = app;
			BLAppUtils.sApplication.registerActivityLifecycleCallbacks(ACTIVITY_LIFECYCLE);
		}
	}

	public static LinkedList<Activity> getActivityList() {
		return ACTIVITY_LIFECYCLE.mActivityList;
	}

	public static Context getTopActivityOrApp() {
		if (isAppForeground()) {
			Activity topActivity = ACTIVITY_LIFECYCLE.getTopActivity();
			return topActivity == null ? getApp() : topActivity;
		} else {
			return getApp();
		}
	}

	public static boolean isAppForeground() {
		ActivityManager am = (ActivityManager) getApp().getSystemService(Context.ACTIVITY_SERVICE);
		if (am == null) return false;
		List<ActivityManager.RunningAppProcessInfo> info = am.getRunningAppProcesses();
		if (info == null || info.size() == 0) return false;
		for (ActivityManager.RunningAppProcessInfo aInfo : info) {
			if (aInfo.importance == ActivityManager.RunningAppProcessInfo.IMPORTANCE_FOREGROUND) {
				return aInfo.processName.equals(getApp().getPackageName());
			}
		}
		return false;
	}

	/**
	 * Return the context of Application object.
	 *
	 * @return the context of Application object
	 */
	public static Application getApp() {
		if (sApplication != null) return sApplication;
		try {
			@SuppressLint("PrivateApi")
			Class<?> activityThread = Class.forName("android.app.ActivityThread");
			Object at = activityThread.getMethod("currentActivityThread").invoke(null);
			Object app = activityThread.getMethod("getApplication").invoke(at);
			if (app == null) {
				throw new NullPointerException("u should init first");
			}
			init((Application) app);
			return sApplication;
		} catch (NoSuchMethodException e) {
			e.printStackTrace();
		} catch (IllegalAccessException e) {
			e.printStackTrace();
		} catch (InvocationTargetException e) {
			e.printStackTrace();
		} catch (ClassNotFoundException e) {
			e.printStackTrace();
		}
		throw new NullPointerException("u should init first");
	}

	/**
	 * Return whether it is a debug application.
	 *
	 * @return {@code true}: yes<br>{@code false}: no
	 */
	public static boolean isAppDebug() {
		return isAppDebug(getApp().getPackageName());
	}

	/**
	 * Return whether it is a debug application.
	 *
	 * @param packageName The name of the package.
	 * @return {@code true}: yes<br>{@code false}: no
	 */
	public static boolean isAppDebug(final String packageName) {
		if (isSpace(packageName)) return false;
		try {
			PackageManager pm = getApp().getPackageManager();
			ApplicationInfo ai = pm.getApplicationInfo(packageName, 0);
			return ai != null && (ai.flags & ApplicationInfo.FLAG_DEBUGGABLE) != 0;
		} catch (PackageManager.NameNotFoundException e) {
			e.printStackTrace();
			return false;
		}
	}

	/**
	 * Return whether it is a system application.
	 *
	 * @return {@code true}: yes<br>{@code false}: no
	 */
	public static boolean isAppSystem() {
		return isAppSystem(getApp().getPackageName());
	}

	/**
	 * Return whether it is a system application.
	 *
	 * @param packageName The name of the package.
	 * @return {@code true}: yes<br>{@code false}: no
	 */
	public static boolean isAppSystem(final String packageName) {
		if (isSpace(packageName)) return false;
		try {
			PackageManager pm = getApp().getPackageManager();
			ApplicationInfo ai = pm.getApplicationInfo(packageName, 0);
			return ai != null && (ai.flags & ApplicationInfo.FLAG_SYSTEM) != 0;
		} catch (PackageManager.NameNotFoundException e) {
			e.printStackTrace();
			return false;
		}
	}

	/**
	 * Exit the application.
	 */
	public static void exitApp() {
		List<Activity> activityList = getActivityList();
		for (int i = activityList.size() - 1; i >= 0; --i) {// remove from top
			Activity activity = activityList.get(i);
			// sActivityList remove the index activity at onActivityDestroyed
			activity.finish();
		}
		System.exit(0);
	}

	/**
	 * Return the application's icon.
	 *
	 * @return the application's icon
	 */
	public static Drawable getAppIcon() {
		return getAppIcon(getApp().getPackageName());
	}

	/**
	 * Return the application's icon.
	 *
	 * @param packageName The name of the package.
	 * @return the application's icon
	 */
	public static Drawable getAppIcon(final String packageName) {
		if (isSpace(packageName)) return null;
		try {
			PackageManager pm = getApp().getPackageManager();
			PackageInfo pi = pm.getPackageInfo(packageName, 0);
			return pi == null ? null : pi.applicationInfo.loadIcon(pm);
		} catch (PackageManager.NameNotFoundException e) {
			e.printStackTrace();
			return null;
		}
	}

	/**
	 * Return the application's package name.
	 *
	 * @return the application's package name
	 */
	public static String getAppPackageName() {
		return getApp().getPackageName();
	}

	/**
	 * Return the application's name.
	 *
	 * @return the application's name
	 */
	public static String getAppName() {
		return getAppName(getApp().getPackageName());
	}

	/**
	 * Return the application's name.
	 *
	 * @param packageName The name of the package.
	 * @return the application's name
	 */
	public static String getAppName(final String packageName) {
		if (isSpace(packageName)) return "";
		try {
			PackageManager pm = getApp().getPackageManager();
			PackageInfo pi = pm.getPackageInfo(packageName, 0);
			return pi == null ? null : pi.applicationInfo.loadLabel(pm).toString();
		} catch (PackageManager.NameNotFoundException e) {
			e.printStackTrace();
			return "";
		}
	}

	/**
	 * Return the application's path.
	 *
	 * @return the application's path
	 */
	public static String getAppPath() {
		return getAppPath(getApp().getPackageName());
	}

	/**
	 * Return the application's path.
	 *
	 * @param packageName The name of the package.
	 * @return the application's path
	 */
	public static String getAppPath(final String packageName) {
		if (isSpace(packageName)) return "";
		try {
			PackageManager pm = getApp().getPackageManager();
			PackageInfo pi = pm.getPackageInfo(packageName, 0);
			return pi == null ? null : pi.applicationInfo.sourceDir;
		} catch (PackageManager.NameNotFoundException e) {
			e.printStackTrace();
			return "";
		}
	}

	/**
	 * Return the application's version name.
	 *
	 * @return the application's version name
	 */
	public static String getAppVersionName() {
		return getAppVersionName(getApp().getPackageName());
	}

	/**
	 * Return the application's version name.
	 *
	 * @param packageName The name of the package.
	 * @return the application's version name
	 */
	public static String getAppVersionName(final String packageName) {
		if (isSpace(packageName)) return "";
		try {
			PackageManager pm = getApp().getPackageManager();
			PackageInfo pi = pm.getPackageInfo(packageName, 0);
			return pi == null ? null : pi.versionName;
		} catch (PackageManager.NameNotFoundException e) {
			e.printStackTrace();
			return "";
		}
	}

	/**
	 * Return the application's version code.
	 *
	 * @return the application's version code
	 */
	public static int getAppVersionCode() {
		return getAppVersionCode(getApp().getPackageName());
	}

	/**
	 * Return the application's version code.
	 *
	 * @param packageName The name of the package.
	 * @return the application's version code
	 */
	public static int getAppVersionCode(final String packageName) {
		if (isSpace(packageName)) return -1;
		try {
			PackageManager pm = getApp().getPackageManager();
			PackageInfo pi = pm.getPackageInfo(packageName, 0);
			return pi == null ? -1 : pi.versionCode;
		} catch (PackageManager.NameNotFoundException e) {
			e.printStackTrace();
			return -1;
		}
	}

	/**
	 * Return the application's information.
	 * <ul>
	 * <li>name of package</li>
	 * <li>icon</li>
	 * <li>name</li>
	 * <li>path of package</li>
	 * <li>version name</li>
	 * <li>version code</li>
	 * <li>is system</li>
	 * </ul>
	 *
	 * @return the application's information
	 */
	public static AppInfo getAppInfo() {
		return getAppInfo(getApp().getPackageName());
	}

	/**
	 * Return the application's information.
	 * <ul>
	 * <li>name of package</li>
	 * <li>icon</li>
	 * <li>name</li>
	 * <li>path of package</li>
	 * <li>version name</li>
	 * <li>version code</li>
	 * <li>is system</li>
	 * </ul>
	 *
	 * @param packageName The name of the package.
	 * @return 当前应用的 AppInfo
	 */
	public static AppInfo getAppInfo(final String packageName) {
		try {
			PackageManager pm = getApp().getPackageManager();
			PackageInfo pi = pm.getPackageInfo(packageName, 0);
			return getBean(pm, pi);
		} catch (PackageManager.NameNotFoundException e) {
			e.printStackTrace();
			return null;
		}
	}

	/**
	 * Return the applications' information.
	 *
	 * @return the applications' information
	 */
	public static List<AppInfo> getAppsInfo() {
		List<AppInfo> list = new ArrayList<>();
		PackageManager pm = getApp().getPackageManager();
		List<PackageInfo> installedPackages = pm.getInstalledPackages(0);
		for (PackageInfo pi : installedPackages) {
			AppInfo ai = getBean(pm, pi);
			if (ai == null) continue;
			list.add(ai);
		}
		return list;
	}

	private static AppInfo getBean(final PackageManager pm, final PackageInfo pi) {
		if (pm == null || pi == null) return null;
		ApplicationInfo ai = pi.applicationInfo;
		String packageName = pi.packageName;
		String name = ai.loadLabel(pm).toString();
		Drawable icon = ai.loadIcon(pm);
		String packagePath = ai.sourceDir;
		String versionName = pi.versionName;
		int versionCode = pi.versionCode;
		boolean isSystem = (ApplicationInfo.FLAG_SYSTEM & ai.flags) != 0;
		return new AppInfo(packageName, name, icon, packagePath, versionName, versionCode, isSystem);
	}

	private static boolean isSpace(final String s) {
		if (s == null) return true;
		for (int i = 0, len = s.length(); i < len; ++i) {
			if (!Character.isWhitespace(s.charAt(i))) {
				return false;
			}
		}
		return true;
	}

	/**
	 * The application's information.
	 */
	public static class AppInfo {
		private String   packageName;
		private String   name;
		private Drawable icon;
		private String   packagePath;
		private String   versionName;
		private int      versionCode;
		private boolean  isSystem;

		public Drawable getIcon() {
			return icon;
		}

		public void setIcon(final Drawable icon) {
			this.icon = icon;
		}

		public boolean isSystem() {
			return isSystem;
		}

		public void setSystem(final boolean isSystem) {
			this.isSystem = isSystem;
		}

		public String getPackageName() {
			return packageName;
		}

		public void setPackageName(final String packageName) {
			this.packageName = packageName;
		}

		public String getName() {
			return name;
		}

		public void setName(final String name) {
			this.name = name;
		}

		public String getPackagePath() {
			return packagePath;
		}

		public void setPackagePath(final String packagePath) {
			this.packagePath = packagePath;
		}

		public int getVersionCode() {
			return versionCode;
		}

		public void setVersionCode(final int versionCode) {
			this.versionCode = versionCode;
		}

		public String getVersionName() {
			return versionName;
		}

		public void setVersionName(final String versionName) {
			this.versionName = versionName;
		}

		public AppInfo(String packageName, String name, Drawable icon, String packagePath,
					   String versionName, int versionCode, boolean isSystem) {
			this.setName(name);
			this.setIcon(icon);
			this.setPackageName(packageName);
			this.setPackagePath(packagePath);
			this.setVersionName(versionName);
			this.setVersionCode(versionCode);
			this.setSystem(isSystem);
		}

		@Override
		public String toString() {
			return "pkg name: " + getPackageName() +
					"\napp icon: " + getIcon() +
					"\napp name: " + getName() +
					"\napp path: " + getPackagePath() +
					"\napp v name: " + getVersionName() +
					"\napp v code: " + getVersionCode() +
					"\nis system: " + isSystem();
		}
	}
}