package cn.bestwu.test.client;

import java.io.IOException;
import java.lang.reflect.Type;
import java.net.URI;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpRequest;
import org.springframework.http.MediaType;
import org.springframework.http.RequestEntity;
import org.springframework.http.ResponseEntity;
import org.springframework.http.client.ClientHttpRequest;
import org.springframework.http.client.ClientHttpRequestExecution;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.http.client.ClientHttpResponse;
import org.springframework.http.converter.ByteArrayHttpMessageConverter;
import org.springframework.http.converter.GenericHttpMessageConverter;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.converter.ResourceHttpMessageConverter;
import org.springframework.http.converter.StringHttpMessageConverter;
import org.springframework.http.converter.feed.AtomFeedHttpMessageConverter;
import org.springframework.http.converter.feed.RssChannelHttpMessageConverter;
import org.springframework.http.converter.json.GsonHttpMessageConverter;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.http.converter.xml.Jaxb2RootElementHttpMessageConverter;
import org.springframework.http.converter.xml.MappingJackson2XmlHttpMessageConverter;
import org.springframework.http.converter.xml.SourceHttpMessageConverter;
import org.springframework.util.Assert;
import org.springframework.util.Base64Utils;
import org.springframework.util.ClassUtils;
import org.springframework.web.client.DefaultResponseErrorHandler;
import org.springframework.web.client.HttpMessageConverterExtractor;
import org.springframework.web.client.RequestCallback;
import org.springframework.web.client.ResourceAccessException;
import org.springframework.web.client.ResponseExtractor;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

/**
 * 可在请求头中设置版本信息
 *
 * @author Peter Wu
 */
public class CustomRestTemplate extends RestTemplate {

  private boolean print;
  private static Charset defaultCharset = Charset.forName("UTF-8");

  public void setPrint(boolean print) {
    this.print = print;
  }

  public CustomRestTemplate() {
    this(null, null);
  }

  public CustomRestTemplate(String username, String password) {
    addAuthentication(username, password);
    setErrorHandler(new DefaultResponseErrorHandler() {
      @Override
      public void handleError(ClientHttpResponse response) {
      }
    });

    boolean romePresent =
        ClassUtils
            .isPresent("com.rometools.rome.feed.WireFeed", RestTemplate.class.getClassLoader());

    final boolean jaxb2Present =
        ClassUtils.isPresent("javax.xml.bind.Binder", RestTemplate.class.getClassLoader());

    final boolean jackson2Present =
        ClassUtils.isPresent("com.fasterxml.jackson.databind.ObjectMapper",
            RestTemplate.class.getClassLoader()) &&
            ClassUtils.isPresent("com.fasterxml.jackson.core.JsonGenerator",
                RestTemplate.class.getClassLoader());

    final boolean jackson2XmlPresent =
        ClassUtils.isPresent("com.fasterxml.jackson.dataformat.xml.XmlMapper",
            RestTemplate.class.getClassLoader());

    final boolean gsonPresent =
        ClassUtils.isPresent("com.google.gson.Gson", RestTemplate.class.getClassLoader());

    final List<HttpMessageConverter<?>> messageConverters = new ArrayList<>();
    messageConverters.add(new ByteArrayHttpMessageConverter());
    messageConverters.add(new StringHttpMessageConverter(defaultCharset));//UTF-8 编码
    messageConverters.add(new ResourceHttpMessageConverter()
    );
    messageConverters.add(new SourceHttpMessageConverter<>());
    messageConverters.add(new UTF8AllEncompassingFormHttpMessageConverter());

    if (romePresent) {
      messageConverters.add(new AtomFeedHttpMessageConverter());
      messageConverters.add(new RssChannelHttpMessageConverter());
    }

    if (jackson2XmlPresent) {
      messageConverters.add(new MappingJackson2XmlHttpMessageConverter());
    } else if (jaxb2Present) {
      messageConverters.add(new Jaxb2RootElementHttpMessageConverter());
    }

    if (jackson2Present) {
      messageConverters.add(new MappingJackson2HttpMessageConverter());
    } else if (gsonPresent) {
      messageConverters.add(new GsonHttpMessageConverter());
    }

    setMessageConverters(messageConverters);
  }

  private void addAuthentication(String username, String password) {
    if (username == null) {
      return;
    }
    List<ClientHttpRequestInterceptor> interceptors = new ArrayList<>(1);
    interceptors.add(new BasicAuthorizationInterceptor(username, password));
    setInterceptors(interceptors);
  }

  private static class BasicAuthorizationInterceptor
      implements ClientHttpRequestInterceptor {

    private final String username;

    private final String password;

    BasicAuthorizationInterceptor(String username, String password) {
      this.username = username;
      this.password = (password == null ? "" : password);
    }

    @Override
    public ClientHttpResponse intercept(HttpRequest request, byte[] body,
        ClientHttpRequestExecution execution) throws IOException {
      String token = Base64Utils.encodeToString(
          (this.username + ":" + this.password).getBytes(defaultCharset));
      request.getHeaders().add("Authorization", "Basic " + token);
      return execution.execute(request, body);
    }

  }

  private cn.bestwu.test.client.RequestCallback requestCallback;

  public void setRequestCallback(cn.bestwu.test.client.RequestCallback requestCallback) {
    this.requestCallback = requestCallback;
  }

  @Override
  protected <T> RequestCallback httpEntityCallback(Object requestBody) {
    return new HttpEntityRequestCallback(requestBody);
  }

  @Override
  protected <T> RequestCallback httpEntityCallback(Object requestBody, Type responseType) {
    return new HttpEntityRequestCallback(requestBody, responseType);
  }

  /**
   * Request callback implementation that prepares the request's accept headers.
   */
  private class AcceptHeaderRequestCallback implements RequestCallback {

    private final Type responseType;

    private AcceptHeaderRequestCallback(Type responseType) {
      this.responseType = responseType;
    }

    @Override
    public void doWithRequest(ClientHttpRequest request) throws IOException {
      if (this.responseType != null) {
        Class<?> responseClass = null;
        if (this.responseType instanceof Class) {
          responseClass = (Class<?>) this.responseType;
        }
        List<MediaType> allSupportedMediaTypes = new ArrayList<>();
        for (HttpMessageConverter<?> converter : getMessageConverters()) {
          if (responseClass != null) {
            if (converter.canRead(responseClass, null)) {
              allSupportedMediaTypes.addAll(getSupportedMediaTypes(converter));
            }
          } else if (converter instanceof GenericHttpMessageConverter) {
            GenericHttpMessageConverter<?> genericConverter = (GenericHttpMessageConverter<?>) converter;
            if (genericConverter.canRead(this.responseType, null, null)) {
              allSupportedMediaTypes.addAll(getSupportedMediaTypes(converter));
            }
          }
        }
        if (!allSupportedMediaTypes.isEmpty()) {
          MediaType.sortBySpecificity(allSupportedMediaTypes);
          if (logger.isDebugEnabled()) {
            logger.debug("Setting request Accept header to " + allSupportedMediaTypes);
          }
          request.getHeaders().setAccept(allSupportedMediaTypes);
        }
      }
    }

    private List<MediaType> getSupportedMediaTypes(HttpMessageConverter<?> messageConverter) {
      List<MediaType> supportedMediaTypes = messageConverter.getSupportedMediaTypes();
      List<MediaType> result = new ArrayList<>(supportedMediaTypes.size());
      for (MediaType supportedMediaType : supportedMediaTypes) {
        if (supportedMediaType.getCharset() != null) {
          supportedMediaType =
              new MediaType(supportedMediaType.getType(), supportedMediaType.getSubtype());
        }
        result.add(supportedMediaType);
      }
      return result;
    }
  }

  /**
   * Request callback implementation that writes the given object to the request stream.
   */
  private class HttpEntityRequestCallback extends AcceptHeaderRequestCallback {

    private final HttpEntity<?> requestEntity;

    private HttpEntityRequestCallback(Object requestBody) {
      this(requestBody, null);
    }

    private HttpEntityRequestCallback(Object requestBody, Type responseType) {
      super(responseType);
      if (requestBody instanceof HttpEntity) {
        this.requestEntity = (HttpEntity<?>) requestBody;
      } else if (requestBody != null) {
        this.requestEntity = new HttpEntity<>(requestBody);
      } else {
        this.requestEntity = HttpEntity.EMPTY;
      }
    }

    @Override
    @SuppressWarnings("unchecked")
    public void doWithRequest(ClientHttpRequest httpRequest) throws IOException {
      super.doWithRequest(httpRequest);
      if (requestCallback != null) {
        requestCallback.doWithRequest(httpRequest, requestEntity);
      }
      if (!this.requestEntity.hasBody()) {
        HttpHeaders httpHeaders = httpRequest.getHeaders();
        HttpHeaders requestHeaders = this.requestEntity.getHeaders();
        if (!requestHeaders.isEmpty()) {
          httpHeaders.putAll(requestHeaders);
        }
        if (httpHeaders.getContentLength() < 0) {
          httpHeaders.setContentLength(0L);
        }
      } else {
        Object requestBody = this.requestEntity.getBody();
        Class<?> requestBodyClass = requestBody.getClass();
        Type requestBodyType = (this.requestEntity instanceof RequestEntity ?
            ((RequestEntity<?>) this.requestEntity).getType() : requestBodyClass);
        HttpHeaders requestHeaders = this.requestEntity.getHeaders();
        MediaType requestContentType = requestHeaders.getContentType();
        for (HttpMessageConverter<?> messageConverter : getMessageConverters()) {
          if (messageConverter instanceof GenericHttpMessageConverter) {
            GenericHttpMessageConverter<Object> genericMessageConverter = (GenericHttpMessageConverter<Object>) messageConverter;
            if (genericMessageConverter
                .canWrite(requestBodyType, requestBodyClass, requestContentType)) {
              if (!requestHeaders.isEmpty()) {
                httpRequest.getHeaders().putAll(requestHeaders);
              }
              if (logger.isDebugEnabled()) {
                if (requestContentType != null) {
                  logger.debug("Writing [" + requestBody + "] as \"" + requestContentType +
                      "\" using [" + messageConverter + "]");
                } else {
                  logger.debug("Writing [" + requestBody + "] using [" + messageConverter + "]");
                }

              }
              genericMessageConverter.write(
                  requestBody, requestBodyType, requestContentType, httpRequest);
              return;
            }
          } else if (messageConverter.canWrite(requestBodyClass, requestContentType)) {
            if (!requestHeaders.isEmpty()) {
              httpRequest.getHeaders().putAll(requestHeaders);
            }
            if (logger.isDebugEnabled()) {
              if (requestContentType != null) {
                logger.debug("Writing [" + requestBody + "] as \"" + requestContentType +
                    "\" using [" + messageConverter + "]");
              } else {
                logger.debug("Writing [" + requestBody + "] using [" + messageConverter + "]");
              }

            }
            ((HttpMessageConverter<Object>) messageConverter).write(
                requestBody, requestContentType, httpRequest);
            return;
          }
        }
        String message =
            "Could not write request: no suitable HttpMessageConverter found for request type [" +
                requestBodyClass.getName() + "]";
        if (requestContentType != null) {
          message += " and content type [" + requestContentType + "]";
        }
        throw new RestClientException(message);
      }
    }
  }

  @Override
  @SuppressWarnings("unchecked")
  protected <T> T doExecute(URI url, HttpMethod method, RequestCallback requestCallback,
      ResponseExtractor<T> responseExtractor) throws RestClientException {

    Assert.notNull(url, "'url' must not be null");
    Assert.notNull(method, "'method' must not be null");
    ClientHttpResponse response = null;
    try {
      ClientHttpRequest request = createRequest(url, method);

      if (requestCallback != null) {
        requestCallback.doWithRequest(request);
      }

      HttpHeaders requestHeaders = request.getHeaders();

      response = request.execute();

      if (print && logger.isDebugEnabled()) {
        logger.debug("------------------------------");
        logger.debug("requestHeaders:");
        for (String key : requestHeaders.keySet()) {
          logger.debug(key + " : " + requestHeaders.get(key));
        }
        logger.debug("------------------------------");
      }

      HttpHeaders responseHeaders = response.getHeaders();
      if (print && logger.isDebugEnabled()) {
        logger.debug("------------------------------");
        logger.debug("responseHeaders:");
        for (String key : responseHeaders.keySet()) {
          logger.debug(key + " : " + responseHeaders.get(key));
        }
        logger.debug("------------------------------");
      }

      handleResponse(url, method, response);
      if (responseExtractor != null) {
        if (print && logger.isDebugEnabled()) {
          logger.debug("------------------------------");
          response = new PushbackBodyClientHttpResponseWrapper(response);
          logger.debug(
              "responseBody: \n" + ((PushbackBodyClientHttpResponseWrapper) response).readBody());
          logger.debug("------------------------------");
        }
        try {
          return responseExtractor.extractData(response);
        } catch (Exception e) {
          logger.warn("解析response body失败");
          return (T) new ResponseEntity<>(null, response.getHeaders(), response.getStatusCode());
        }
      } else {
        return null;
      }
    } catch (IOException ex) {
      throw new ResourceAccessException("I/O error on " + method.name() +
          " request for \"" + url + "\":" + ex.getMessage(), ex);
    } finally {
      if (response != null) {
        response.close();
      }
    }
  }

  public <T> T putForObject(String url, Object request, Class<T> responseType,
      Object... uriVariables) {
    RequestCallback requestCallback = httpEntityCallback(request, responseType);
    HttpMessageConverterExtractor<T> responseExtractor =
        new HttpMessageConverterExtractor<>(responseType, getMessageConverters());
    return execute(url, HttpMethod.PUT, requestCallback, responseExtractor, null, null,
        uriVariables);
  }


  public <T> ResponseEntity<T> putForEntity(String url, Object request, Class<T> responseType,
      Object... uriVariables)
      throws RestClientException {

    RequestCallback requestCallback = httpEntityCallback(request, responseType);
    ResponseExtractor<ResponseEntity<T>> responseExtractor = responseEntityExtractor(
        responseType);
    return execute(url, HttpMethod.PUT, requestCallback, responseExtractor, null, uriVariables);
  }

  public <T> ResponseEntity<T> deleteForEntity(String url, Object... uriVariables)
      throws RestClientException {
    ResponseExtractor<ResponseEntity<T>> responseExtractor = responseEntityExtractor(null);
    return execute(url, HttpMethod.DELETE, null, responseExtractor, null, uriVariables);
  }

  public <T> ResponseEntity<T> deleteForEntity(String url, Map<String, ?> urlVariables)
      throws RestClientException {
    ResponseExtractor<ResponseEntity<T>> responseExtractor = responseEntityExtractor(null);
    return execute(url, HttpMethod.DELETE, null, responseExtractor, null, urlVariables);
  }

}
