package cn.bestwu.gradle.profile

import cn.bestwu.gradle.profile.tasks.ProfileTask
import cn.bestwu.gradle.profile.tasks.doProfileActivate
import org.apache.tools.ant.filters.ReplaceTokens
import org.apache.tools.ant.types.resources.FileResource
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.distribution.DistributionContainer
import org.gradle.api.distribution.plugins.DistributionPlugin
import org.gradle.api.plugins.ApplicationPlugin
import org.gradle.api.plugins.ApplicationPluginConvention
import org.gradle.api.plugins.JavaPlugin
import org.gradle.api.plugins.WarPlugin
import org.gradle.api.tasks.application.CreateStartScripts
import org.gradle.jvm.tasks.Jar
import org.gradle.language.jvm.tasks.ProcessResources
import java.io.File
import java.util.*

/**
 *
 * 注册task
 *
 * @author Peter Wu
 */
class ProfilePlugin : Plugin<Project> {

    override fun apply(project: Project) {
        project.plugins.apply(JavaPlugin::class.java)
        project.plugins.apply(ApplicationPlugin::class.java)
        project.plugins.apply(WarPlugin::class.java)

        project.extensions.create("profile", ProfileExtension::class.java)
        project.tasks.addRule("Pattern: activate<ID>") { taskName ->
            val prefix = "activate"
            val length = prefix.length
            if (taskName.startsWith(prefix) && taskName.length > length) {
                var profile = taskName.substring(length)
                profile = profile[0].toLowerCase().toString() + profile.subSequence(1, profile.length)

                project.tasks.create(taskName, ProfileTask::class.java) { task ->
                    task.group = "profile"
                    task.dependsOn("clean")
                    task.doFirst {
                        project.extensions.getByType(ProfileExtension::class.java).active = profile
                    }
                    task.finalizedBy("processResources")
                }
            }
        }

        project.tasks.getByName("war") {
            it.enabled = true
            it.mustRunAfter("clean")
        }
        project.tasks.getByName("installDist") {
            it.mustRunAfter("clean")
        }
        project.tasks.getByName("assembleDist") {
            it.mustRunAfter("clean")
        }
        project.tasks.getByName("distTar") {
            it.mustRunAfter("clean")
        }
        project.tasks.getByName("distZip") {
            it.mustRunAfter("clean")
        }

        project.tasks.getByName("jar") { task ->
            task.enabled = true
            task as Jar
            task.manifest {
                it.attributes(mapOf("Manifest-Version" to project.version, "Implementation-Title"
                        to (project.findProperty("application.name") as? String
                        ?: project.name), "Implementation-Version" to project.version))
            }
        }

        project.afterEvaluate { _ ->
            if (project.extensions.getByType(ProfileExtension::class.java).unwrapResources) {
                project.tasks.getByName("jar") { task ->
                    task as Jar
                    task.exclude {
                        (project.tasks.getByName("processResources") as ProcessResources).destinationDir.listFiles().contains(it.file)
                    }
                }
                project.tasks.getByName("startScripts") { task ->
                    task as CreateStartScripts
                    task.classpath = project.files(task.classpath).from("\$APP_HOME/resources")
                    task.doLast {
                        it as CreateStartScripts
                        it.unixScript.writeText(it.unixScript.readText()
                                .replace("\$APP_HOME/lib/resources", "\$APP_HOME/resources")
                        )
                        it.windowsScript.writeText(it.windowsScript.readText()
                                .replace("%APP_HOME%\\lib\\resources", "%APP_HOME%\\resources")
                        )
                    }
                }

                project.extensions.getByType(DistributionContainer::class.java).getAt(DistributionPlugin.MAIN_DISTRIBUTION_NAME).contents { copySpec ->
                    copySpec.from((project.tasks.getByName("processResources") as ProcessResources).destinationDir) {
                        it.into("resources")
                    }
                }
            }
        }

        project.tasks.getByName("processTestResources") {
            it as ProcessResources
            doFilter(it, project)
            it.inputs.file(project.rootProject.file("gradle.properties"))
        }

        val processResources = project.tasks.getByName("processResources") {
            it as ProcessResources
            doFilter(it, project)
            it.inputs.file(project.rootProject.file("gradle.properties"))
        }
        project.tasks.create("activateDefault", ProcessResources::class.java) { resources ->
            processResources as ProcessResources
            resources.group = "profile"
            resources.destinationDir = processResources.destinationDir
            resources.from(processResources.source)
            resources.doFirst {
                processResources.destinationDir.deleteRecursively()
                File(project.buildDir, "tmp").deleteRecursively()
                val extension = project.extensions.getByType(ProfileExtension::class.java)
                extension.active = extension.defaultActive
                doProfileActivate(project)
            }
            doFilter(resources, project)
        }
        (project.convention.plugins["application"] as ApplicationPluginConvention).applicationDefaultJvmArgs += "-Dfile.encoding=UTF-8"

        project.tasks.getByName("compileJava") {
            it.dependsOn("processResources")
        }

        project.tasks.whenTaskAdded { task ->
            val taskName: String = task.name
            val prefix = "publish"
            val length = prefix.length
            val profile = project.extensions.getByType(ProfileExtension::class.java)
            var actives = profile.actives
            actives += profile.defaultActive
            val defaultProfile = middlePublishTaskName(profile.defaultActive)

            if (taskName.startsWith(prefix) && taskName.length > length && !containsActive(actives, taskName)) {
                for (active in actives) {
                    val activeProfile = middlePublishTaskName(active)
                    val activeTaskName = "publish" + activeProfile + taskName.substring(length)

                    val activeTask = project.tasks.findByName("activate$activeProfile")

                    project.tasks.create(activeTaskName) {
                        it.group = "publish"
                        it.dependsOn(activeTask, taskName)
                        if (active != defaultProfile) {
                            it.finalizedBy("activateDefault")
                        }
                    }
                    project.tasks.findByName(taskName)?.mustRunAfter(activeTask)
                }
            }
        }
    }

    private fun doFilter(it: ProcessResources, project: Project) {
        it.filesMatching(setOf("**/*.yml", "**/*.yaml", "**/*.properties", "**/*.xml")) {
            val profileExtension = project.extensions.getByType(ProfileExtension::class.java)
            val gradleProperties = project.rootProject.file("gradle.properties")
            val props = Properties()
            props.load(gradleProperties.inputStream())
            props["profiles.active"] = profileExtension.active
            val activeFileName = profileExtension.activeFileName
            val rootActiveFile = project.rootProject.file(activeFileName)
            if (rootActiveFile.exists()) {
                props.load(rootActiveFile.inputStream())
            }
            val activeFile = project.file(activeFileName)
            if (activeFile.exists()) {
                props.load(activeFile.inputStream())
            }
            val fileResource = FileResource(File(project.buildDir, "tmp/${profileExtension.active}.properties"))
            props.store(fileResource.outputStream, "合成的配置文件")
            it.filter(mapOf("propertiesResource" to fileResource), ReplaceTokens::class.java)
        }
    }

    private fun containsActive(actives: Array<String>, taskName: String): Boolean {
        return actives
                .map { middlePublishTaskName(it) }
                .any { taskName.startsWith("publish$it") }
    }

    private fun middlePublishTaskName(active: String) = active[0].toUpperCase().toString() + active.subSequence(1, active.length)
}