package cn.bestwu.generator.puml

import cn.bestwu.generator.DatabaseDriver
import cn.bestwu.generator.GeneratorExtension
import cn.bestwu.generator.database.domain.Column
import cn.bestwu.generator.database.domain.Indexed
import cn.bestwu.generator.database.domain.Table
import cn.bestwu.generator.dom.java.JavaTypeResolver
import cn.bestwu.generator.dsl.Generators
import cn.bestwu.generator.dsl.def.PlantUML
import java.io.File

/**
 * @author Peter Wu
 * @since 0.0.45
 */
object PumlConverter {

    fun toTables(puml: File, call: (Table) -> Unit = {}): List<Table> {
        return toTableOrAnys(puml, call).asSequence().filter { it is Table }.map { it as Table }.toList()
    }

    fun toTableOrAnys(puml: File, call: (Table) -> Unit = {}): List<Any> {
        val tables = mutableListOf<Any>()
        var remarks = ""
        var primaryKeyNames = mutableListOf<String>()
        var indexes = mutableListOf<Indexed>()
        var pumlColumns = mutableListOf<Any>()
        var tableName = ""
        var desc = "<<(T,#DDDDDD)>>"
        var sequenceStartWith: Int? = null
        var isField = false
        var isUml = false
        puml.readLines().forEach { it ->
            if (it.isNotBlank()) {
                val line = it.trim()
                if (line.startsWith("@startuml")) {
                    PlantUML.umlName = line.substringAfter("@startuml").trim()
                    isUml = true
                } else if (line.startsWith("class")) {
                    val fieldDef = line.split(" ")
                    tableName = fieldDef[1].trim()
                    if (fieldDef.size > 3) {
                        val tableDesc = fieldDef[2]
                        desc = tableDesc
                        sequenceStartWith = if ("<<(T,#DDDDDD)>>" == tableDesc)
                            1
                        else
                            tableDesc.replace(Regex("<<\\(T,#DDDDD(\\d)\\)>>"), "$1").toIntOrNull()
                    }
                } else if (tableName.isNotBlank() && !isField) {
                    if ("==" == line)
                        isField = true
                    else
                        remarks = line
                } else if (isField) {
                    val uniqueMult = line.startsWith("'UNIQUE")
                    if (uniqueMult || line.startsWith("'INDEX")) {
                        val columnNames = line.substringAfter(if (uniqueMult) "'UNIQUE" else "'INDEX").trim()
                        indexes.add(Indexed("${if (uniqueMult) "UK" else "IDX"}_${tableName.replace("_", "").takeLast(7)}_${columnNames.replace(tableName, "").replace("_", "").replace(",", "").takeLast(7)}", uniqueMult, columnNames.split(",").toMutableList()))
                    } else if ("}" == line) {
                        val table = Table(productName = Generators.puml, catalog = null, schema = null, tableName = tableName, tableType = "", remarks = remarks, primaryKeyNames = primaryKeyNames, indexes = indexes, pumlColumns = pumlColumns, desc = desc, sequenceStartWith = sequenceStartWith)
                        call(table)
                        tables.add(table)

                        primaryKeyNames = mutableListOf()
                        indexes = mutableListOf()
                        pumlColumns = mutableListOf()
                        tableName = ""
                        remarks = ""
                        desc = "<<(T,#DDDDDD)>>"
                        sequenceStartWith = null
                        isField = false
                    } else if (!line.startsWith("'")) {
                        val lineDef = line.trim().split("--")
                        val fieldDef = lineDef[0].trim()
                        val fieldDefs = fieldDef.split(" ")
                        val columnName = fieldDefs[0]
                        val columnDef = fieldDef.substringAfter(columnName).replace(":", "").trim()
                        val type = columnDef.split(" ")[0].trim()
                        var extra = columnDef.substringAfter(type)
                        val (columnSize, decimalDigits) = parseType(type)
                        var defaultVal: String? = null
                        if (columnDef.contains("DEFAULT")) {
                            defaultVal = columnDef.substringAfter("DEFAULT").trim().substringBefore(" ").trim('\'').trim()
                            extra = extra.replace(Regex(" DEFAULT +'?$defaultVal'?"), "")
                        }
                        var fk = false
                        var refTable = ""
                        var refColumn = ""
                        if (columnDef.contains("FK")) {//FK > docs.id
                            val ref = columnDef.substringAfter("FK >").trim().substringBefore(" ").trim()
                            extra = extra.replace(Regex(" FK > +$ref"), "")
                            val refs = ref.split(".")
                            fk = true
                            refTable = refs[0]
                            refColumn = refs[1]
                        }
                        val typeName = type.substringBefore("(")
                        val unique = columnDef.contains("UNIQUE")
                        val indexed = columnDef.contains("INDEX")
                        val autoIncrement = columnDef.contains("AUTO_INCREMENT")
                        extra = extra.replace(" UNIQUE", "", true)
                        extra = extra.replace(" INDEX", "", true)
                        extra = extra.replace(" AUTO_INCREMENT", "", true)
                        extra = extra.replace(" NOT NULL", "", true)
                        extra = extra.replace(" NULL", "", true)
                        extra = extra.replace(" PK", "", true).trim()
                        val column = Column(tableCat = null, columnName = columnName, remarks = lineDef.last().trim(), typeName = typeName, columnSize = columnSize, decimalDigits = decimalDigits, nullable = !columnDef.contains("NOT NULL"), unique = unique, indexed = indexed, columnDef = defaultVal, extra = extra, dataType = JavaTypeResolver.calculateDataType(JavaTypeResolver.calculateJdbcTypeName(typeName))
                                ?: 0, tableSchem = null, isForeignKey = fk, pktableName = refTable, pkcolumnName = refColumn, autoIncrement = autoIncrement)
                        if (unique)
                            indexes.add(Indexed("UK_${tableName.replace("_", "").takeLast(7)}_${columnName.replace(tableName, "").replace("_", "").replace(",", "").takeLast(7)}", true, mutableListOf(columnName)))
                        if (indexed)
                            indexes.add(Indexed("IDX_${tableName.replace("_", "").takeLast(7)}_${columnName.replace(tableName, "").replace("_", "").replace(",", "").takeLast(7)}", false, mutableListOf(columnName)))
                        if (columnDef.contains("PK")) {
                            column.isPrimary = true
                            primaryKeyNames.add(column.columnName)
                        }
                        pumlColumns.add(column)
                    } else {
                        pumlColumns.add(line)
                    }
                } else if (line.startsWith("@enduml")) {
                    isUml = false
                } else if (isUml && line.isNotBlank() && !line.matches(Regex("^.* \"1\" -- \"0\\.\\.\\*\" .*$"))) {
                    tables.add(line)
                }
            }
        }

        return tables
    }

    fun parseType(type: String): Pair<Int, Int> {
        var columnSize = 0
        var decimalDigits = 0
        if (type.contains("(")) {
            val lengthScale = type.substringAfter("(").substringBefore(")")
            if (lengthScale.contains(",")) {
                val ls = lengthScale.split(",")
                columnSize = ls[0].toInt()
                decimalDigits = ls[1].toInt()
            } else {
                columnSize = lengthScale.toInt()
            }
        }
        return Pair(columnSize, decimalDigits)
    }

    private fun compile(extension: GeneratorExtension, tables: List<Any>, out: File) {
        val plantUML = PlantUML(out.absolutePath)
        plantUML.setUp()
        tables.forEach {
            if (it is Table) {
                plantUML.call(extension, it)
            } else {
                plantUML.appendlnText(it.toString())
            }
        }
        plantUML.tearDown()
    }

    fun compile(extension: GeneratorExtension) {
        extension.pumlSources.forEach { src ->
            when (extension.pumlDatabaseDriver) {
                DatabaseDriver.ORACLE -> PumlConverter.toOracle(extension, src, extension.pumlOutputFile(src))
                DatabaseDriver.MYSQL -> PumlConverter.toMysql(extension, src, extension.pumlOutputFile(src))
                else -> compile(extension, src, extension.pumlOutputFile(src))
            }
        }
    }

    fun reformat(extension: GeneratorExtension) {
        extension.pumlSrcSources.forEach {
            compile(extension, it, it)
        }
    }

    fun compile(extension: GeneratorExtension, src: File, out: File) {
        compile(extension, toTableOrAnys(src), out)
    }

    fun toMysql(extension: GeneratorExtension, src: File, out: File) {
        val tables = toTableOrAnys(src)
        tables.forEach { t ->
            if (t is Table) {
                t.pumlColumns.forEach {
                    if (it is Column) {
                        when (it.typeName) {
                            "VARCHAR2" -> it.typeName = "VARCHAR"
                            "RAW" -> it.typeName = "BINARY"
                            "CLOB" -> it.typeName = "TEXT"
                            "NUMBER" -> {
                                if (it.decimalDigits == 0) {
                                    when (it.columnSize) {
                                        in 1..3 -> {
                                            it.typeName = "TINYINT"
                                        }
                                        in 4..5 -> {
                                            it.typeName = "SMALLINT"
                                        }
                                        in 6..7 -> {
                                            it.typeName = "MEDUIMINT"
                                        }
                                        in 8..10 -> {
                                            it.typeName = "INT"
                                        }
                                        in 11..20 -> {
                                            it.typeName = "BIGINT"
                                        }
                                        else -> it.typeName = "DECIMAL"
                                    }
                                } else {
                                    it.typeName = "DECIMAL"
                                }
                            }
                        }
                    }
                }
            }
        }
        compile(extension, tables, out)
    }

    fun toOracle(extension: GeneratorExtension, src: File, out: File) {
        val tables = toTableOrAnys(src)
        tables.forEach { t ->
            if (t is Table) {
                t.pumlColumns.forEach {
                    if (it is Column) {
                        when (it.typeName) {
                            "VARCHAR" -> it.typeName = "VARCHAR2"
                            "TINYINT" -> {
                                it.typeName = "NUMBER"
                            }
                            "SMALLINT" -> {
                                it.typeName = "NUMBER"
                            }
                            "MEDUIMINT" -> {
                                it.typeName = "NUMBER"
                            }
                            "INT" -> {
                                it.typeName = "NUMBER"
                            }
                            "BIGINT" -> {
                                it.typeName = "NUMBER"
                            }
                            "FLOAT", "DOUBLE", "DECIMAL" -> {
                                it.typeName = "NUMBER"
                            }
                            "TINYTEXT" -> it.typeName = "CLOB"
                            "TINYBLOB" -> it.typeName = "BLOB"
                            "BINARY" -> it.typeName = "RAW"
                            "TEXT" -> it.typeName = "CLOB"
                            "LONGTEXT" -> it.typeName = "CLOB"
                        }
                    }
                }
            }
        }
        compile(extension, tables, out)
    }

    fun toDDL(extension: GeneratorExtension) {
        MysqlToDDL.useQuote = extension.sqlQuote
        OracleToDDL.useQuote = extension.sqlQuote
        MysqlToDDL.useForeignKey = extension.useForeignKey
        OracleToDDL.useForeignKey = extension.useForeignKey
        extension.pumlSrcSources.forEach {
            when (extension.pumlDatabaseDriver) {
                DatabaseDriver.MYSQL -> MysqlToDDL.toDDL(it, extension.pumlSqlOutputFile(it))
                DatabaseDriver.ORACLE -> OracleToDDL.toDDL(it, extension.pumlSqlOutputFile(it))
                else -> {
                }
            }
        }
    }

    fun toDDLUpdate(extension: GeneratorExtension) {
        MysqlToDDL.useQuote = extension.sqlQuote
        OracleToDDL.useQuote = extension.sqlQuote
        MysqlToDDL.useForeignKey = extension.useForeignKey
        OracleToDDL.useForeignKey = extension.useForeignKey
        extension.pumlSqlUpdateOutputFile().printWriter().use { pw ->
            extension.pumlSrcSources.forEach {
                val pumlOutputFile = extension.pumlOutputFile(it)
                if (pumlOutputFile.exists()) {
                    when (extension.pumlDatabaseDriver) {
                        DatabaseDriver.MYSQL -> MysqlToDDL.toDDLUpdate(pumlOutputFile, it, pw)
                        DatabaseDriver.ORACLE -> OracleToDDL.toDDLUpdate(pumlOutputFile, it, pw)
                        else -> {
                        }
                    }
                }
            }
        }

    }

}
