package cn.bestwu.generator.puml

import cn.bestwu.generator.database.domain.Table
import java.io.File
import java.io.PrintWriter

object OracleToDDL : ToDDL() {
    override val quoteMark: String = "\""
    override val commentPrefix: String = "--"

    override fun toDDLUpdate(old: File, src: File, out: PrintWriter) {
        val tables = PumlConverter.toTables(src)
        val oldTables = PumlConverter.toTables(old)
        if (tables != oldTables) {
            val tableNames = tables.map { it.tableName }
            val oldTableNames = oldTables.map { it.tableName }
            (oldTableNames - tableNames).forEach { tableName ->
                out.println("$commentPrefix $tableName")
                if (oldTables.find { it.tableName == tableName }!!.sequenceStartWith != null)
                    out.println("DROP SEQUENCE $quote${tableName}_S$quote;")
                out.println("DROP TABLE $quote$tableName$quote;")
                out.println()
            }
            val newTableNames = tableNames - oldTableNames
            tables.forEach { table ->
                val tableName = table.tableName
                if (newTableNames.contains(tableName)) {
                    appendTable(table, out)
                } else {
                    val oldTable = oldTables.find { it.tableName == tableName }!!
                    if (oldTable != table)
                        out.println("$commentPrefix $tableName")
                    val oldColumns = oldTable.columns
                    val columns = table.columns
                    val oldPrimaryKeys = oldTable.primaryKeys
                    val primaryKeys = table.primaryKeys
                    val oldPrimaryKey = oldPrimaryKeys[0]
                    val primaryKey = primaryKeys[0]
                    if (oldPrimaryKeys.size == 1 && primaryKeys.size == 1 && oldPrimaryKey != primaryKey) {
                        out.println("ALTER TABLE $quote$tableName$quote RENAME COLUMN $quote${oldPrimaryKey.columnName}$quote TO $quote${primaryKey.columnName}$quote;")
                        out.println("ALTER TABLE $quote$tableName$quote MODIFY (${columnDef(primaryKey, quote)});")
                        out.println("COMMENT ON COLUMN $quote$tableName$quote.$quote${primaryKey.columnName}$quote IS '${primaryKey.remarks}';")
                        oldColumns.remove(oldPrimaryKey)
                        columns.remove(primaryKey)
                    }
                    val oldColumnNames = oldColumns.map { it.columnName }
                    val columnNames = columns.map { it.columnName }
                    val dropColumnNames = oldColumnNames - columnNames
                    if (dropColumnNames.isNotEmpty()) {
                        out.println("ALTER TABLE $quote$tableName$quote DROP (${dropColumnNames.joinToString(",") { "$quote$it$quote" }});")
                    }
                    dropFk(oldColumns, dropColumnNames, out, tableName)
                    val newColumnNames = columnNames - oldColumnNames
                    columns.forEach { column ->
                        val columnName = column.columnName
                        if (newColumnNames.contains(columnName)) {
                            out.println("ALTER TABLE $quote$tableName$quote ADD (${columnDef(column, quote)});")
                            out.println("COMMENT ON COLUMN $quote$tableName$quote.$quote$columnName$quote IS '${column.remarks}';")
                            addFk(column, out, tableName, columnName)
                        } else {
                            val oldColumn = oldColumns.find { it.columnName == columnName }!!
                            if (column != oldColumn) {
                                out.println("ALTER TABLE $quote$tableName$quote MODIFY (${columnDef(column, quote)});")
                                out.println("COMMENT ON COLUMN $quote$tableName$quote.$quote$columnName$quote IS '${column.remarks}';")
                                updateFk(column, oldColumn, out, tableName)
                            }
                        }
                    }

                    updateIndexes(oldTable, table, out)

                    if (oldTable != table)
                        out.println()
                }
            }
        }
    }

    override fun appendTable(table: Table, pw: PrintWriter) {
        val tableName = table.tableName
        pw.println("$commentPrefix $tableName")
        if (table.sequenceStartWith != null) {
            pw.println("DROP SEQUENCE $quote${tableName}_S$quote;")
            pw.println("CREATE SEQUENCE $quote${tableName}_S$quote INCREMENT BY 1 START WITH 1${fill(table.sequenceStartWith!!)};")
        }
        pw.println()
        pw.println("DROP TABLE $quote$tableName$quote;")
        pw.println("CREATE TABLE $quote$tableName$quote (")
        val hasPrimary = table.primaryKeyNames.isNotEmpty()
        val lastIndex = table.columns.size - 1
        table.columns.forEachIndexed { index, column ->
            pw.println("  ${columnDef(column, quote)}${if (index < lastIndex || hasPrimary) "," else ""}")
        }
        appendKeys(table, hasPrimary, pw, quote, tableName, useForeignKey)
        pw.println(");")
        appendIndexes(table, pw, quote)

        pw.println("COMMENT ON TABLE $quote$tableName$quote IS '${table.remarks}';")
        table.columns.forEach {
            pw.println("COMMENT ON COLUMN $quote$tableName$quote.$quote${it.columnName}$quote IS '${it.remarks}';")
        }
        pw.println()
    }


    private fun fill(length: Int): String {
        var str = ""
        for (i in 1..length) {
            str += "0"
        }
        return str
    }

}