package cn.bestwu.generator.puml

import cn.bestwu.generator.database.domain.Column
import cn.bestwu.generator.database.domain.Indexed
import cn.bestwu.generator.database.domain.Table
import cn.bestwu.generator.dom.java.JavaTypeResolver
import cn.bestwu.generator.dsl.Generators
import cn.bestwu.generator.dsl.def.PlantUML
import java.io.File
import java.io.PrintWriter

/**
 * @author Peter Wu
 * @since 0.0.45
 */
object PumlConverter {

    fun toTables(puml: File, call: (Table) -> Unit = {}): List<Table> {
        return toTableOrAnys(puml, call).asSequence().filter { it is Table }.map { it as Table }.toList()
    }

    fun toTableOrAnys(puml: File, call: (Table) -> Unit = {}): List<Any> {
        val tables = mutableListOf<Any>()
        var remarks = ""
        var primaryKeyNames = mutableListOf<String>()
        var indexes = mutableListOf<Indexed>()
        var pumlColumns = mutableListOf<Any>()
        var tableName = ""
        var desc = "<<(T,#DDDDDD)>>"
        var sequenceStartWith: Int? = null
        var isField = false
        var isUml = false
        puml.readLines().forEach { it ->
            if (it.isNotBlank()) {
                val line = it.trim()
                if (line.startsWith("@startuml")) {
                    PlantUML.umlName = line.substringAfter("@startuml").trim()
                    isUml = true
                } else if (line.startsWith("class")) {
                    val fieldDef = line.split(" ")
                    tableName = fieldDef[1].trim()
                    if (fieldDef.size > 3) {
                        val tableDesc = fieldDef[2]
                        desc = tableDesc
                        sequenceStartWith = if ("<<(T,#DDDDDD)>>" == tableDesc)
                            1
                        else
                            tableDesc.replace(Regex("<<\\(T,#DDDDD(\\d)\\)>>"), "$1").toIntOrNull()
                    }
                } else if (tableName.isNotBlank() && !isField) {
                    if ("==" == line)
                        isField = true
                    else
                        remarks = line
                } else if (isField) {
                    val uniqueMult = line.startsWith("'UNIQUE")
                    if (uniqueMult || line.startsWith("'INDEX")) {
                        val columnNames = line.substringAfter(if (uniqueMult) "'UNIQUE" else "'INDEX").trim()
                        indexes.add(Indexed("${if (uniqueMult) "UK" else "IDX"}_${tableName.replace("_", "").takeLast(7)}_${columnNames.replace(tableName, "").replace("_", "").replace(",", "").takeLast(7)}", uniqueMult, columnNames.split(",").toMutableList()))
                    } else if ("}" == line) {
                        val table = Table(productName = Generators.puml, catalog = null, schema = null, tableName = tableName, tableType = "", remarks = remarks, primaryKeyNames = primaryKeyNames, indexes = indexes, pumlColumns = pumlColumns, desc = desc, sequenceStartWith = sequenceStartWith)
                        call(table)
                        tables.add(table)

                        primaryKeyNames = mutableListOf()
                        indexes = mutableListOf()
                        pumlColumns = mutableListOf()
                        tableName = ""
                        remarks = ""
                        desc = "<<(T,#DDDDDD)>>"
                        sequenceStartWith = null
                        isField = false
                    } else if (!line.startsWith("'")) {
                        val lineDef = line.trim().split("--")
                        val fieldDef = lineDef[0].trim()
                        val fieldDefs = fieldDef.split(" ")
                        val columnName = fieldDefs[0]
                        val columnDef = fieldDef.substringAfter(columnName).replace(":", "").trim()
                        val type = columnDef.split(" ")[0].trim()
                        var extra = columnDef.substringAfter(type)
                        val (columnSize, decimalDigits) = parseType(type)
                        var defaultVal: String? = null
                        if (columnDef.contains("DEFAULT")) {
                            defaultVal = columnDef.substringAfter("DEFAULT").trim().substringBefore(" ").trim('\'').trim()
                            extra = extra.replace(Regex(" DEFAULT +'?$defaultVal'?"), "")
                        }
                        var fk = false
                        var refTable = ""
                        var refColumn = ""
                        if (columnDef.contains("FK")) {//FK > docs.id
                            val ref = columnDef.substringAfter("FK >").trim().substringBefore(" ").trim()
                            extra = extra.replace(Regex(" FK > +$ref"), "")
                            val refs = ref.split(".")
                            fk = true
                            refTable = refs[0]
                            refColumn = refs[1]
                        }
                        val typeName = type.substringBefore("(")
                        val unique = columnDef.contains("UNIQUE")
                        val indexed = columnDef.contains("INDEX")
                        val autoIncrement = columnDef.contains("AUTO_INCREMENT")
                        extra = extra.replace(" UNIQUE", "", true)
                        extra = extra.replace(" INDEX", "", true)
                        extra = extra.replace(" AUTO_INCREMENT", "", true)
                        extra = extra.replace(" NOT NULL", "", true)
                        extra = extra.replace(" NULL", "", true)
                        extra = extra.replace(" PK", "", true).trim()
                        val column = Column(tableCat = null, columnName = columnName, remarks = lineDef.last().trim(), typeName = typeName, columnSize = columnSize, decimalDigits = decimalDigits, nullable = !columnDef.contains("NOT NULL"), unique = unique, indexed = indexed, columnDef = defaultVal, extra = extra, dataType = JavaTypeResolver.calculateDataType(JavaTypeResolver.calculateJdbcTypeName(typeName))
                                ?: 0, tableSchem = null, isForeignKey = fk, pktableName = refTable, pkcolumnName = refColumn, autoIncrement = autoIncrement)
                        if (unique)
                            indexes.add(Indexed("UK_${tableName.replace("_", "").takeLast(7)}_${columnName.replace(tableName, "").replace("_", "").replace(",", "").takeLast(7)}", true, mutableListOf(columnName)))
                        if (indexed)
                            indexes.add(Indexed("IDX_${tableName.replace("_", "").takeLast(7)}_${columnName.replace(tableName, "").replace("_", "").replace(",", "").takeLast(7)}", false, mutableListOf(columnName)))
                        if (columnDef.contains("PK")) {
                            column.isPrimary = true
                            primaryKeyNames.add(column.columnName)
                        }
                        pumlColumns.add(column)
                    } else {
                        pumlColumns.add(line)
                    }
                } else if (line.startsWith("@enduml")) {
                    isUml = false
                } else if (isUml && line.isNotBlank()) {
                    tables.add(line)
                }
            }
        }

        return tables
    }

    fun parseType(type: String): Pair<Int, Int> {
        var columnSize = 0
        var decimalDigits = 0
        if (type.contains("(")) {
            val lengthScale = type.substringAfter("(").substringBefore(")")
            if (lengthScale.contains(",")) {
                val ls = lengthScale.split(",")
                columnSize = ls[0].toInt()
                decimalDigits = ls[1].toInt()
            } else {
                columnSize = lengthScale.toInt()
            }
        }
        return Pair(columnSize, decimalDigits)
    }

    private fun compile(tables: List<Any>, out: File) {
        val plantUML = PlantUML(out.absolutePath)
        plantUML.setUp()
        tables.forEach {
            if (it is Table) {
                plantUML.call(it)
            } else {
                plantUML.appendlnText(it.toString())
            }
        }
        plantUML.tearDown()
    }

    fun compile(src: File, out: File) {
        compile(toTableOrAnys(src), out)
    }

    fun toMysql(src: File, out: File) {
        val tables = toTableOrAnys(src)
        tables.forEach { t ->
            if (t is Table) {
                t.pumlColumns.forEach {
                    if (it is Column) {
                        when (it.typeName) {
                            "VARCHAR2" -> it.typeName = "VARCHAR"
                            "RAW" -> it.typeName = "BINARY"
                            "CLOB" -> it.typeName = "TEXT"
                            "NUMBER" -> {
                                if (it.decimalDigits == 0) {
                                    when (it.columnSize) {
                                        in 1..3 -> {
                                            it.typeName = "TINYINT"
                                        }
                                        in 4..5 -> {
                                            it.typeName = "SMALLINT"
                                        }
                                        in 6..7 -> {
                                            it.typeName = "MEDUIMINT"
                                        }
                                        in 8..10 -> {
                                            it.typeName = "INT"
                                        }
                                        in 11..20 -> {
                                            it.typeName = "BIGINT"
                                        }
                                        else -> it.typeName = "DECIMAL"
                                    }
                                } else {
                                    it.typeName = "DECIMAL"
                                }
                            }
                        }
                    }
                }
            }
        }
        compile(tables, out)
    }

    fun toOracle(src: File, out: File) {
        val tables = toTableOrAnys(src)
        tables.forEach { t ->
            if (t is Table) {
                t.pumlColumns.forEach {
                    if (it is Column) {
                        when (it.typeName) {
                            "VARCHAR" -> it.typeName = "VARCHAR2"
                            "TINYINT" -> {
                                it.typeName = "NUMBER"
                            }
                            "SMALLINT" -> {
                                it.typeName = "NUMBER"
                            }
                            "MEDUIMINT" -> {
                                it.typeName = "NUMBER"
                            }
                            "INT" -> {
                                it.typeName = "NUMBER"
                            }
                            "BIGINT" -> {
                                it.typeName = "NUMBER"
                            }
                            "FLOAT", "DOUBLE", "DECIMAL" -> {
                                it.typeName = "NUMBER"
                            }
                            "TINYTEXT" -> it.typeName = "CLOB"
                            "TINYBLOB" -> it.typeName = "BLOB"
                            "BINARY" -> it.typeName = "RAW"
                            "TEXT" -> it.typeName = "CLOB"
                            "LONGTEXT" -> it.typeName = "CLOB"
                        }
                    }
                }
            }
        }
        compile(tables, out)
    }

    fun toMySqlDDL(src: File, out: File, sqlQuote: Boolean = true) {
        val tables = toTables(src)
        out.printWriter().use { pw ->
            pw.println("# ${PlantUML.umlName}")
            val quote = if (sqlQuote) "`" else ""
            tables.forEach { table ->
                appendMysqlTable(table, pw, quote)
            }

        }

    }

    fun toMysqlDDLUpdate(old: File, src: File, out: PrintWriter, sqlQuote: Boolean = true) {
        val tables = toTables(src)
        val oldTables = toTables(old)
        if (tables != oldTables) {
            val quote = if (sqlQuote) "`" else ""
            val tableNames = tables.map { it.tableName }
            val oldTableNames = oldTables.map { it.tableName }
            (oldTableNames - tableNames).forEach {
                out.println("# $it")
                out.println("DROP TABLE IF EXISTS $quote$it$quote;")
                out.println()
            }
            val newTableNames = tableNames - oldTableNames
            tables.forEach { table ->
                val tableName = table.tableName
                if (newTableNames.contains(tableName)) {
                    appendMysqlTable(table, out, quote)
                } else {
                    val oldTable = oldTables.find { it.tableName == tableName }!!
                    if (oldTable != table)
                        out.println("# $tableName")
                    val oldColumns = oldTable.columns
                    val columns = table.columns
                    val oldPrimaryKeys = oldTable.primaryKeys
                    val primaryKeys = table.primaryKeys
                    val oldPrimaryKey = oldPrimaryKeys[0]
                    val primaryKey = primaryKeys[0]
                    if (oldPrimaryKeys.size == 1 && primaryKeys.size == 1 && oldPrimaryKey != primaryKey) {
                        out.println("ALTER TABLE $quote$tableName$quote CHANGE $quote${oldPrimaryKey.columnName}$quote ${columnDef(primaryKey, quote)} COMMENT '${primaryKey.remarks}';")
                        oldColumns.remove(oldPrimaryKey)
                        columns.remove(primaryKey)
                    }
                    val oldColumnNames = oldColumns.map { it.columnName }
                    val columnNames = columns.map { it.columnName }
                    (oldColumnNames - columnNames).forEach {
                        out.println("ALTER TABLE $quote$tableName$quote DROP COLUMN $it;")
                    }

                    val newColumnNames = columnNames - oldColumnNames
                    columns.forEach { column ->
                        if (newColumnNames.contains(column.columnName)) {
                            out.println("ALTER TABLE $quote$tableName$quote ADD COLUMN ${columnDef(column, quote)} COMMENT '${column.remarks}';")
                        } else if (column != oldColumns.find { it.columnName == column.columnName }) {
                            out.println("ALTER TABLE $quote$tableName$quote MODIFY ${columnDef(column, quote)} COMMENT '${column.remarks}';")
                        }
                    }
                    val delIndexes = oldTable.indexes - table.indexes
                    if (delIndexes.isNotEmpty()) {
                        delIndexes.forEach {
                            out.println("DROP INDEX ${it.name} ON $quote$tableName$quote;")
                        }
                    }
                    val newIndexes = table.indexes - oldTable.indexes
                    if (newIndexes.isNotEmpty()) {
                        newIndexes.forEach { indexed ->
                            if (indexed.unique) {
                                out.println("CREATE UNIQUE INDEX ${indexed.name} ON $quote$tableName$quote (${indexed.columnName.joinToString(",") { "$quote$it$quote" }});")
                            } else {
                                out.println("CREATE INDEX ${indexed.name} ON $quote$tableName$quote (${indexed.columnName.joinToString(",") { "$quote$it$quote" }});")
                            }
                        }
                    }

                    if (oldTable != table)
                        out.println()
                }
            }

        }
    }

    private fun appendMysqlTable(table: Table, pw: PrintWriter, quote: String) {
        val tableName = table.tableName
        pw.println("# $tableName")
        pw.println("DROP TABLE IF EXISTS $quote$tableName$quote;")
        pw.println("CREATE TABLE $quote$tableName$quote (")
        val hasPrimary = table.primaryKeyNames.isNotEmpty()
        val lastIndex = table.columns.size - 1
        table.columns.forEachIndexed { index, column ->
            pw.println("  ${columnDef(column, quote)} COMMENT '${column.remarks}'${if (index < lastIndex || hasPrimary) "," else ""}")
        }
        if (hasPrimary)
            appendPrimaryKeys(table, pw, quote)
        pw.println(") COMMENT = '${table.remarks}';")
        table.indexes.forEach { t ->
            if (t.unique) {
                pw.println("CREATE UNIQUE INDEX ${t.name} ON $quote$tableName$quote (${t.columnName.joinToString(",") { "$quote$it$quote" }});")
            } else {
                pw.println("CREATE INDEX ${t.name} ON $quote$tableName$quote (${t.columnName.joinToString(",") { "$quote$it$quote" }});")
            }
        }
        table.columns.filter { it.isForeignKey }.forEach {
            pw.println("CONSTRAINT FK_${tableName.replace("_", "").takeLast(7)}_${it.columnName.replace(tableName, "").replace("_", "").replace(",", "").takeLast(7)} FOREIGN KEY ($quote${it.columnName}$quote) REFERENCES $quote${it.pktableName}$quote ($quote${it.pkcolumnName}$quote);")
        }
        pw.println()
    }

    fun toOracleDDL(src: File, out: File, sqlQuote: Boolean = true) {
        val tables = toTables(src)
        out.printWriter().use { pw ->
            pw.println("-- ${PlantUML.umlName}")
            val quote = if (sqlQuote) "\"" else ""
            tables.forEach { table ->
                appendOracleTable(table, pw, quote)
            }

        }
    }

    fun toOracleDDLUpdate(old: File, src: File, out: PrintWriter, sqlQuote: Boolean = true) {
        val tables = toTables(src)
        val oldTables = toTables(old)
        if (tables != oldTables) {
            val quote = if (sqlQuote) "\"" else ""
            val tableNames = tables.map { it.tableName }
            val oldTableNames = oldTables.map { it.tableName }
            (oldTableNames - tableNames).forEach {
                out.println("-- $it")
                out.println("DROP SEQUENCE $quote${it}_S$quote;")
                out.println("DROP TABLE $quote$it$quote;")
                out.println()
            }
            val newTableNames = tableNames - oldTableNames
            tables.forEach { table ->
                val tableName = table.tableName
                if (newTableNames.contains(tableName)) {
                    appendOracleTable(table, out, quote)
                } else {
                    val oldTable = oldTables.find { it.tableName == tableName }!!
                    if (oldTable != table)
                        out.println("-- $tableName")
                    val oldColumns = oldTable.columns
                    val columns = table.columns
                    val oldPrimaryKeys = oldTable.primaryKeys
                    val primaryKeys = table.primaryKeys
                    val oldPrimaryKey = oldPrimaryKeys[0]
                    val primaryKey = primaryKeys[0]
                    if (oldPrimaryKeys.size == 1 && primaryKeys.size == 1 && oldPrimaryKey != primaryKey) {
                        out.println("ALTER TABLE $quote$tableName$quote RENAME COLUMN $quote${oldPrimaryKey.columnName}$quote TO $quote${primaryKey.columnName}$quote;")
                        out.println("ALTER TABLE $quote$tableName$quote MODIFY (${columnDef(primaryKey, quote)});")
                        out.println("COMMENT ON COLUMN $quote$tableName$quote.$quote${primaryKey.columnName}$quote IS '${primaryKey.remarks}';")
                        oldColumns.remove(oldPrimaryKey)
                        columns.remove(primaryKey)
                    }
                    val oldColumnNames = oldColumns.map { it.columnName }
                    val columnNames = columns.map { it.columnName }
                    val dropColumnNames = oldColumnNames - columnNames
                    if (dropColumnNames.isNotEmpty()) {
                        out.println("ALTER TABLE $quote$tableName$quote DROP (${dropColumnNames.joinToString(",") { "$quote$it$quote" }});")
                    }
                    val newColumnNames = columnNames - oldColumnNames
                    columns.forEach { column ->
                        if (newColumnNames.contains(column.columnName)) {
                            out.println("ALTER TABLE $quote$tableName$quote ADD (${columnDef(column, quote)});")
                            out.println("COMMENT ON COLUMN $quote$tableName$quote.$quote${column.columnName}$quote IS '${column.remarks}';")
                        } else {
                            val oldColumn = oldColumns.find { it.columnName == column.columnName }
                            if (column != oldColumn) {
                                out.println("ALTER TABLE $quote$tableName$quote MODIFY (${columnDef(column, quote)});")
                                out.println("COMMENT ON COLUMN $quote$tableName$quote.$quote${column.columnName}$quote IS '${column.remarks}';")
                            }
                        }
                    }
                    val delIndexes = oldTable.indexes - table.indexes
                    if (delIndexes.isNotEmpty()) {
                        delIndexes.forEach {
                            out.println("DROP INDEX ${it.name} ON $quote$tableName$quote;")
                        }
                    }
                    val newIndexes = table.indexes - oldTable.indexes
                    if (newIndexes.isNotEmpty()) {
                        newIndexes.forEach { indexed ->
                            if (indexed.unique) {
                                out.println("CREATE UNIQUE INDEX ${indexed.name} ON $quote$tableName$quote (${indexed.columnName.joinToString(",") { "$quote$it$quote" }});")
                            } else {
                                out.println("CREATE INDEX ${indexed.name} ON $quote$tableName$quote (${indexed.columnName.joinToString(",") { "$quote$it$quote" }});")
                            }
                        }
                    }
                    if (oldTable != table)
                        out.println()
                }
            }
        }
    }

    private fun appendOracleTable(table: Table, pw: PrintWriter, quote: String) {
        val tableName = table.tableName
        pw.println("-- $tableName")
        if (table.sequenceStartWith != null) {
            pw.println("DROP SEQUENCE $quote${tableName}_S$quote;")
            pw.println("CREATE SEQUENCE $quote${tableName}_S$quote INCREMENT BY 1 START WITH 1${fill(table.sequenceStartWith!!)};")
        }
        pw.println()
        pw.println("DROP TABLE $quote$tableName$quote;")
        pw.println("CREATE TABLE $quote$tableName$quote (")
        val hasPrimary = table.primaryKeyNames.isNotEmpty()
        val lastIndex = table.columns.size - 1
        table.columns.forEachIndexed { index, column ->
            pw.println("  ${columnDef(column, quote)}${if (index < lastIndex || hasPrimary) "," else ""}")
        }
        if (hasPrimary)
            appendPrimaryKeys(table, pw, quote)
        pw.println(");")
        table.indexes.forEach { t ->
            if (t.unique) {
                pw.println("CREATE UNIQUE INDEX ${t.name} ON $quote$tableName$quote (${t.columnName.joinToString(",") { "$quote$it$quote" }});")
            } else {
                pw.println("CREATE INDEX ${t.name} ON $quote$tableName$quote (${t.columnName.joinToString(",") { "$quote$it$quote" }});")
            }
        }
        table.columns.filter { it.isForeignKey }.forEach {
            pw.println("CONSTRAINT FK_${tableName.replace("_", "").takeLast(7)}_${it.columnName.replace(tableName, "").replace("_", "").replace(",", "").takeLast(7)} FOREIGN KEY (${it.columnName}) REFERENCES ${it.pktableName} (${it.pkcolumnName});")
        }
        pw.println("COMMENT ON TABLE $quote$tableName$quote IS '${table.remarks}';")
        table.columns.forEach {
            pw.println("COMMENT ON COLUMN $quote$tableName$quote.$quote${it.columnName}$quote IS '${it.remarks}';")
        }
        pw.println()
    }

    private fun fill(length: Int): String {
        var str = ""
        for (i in 1..length) {
            str += "0"
        }
        return str
    }

    private fun columnDef(it: Column, quote: String): String {
        val isString = it.typeName.startsWith("VARCHAR")
        return "$quote${it.columnName}$quote ${it.typeName}${if (it.columnSize > 0) "(${it.columnSize}${if (it.decimalDigits > 0) ",${it.decimalDigits}" else ""})" else ""}${if (it.columnDef.isNullOrBlank()) "" else " DEFAULT ${if (isString) "'" else ""}${it.columnDef}${if (isString) "'" else ""}"}${if (it.extra.isNotBlank()) " ${it.extra}" else ""}${if (it.autoIncrement) " AUTO_INCREMENT" else ""}${if (it.nullable) " NULL" else " NOT NULL"}"
    }

    private fun appendPrimaryKeys(table: Table, pw: PrintWriter, quote: String) {
        pw.println("  PRIMARY KEY (${table.primaryKeyNames.joinToString(",") { "$quote$it$quote" }})")
    }
}
