package cn.bestwu.generator.dom.java.element

import cn.bestwu.generator.dom.java.JavaDomUtils
import cn.bestwu.generator.dom.java.JavaType

/**
 * This class encapsulates the idea of an inner enum - it has methods that make
 * it easy to generate inner enum.
 *
 */
open class InnerEnum(type: JavaType) : InnerUnit(type) {

    /** The inner classes.  */
    private val innerClasses: MutableList<InnerClass> = mutableListOf()

    /** The inner enums.  */
    private val innerEnums: MutableList<InnerEnum> = mutableListOf()


    /** The enum constants.  */
    private val enumConstants: MutableList<String> = mutableListOf()


    operator fun String.unaryPlus() {
        enumConstants.add(this)
    }

    /**
     * Gets the formatted content.
     *
     * @param indentLevel
     * the indent level
     * @param compilationUnit the compilation unit
     * @return the formatted content
     */
    override fun getFormattedContent(indentLevel: Int, compilationUnit: CompilationUnit): String {
        var indentLevelInner = indentLevel
        val sb = StringBuilder()

        addFormattedJavadoc(sb, indentLevelInner)
        addFormattedAnnotations(sb, indentLevelInner)

        indent(sb, indentLevelInner)
        if (visibility == JavaVisibility.PUBLIC) {
            sb.append(visibility.value)
        }

        sb.append("enum ")
        sb.append(type.shortName)

        if (superInterfaceTypes.size > 0) {
            sb.append(" implements ")

            var comma = false
            for (fqjt in superInterfaceTypes) {
                if (comma) {
                    sb.append(", ")
                } else {
                    comma = true
                }

                sb.append(JavaDomUtils.calculateTypeName(compilationUnit, fqjt))
            }
        }

        sb.append(" {")
        newLine(sb)
        indentLevelInner++

        val strIter = enumConstants.iterator()
        while (strIter.hasNext()) {
            newLine(sb)
            indent(sb, indentLevelInner)
            val enumConstant = strIter.next()
            sb.append(enumConstant)

            if (strIter.hasNext()) {
                sb.append(',')
            } else {
                sb.append(';')
            }
        }

        if (fields.size > 0) {
            newLine(sb)
        }

        val fldIter = fields.iterator()
        while (fldIter.hasNext()) {
            newLine(sb)
            val field = fldIter.next()
            sb.append(field.getFormattedContent(indentLevelInner, compilationUnit))
            if (fldIter.hasNext()) {
                newLine(sb)
            }
        }

        if (methods.size > 0) {
            newLine(sb)
        }

        val mtdIter = methods.iterator()
        while (mtdIter.hasNext()) {
            newLine(sb)
            val method = mtdIter.next()
            method.interfaceMethod = false
            sb.append(method.getFormattedContent(indentLevelInner, compilationUnit))
            if (mtdIter.hasNext()) {
                newLine(sb)
            }
        }

        if (innerClasses.size > 0) {
            newLine(sb)
        }

        val icIter = innerClasses.iterator()
        while (icIter.hasNext()) {
            newLine(sb)
            val innerClass = icIter.next()
            sb.append(innerClass.getFormattedContent(indentLevelInner, compilationUnit))
            if (icIter.hasNext()) {
                newLine(sb)
            }
        }

        if (innerEnums.size > 0) {
            newLine(sb)
        }

        val ieIter = innerEnums.iterator()
        while (ieIter.hasNext()) {
            newLine(sb)
            val innerEnum = ieIter.next()
            sb.append(innerEnum.getFormattedContent(indentLevelInner, compilationUnit))
            if (ieIter.hasNext()) {
                newLine(sb)
            }
        }

        indentLevelInner--
        newLine(sb)
        indent(sb, indentLevelInner)
        sb.append('}')

        return sb.toString()
    }


    /**
     * Adds the inner class.
     *
     * @param innerClass
     * the inner class
     */
    fun innerClass(vararg innerClass: InnerClass) {
        innerClasses.addAll(innerClass)
    }


    /**
     * Adds the inner enum.
     *
     * @param innerEnum
     * the inner enum
     */
    fun innerEnum(vararg innerEnum: InnerEnum) {
        innerEnums.addAll(innerEnum)
    }

    /**
     * Adds the enum constant.
     *
     * @param enumConstant
     * the enum constant
     */
    fun enumConstant(vararg enumConstant: String) {
        enumConstants.addAll(enumConstant)
    }

}
