package cn.bestwu.generator

import cn.bestwu.generator.database.DatabaseMetaData
import cn.bestwu.generator.dom.java.element.JavaElement
import cn.bestwu.generator.dsl.Generator
import cn.bestwu.generator.dsl.Generators
import java.io.File
import java.sql.Connection
import java.sql.DriverManager
import java.util.*


/**
 * @author Peter Wu
 */
open class GeneratorExtension(
        /**
         * JDBC连接配置
         */
        val datasource: JDBCConnectionConfiguration = JDBCConnectionConfiguration(),
        /**
         * 生成文件基础路径,项目路径
         */
        var basePath: File? = null,
        /**
         * 数据源类型，默认数据库
         */
        var dataType: String = Generators.database,
        /**
         * PlantUML 脚本目录
         */
        var pumlSrc: String = "puml/src",
        var pumlDatabaseDriver: DatabaseDriver = datasource.databaseDriver,
        var pumlDatabase: String = "puml/database",
        /**
         * PlantUML 脚本输出目录
         */
        var pumlOutput: String = "build/gen/puml",
        var pumlOutputFileFormat: String = "PNG",
        /**
         * SQL 脚本目录
         */
        var sqlOutput: String = "database/ddl",
        var sqlQuote: Boolean = true,
        /**
         * 基础路径下相对路径
         */
        var dir: String = "",
        /**
         * 覆盖所有已生成文件
         */
        var replaceAll: Boolean = false,
        /**
         * 生成代码包名
         */
        var packageName: String = "",
        /**
         * 基础代码包名
         */
        var basePackageName: String = "",
        /**
         * 子模块
         */
        var module: String = "",
        /**
         * 子模块名称
         */
        var moduleName: String = "",

        var projectName: String = "",
        /**
         * 表前缀
         */
        var tablePrefix: String = "",
        /**
         * 注释说明
         */
        var remarks: String = "",
        /**
         * 手动主键名
         */
        var primaryKeyName: String = "id",
        /**
         * 缩进
         */
        var indent: String = JavaElement.defaultIndent,
        /**
         * 模板
         */
        var generators: Array<Generator> = arrayOf(),
        /**
         * 相关数据表
         */
        var tableNames: Array<String> = arrayOf(),
        var pumlTableNames: Array<String> = arrayOf(),
        /**
         * 额外设置
         */
        var settings: MutableMap<String, Any?> = mutableMapOf()
) {

    companion object {
        /**
         * javaName
         */
        var javaName: (String) -> String = {
            javaName(it, false)
        }

        private fun javaName(str: String, capitalize: Boolean = false): String {
            val s = str.split(Regex("[^\\p{Alnum}]")).joinToString("") {
                it.toLowerCase().capitalize()
            }
            return if (capitalize) s else s.decapitalize()
        }

    }

    /**
     * ClassName
     */
    var className: (String) -> String = { str ->
        javaName(str.substringAfter(tablePrefix), true)
    }

    fun <T> use(metaData: DatabaseMetaData.() -> T): T {
        Class.forName(datasource.driverClass).newInstance()
        val connection = DriverManager.getConnection(datasource.url, datasource.properties)
        try {
            return metaData(DatabaseMetaData(connection.metaData, datasource.catalog, datasource.schema))
        } finally {
            connection.close()
        }
    }

    fun <T> run(connectionFun: Connection.() -> T): T {
        Class.forName(datasource.driverClass).newInstance()
        val connection = DriverManager.getConnection(datasource.url, datasource.properties)
        try {
            return connectionFun(connection)
        } finally {
            connection.close()
        }
    }

    fun file(subfile: String): File {
        return if (subfile.startsWith("/")) {
            File(subfile)
        } else {
            if (File(pumlSrc).exists()) File(subfile) else File(File("../"), subfile)
        }
    }

    fun pumlSqlOutputFile(src: File): File {
        val dest = File(File(file(sqlOutput), src.parentFile.absolutePath.replace(file(pumlSrc).absolutePath, "")), src.nameWithoutExtension + ".sql")
        dest.parentFile.mkdirs()
        return dest
    }

    fun pumlSqlUpdateOutputFile(): File {
        val dest = File(file(sqlOutput), "update.sql")
        dest.parentFile.mkdirs()
        return dest
    }

    fun pumlOutputFile(src: File): File {
        val dest = File(file(pumlOutput), src.absolutePath.replace(file(pumlSrc).absolutePath, "").replace(file(pumlDatabase).absolutePath, ""))
        dest.parentFile.mkdirs()
        return dest
    }

    val pumlSources: Sequence<File>
        get() {
            return (file(pumlDatabase).walkTopDown() + file(pumlSrc).walkTopDown()).filter { it.isFile && it.extension == "puml" }
        }
    val pumlSrcSources: Sequence<File>
        get() {
            return file(pumlSrc).walkTopDown().filter { it.isFile && it.extension == "puml" }
        }
}

class JDBCConnectionConfiguration(
        var url: String = "",
        var catalog: String? = null,
        val properties: Properties = Properties().apply {
            set("remarksReporting", "true") //oracle 读取表注释
            set("useInformationSchema", "true")//mysql 读取表注释
            set("characterEncoding", "utf8")
            set("user", "root")
            set("password", "root")
        }
) {
    var schema: String? = null
        get() {
            return if (field.isNullOrBlank() && url.startsWith("jdbc:oracle")) {
                username.toUpperCase()
            } else {
                field
            }
        }

    val databaseDriver
        get() = DatabaseDriver.fromJdbcUrl(url)

    var driverClass: String = ""
        get() {
            return if (field.isBlank() && !url.isBlank()) {
                databaseDriver.driverClassName ?: ""
            } else {
                field
            }

        }

    var username: String
        set(value) = properties.set("user", value)
        get() = properties.getProperty("user")
    var password: String by properties
}