package cn.bestwu.generator.puml

import cn.bestwu.generator.database.domain.Column
import cn.bestwu.generator.database.domain.Indexed
import cn.bestwu.generator.database.domain.Table
import cn.bestwu.generator.dom.java.JavaTypeResolver
import cn.bestwu.generator.dsl.def.PlantUML
import java.io.File
import java.io.PrintWriter

/**
 * @author Peter Wu
 * @since 0.0.45
 */
object PumlConverter {

    fun toTables(puml: File, call: (Table) -> Unit = {}): List<Table> {
        return toTableOrAnys(puml, call).filter { it is Table }.map { it as Table }
    }

    fun toTableOrAnys(puml: File, call: (Table) -> Unit = {}): List<Any> {
        val tables = mutableListOf<Any>()
        var table: Table? = null
        var isField = false
        var isUml = false
        puml.readLines().forEach { it ->
            if (it.isNotBlank()) {
                val line = it.trim()
                if (line.startsWith("@startuml")) {
                    PlantUML.umlName = line.substringAfter("@startuml").trim()
                    isUml = true
                } else if (line.startsWith("class")) {
                    val fieldDef = line.split(" ")
                    val tableName = fieldDef[1].trim()
                    table = Table("puml", tableName = tableName)
                    if (fieldDef.size > 3) {
                        table!!.desc = fieldDef[2]
                        table!!.withSequence = fieldDef[2].matches(Regex("<<\\(T,#DDDDD[\\dD]\\)>>"))
                    }
                } else if (table != null && !isField) {
                    if ("==" == line)
                        isField = true
                    else
                        table!!.comment = line
                } else if (isField) {
                    val uniqueMult = line.startsWith("'UNIQUE")
                    if (uniqueMult || line.startsWith("'INDEX")) {
                        val columnNames = line.substringAfter(if (uniqueMult) "'UNIQUE" else "'INDEX").trim()
                        val tableName = table!!.tableName
                        table!!.indexes.add(Indexed("${if (uniqueMult) "UK" else "IDX"}_${tableName.replace("_", "").takeLast(7)}_${columnNames.replace(tableName, "").replace("_", "").replace(",", "").takeLast(7)}", uniqueMult, columnNames.split(",").toMutableList()))
                    } else if ("}" == line) {
                        call(table!!)
                        tables.add(table!!)
                        table = null
                        isField = false
                    } else if (!line.startsWith("'")) {
                        val lineDef = line.trim().split("--")
                        val fieldDef = lineDef[0].trim()
                        val fieldDefs = fieldDef.split(" ")
                        val columnName = fieldDefs[0]
                        val columnDef = fieldDef.substringAfter(columnName).replace(":", "").trim()
                        val type = columnDef.split(" ")[0].trim()
                        var length = 0
                        var scale = 0
                        if (type.contains("(")) {
                            val lengthScale = type.substringAfter("(").substringBefore(")")
                            if (lengthScale.contains(",")) {
                                val ls = lengthScale.split(",")
                                length = ls[0].toInt()
                                scale = ls[1].toInt()
                            } else {
                                length = lengthScale.toInt()
                            }
                        }
                        var defaultVal: String? = null
                        if (columnDef.contains("DEFAULT")) {
                            defaultVal = columnDef.substringAfter("DEFAULT").trim().substringBefore(" ").trim('\'').trim()
                        }
                        var fk = false
                        var refTable = ""
                        var refColumn = ""
                        if (columnDef.contains("FK")) {//FK > docs.id
                            val ref = columnDef.substringAfter("FK >").trim().substringBefore(" ").trim()
                            val refs = ref.split(".")
                            fk = true
                            refTable = refs[0]
                            refColumn = refs[1]
                        }
                        val typeName = type.substringBefore("(")
                        val unique = columnDef.contains("UNIQUE")
                        val indexed = columnDef.contains("INDEX")
                        val column = Column(tableCat = null, columnName = columnName, comment = lineDef.last().trim(), typeName = typeName, length = length, scale = scale, nullable = !columnDef.contains("NOT NULL"), unique = unique, indexed = indexed, defaultVal = defaultVal, dataType = JavaTypeResolver.calculateDataType(JavaTypeResolver.calculateJdbcTypeName(typeName))
                                ?: 0, tableSchem = null, foreignKey = fk, refTable = refTable, refColumn = refColumn)
                        if (unique)
                            table!!.indexes.add(Indexed("UK_${table!!.tableName.replace("_", "").takeLast(7)}_${columnName.replace(table!!.tableName, "").replace("_", "").replace(",", "").takeLast(7)}", true, mutableListOf(columnName)))
                        if (indexed)
                            table!!.indexes.add(Indexed("IDX_${table!!.tableName.replace("_", "").takeLast(7)}_${columnName.replace(table!!.tableName, "").replace("_", "").replace(",", "").takeLast(7)}", false, mutableListOf(columnName)))
                        if (columnDef.contains("PK")) {
                            table!!.primaryKeys.add(column)
                        }
                        table!!.pumlColumns.add(column)
                    } else {
                        table!!.pumlColumns.add(line)
                    }
                } else if (line.startsWith("@enduml")) {
                    isUml = false
                } else if (isUml && line.isNotBlank()) {
                    tables.add(line)
                }
            }
        }

        return tables
    }

    private fun compile(tables: List<Any>, out: File) {
        val plantUML = PlantUML(out.absolutePath)
        plantUML.setUp()
        tables.forEach {
            if (it is Table) {
                plantUML.call(it)
            } else {
                plantUML.appendlnText(it.toString())
            }
        }
        plantUML.tearDown()
    }

    fun compile(src: File, out: File) {
        compile(toTableOrAnys(src), out)
    }

    fun toMysql(src: File, out: File) {
        val tables = toTableOrAnys(src)
        tables.forEach {
            if (it is Table) {
                it.pumlColumns.forEach {
                    if (it is Column) {
                        when (it.typeName) {
                            "VARCHAR2" -> it.typeName = "VARCHAR"
                            "RAW" -> it.typeName = "BINARY"
                            "CLOB" -> it.typeName = "TEXT"
                            "NUMBER" -> {
                                if (it.scale == 0) {
                                    when (it.length) {
                                        in 1..3 -> {
                                            it.typeName = "TINYINT"
                                            it.length = 0
                                        }
                                        in 4..5 -> {
                                            it.typeName = "SMALLINT"
                                            it.length = 0
                                        }
                                        in 6..7 -> {
                                            it.typeName = "MEDUIMINT"
                                            it.length = 0
                                        }
                                        in 8..10 -> {
                                            it.typeName = "INT"
                                            it.length = 0
                                        }
                                        in 11..20 -> {
                                            it.typeName = "BIGINT"
                                            it.length = 0
                                        }
                                        else -> it.typeName = "DECIMAL"
                                    }
                                } else {
                                    it.typeName = "DECIMAL"
                                }
                            }
                        }
                    }
                }
            }
        }
        compile(tables, out)
    }

    fun toOracle(src: File, out: File) {
        val tables = toTableOrAnys(src)
        tables.forEach {
            if (it is Table) {
                it.pumlColumns.forEach {
                    if (it is Column) {
                        when (it.typeName) {
                            "VARCHAR" -> it.typeName = "VARCHAR2"
                            "TINYINT" -> {
                                it.typeName = "NUMBER"
                                it.length = 3
                            }
                            "SMALLINT" -> {
                                it.typeName = "NUMBER"
                                it.length = 5
                            }
                            "MEDUIMINT" -> {
                                it.typeName = "NUMBER"
                                it.length = 7
                            }
                            "INT" -> {
                                it.typeName = "NUMBER"
                                it.length = 10
                            }
                            "BIGINT" -> {
                                it.typeName = "NUMBER"
                                it.length = 20
                            }
                            "FLOAT", "DOUBLE", "DECIMAL" -> {
                                it.typeName = "NUMBER"
                            }
                            "TINYTEXT" -> it.typeName = "CLOB"
                            "TINYBLOB" -> it.typeName = "BLOB"
                            "BINARY" -> it.typeName = "RAW"
                            "TEXT" -> it.typeName = "CLOB"
                            "LONGTEXT" -> it.typeName = "CLOB"
                        }
                    }
                }
            }
        }
        compile(tables, out)
    }

    fun toMySqlDDL(src: File, out: File, sqlQuote: Boolean = true) {
        val tables = toTables(src)
        out.printWriter().use { pw ->
            pw.println("# ${PlantUML.umlName}")
            val quote = if (sqlQuote) "`" else ""
            tables.forEach { table ->
                appendMysqlTable(table, out, pw, quote)
            }

        }

    }

    fun toMysqlDDLUpdate(old: File, src: File, out: File, sqlQuote: Boolean = true) {
        val tables = toTables(src)
        val oldTables = toTables(old)
        if (tables != oldTables)
            out.printWriter().use { pw ->
                val quote = if (sqlQuote) "`" else ""
                val tableNames = tables.map { it.tableName }
                val oldTableNames = oldTables.map { it.tableName }
                (oldTableNames - tableNames).forEach {
                    pw.println("# $it")
                    pw.println("DROP TABLE IF EXISTS $quote$it$quote;")
                    pw.println()
                }
                val newTableNames = tableNames - oldTableNames
                tables.forEach { table ->
                    if (newTableNames.contains(table.tableName)) {
                        appendMysqlTable(table, out, pw, quote)
                    } else {
                        val oldTable = oldTables.find { it.tableName == table.tableName }!!
                        if (oldTable != table)
                            pw.println("# ${table.tableName}")
                        val oldColumns = oldTable.columns
                        val oldColumnNames = oldColumns.map { it.columnName }
                        val columns = table.columns
                        val columnNames = columns.map { it.columnName }
                        (oldColumnNames - columnNames).forEach {
                            pw.println("ALTER TABLE $quote${table.tableName}$quote DROP COLUMN $it;")
                        }

                        val newColumnNames = columnNames - oldColumnNames
                        columns.forEach { column ->
                            if (newColumnNames.contains(column.columnName)) {
                                pw.println("ALTER TABLE $quote${table.tableName}$quote ADD COLUMN ${columnDef(table, column, quote)} COMMENT '${column.comment}';")
                            } else if (column != oldColumns.find { it.columnName == column.columnName }) {
                                pw.println("ALTER TABLE $quote${table.tableName}$quote MODIFY ${columnDef(table, column, quote)} COMMENT '${column.comment}';")
                            }
                        }
                        if (oldTable != table)
                            pw.println()
                    }
                }

            }
    }

    private fun appendMysqlTable(table: Table, out: File, pw: PrintWriter, quote: String) {
        val tableName = table.tableName
        pw.println("# $tableName")
        pw.println("DROP TABLE IF EXISTS $quote$tableName$quote;")
        pw.println("CREATE TABLE $quote$tableName$quote (")
        table.columns.forEach {
            pw.println("  ${columnDef(table, it, quote)} COMMENT '${it.comment}',")
        }
        val uniques = table.indexes.filter { it.unique }
        if (uniques.isNotEmpty()) {
            pw.println("  UNIQUE INDEX (${uniques.flatMap { it.columnName }.joinToString(",") { "$quote$it$quote" }}),")
        }
        val indexes = table.indexes.filter { !it.unique }
        if (indexes.isNotEmpty())
            pw.println("  INDEX (${indexes.flatMap { it.columnName }.joinToString(",") { "$quote$it$quote" }}),")
        appendPrimaryKeys(table, out, pw, quote)
        pw.println(") COMMENT = '${table.comment}';")
        table.columns.filter { it.foreignKey }.forEach {
            pw.println("CONSTRAINT FK_${tableName.replace("_", "").takeLast(7)}_${it.columnName.replace(tableName, "").replace("_", "").replace(",", "").takeLast(7)} FOREIGN KEY ($quote${it.columnName}$quote) REFERENCES $quote${it.refTable}$quote ($quote${it.refColumn}$quote);")
        }
        pw.println()
    }

    fun toOracleDDL(src: File, out: File, sqlQuote: Boolean = true) {
        val tables = toTables(src)
        out.printWriter().use { pw ->
            pw.println("-- ${PlantUML.umlName}")
            val quote = if (sqlQuote) "\"" else ""
            tables.forEach { table ->
                appendOracleTable(table, out, pw, quote)
            }

        }
    }

    fun toOracleDDLUpdate(old: File, src: File, out: File, sqlQuote: Boolean = true) {
        val tables = toTables(src)
        val oldTables = toTables(old)
        if (tables != oldTables)
            out.printWriter().use { pw ->
                val quote = if (sqlQuote) "\"" else ""
                val tableNames = tables.map { it.tableName }
                val oldTableNames = oldTables.map { it.tableName }
                (oldTableNames - tableNames).forEach {
                    pw.println("-- $it")
                    pw.println("DROP SEQUENCE $quote${it}_S$quote;")
                    pw.println("DROP TABLE $quote$it$quote;")
                    pw.println()
                }
                val newTableNames = tableNames - oldTableNames
                tables.forEach { table ->
                    if (newTableNames.contains(table.tableName)) {
                        appendOracleTable(table, out, pw, quote)
                    } else {
                        val oldTable = oldTables.find { it.tableName == table.tableName }!!
                        if (oldTable != table)
                            pw.println("-- ${table.tableName}")
                        val oldColumns = oldTable.columns
                        val oldColumnNames = oldColumns.map { it.columnName }
                        val columns = table.columns
                        val columnNames = columns.map { it.columnName }
                        val dropColumnNames = oldColumnNames - columnNames
                        if (dropColumnNames.isNotEmpty()) {
                            pw.println("ALTER TABLE $quote${table.tableName}$quote DROP (${dropColumnNames.joinToString(",") { "$quote$it$quote" }});")
                        }
                        val newColumnNames = columnNames - oldColumnNames
                        columns.forEach { column ->
                            if (newColumnNames.contains(column.columnName)) {
                                pw.println("ALTER TABLE $quote${table.tableName}$quote ADD (${columnDef(table, column, quote)});")
                                pw.println("COMMENT ON COLUMN $quote${table.tableName}$quote.$quote${column.columnName}$quote IS '${column.comment}';")
                            } else {
                                val oldColumn = oldColumns.find { it.columnName == column.columnName }
                                if (column != oldColumn) {
                                    pw.println("ALTER TABLE $quote${table.tableName}$quote MODIFY (${if (oldColumn!!.nullable != column.nullable) columnDef(table, column, quote) else columnDefNull(column, quote)});")
                                    pw.println("COMMENT ON COLUMN $quote${table.tableName}$quote.$quote${column.columnName}$quote IS '${column.comment}';")
                                }
                            }
                        }
                        if (oldTable != table)
                            pw.println()
                    }
                }
            }
    }

    private fun appendOracleTable(table: Table, out: File, pw: PrintWriter, quote: String) {
        val tableName = table.tableName
        pw.println("-- $tableName")
        when {
            table.desc?.contains("(T,#DDDDDD)") == true -> {
                pw.println("DROP SEQUENCE $quote${tableName}_S$quote;")
                pw.println("CREATE SEQUENCE $quote${tableName}_S$quote INCREMENT BY 1 START WITH 1;")
            }
            table.desc?.matches(Regex("<<\\(T,#DDDDD\\d\\)>>")) == true -> {
                val startWith = table.desc!!.replace(Regex("<<\\(T,#DDDDD(\\d)\\)>>"), "$1").toInt()
                pw.println("DROP SEQUENCE $quote${tableName}_S$quote;")
                pw.println("CREATE SEQUENCE $quote${tableName}_S$quote INCREMENT BY 1 START WITH 1${fill(startWith)};")
            }
        }
        pw.println()
        pw.println("DROP TABLE $quote$tableName$quote;")
        pw.println("CREATE TABLE $quote$tableName$quote (")
        table.columns.forEach {
            pw.println("  ${columnDef(table, it, quote)},")
        }
        appendPrimaryKeys(table, out, pw, quote)
        pw.println(");")
        table.indexes.forEach {
            if (it.unique) {
                pw.println("CREATE UNIQUE INDEX ${it.name} ON $quote$tableName$quote (${it.columnName.joinToString(",") { "$quote$it$quote" }});")
            } else {
                pw.println("CREATE INDEX ${it.name} ON $quote$tableName$quote (${it.columnName.joinToString(",") { "$quote$it$quote" }});")
            }
        }
        table.columns.filter { it.foreignKey }.forEach {
            pw.println("CONSTRAINT FK_${tableName.replace("_", "").takeLast(7)}_${it.columnName.replace(tableName, "").replace("_", "").replace(",", "").takeLast(7)} FOREIGN KEY (${it.columnName}) REFERENCES ${it.refTable} (${it.refColumn});")
        }
        pw.println("COMMENT ON TABLE $quote$tableName$quote IS '${table.comment}';")
        table.columns.forEach {
            pw.println("COMMENT ON COLUMN $quote$tableName$quote.$quote${it.columnName}$quote IS '${it.comment}';")
        }
        pw.println()
    }

    private fun fill(length: Int): String {
        var str = ""
        for (i in 1..length) {
            str += "0"
        }
        return str
    }

    private fun columnDef(table: Table, it: Column, quote: String): String {
        val isString = it.typeName.startsWith("VARCHAR")
        return "$quote${it.columnName}$quote ${it.typeName}${if (it.length > 0) "(${it.length}${if (it.scale > 0) ",${it.scale}" else ""})" else ""}${if (it.defaultVal.isNullOrBlank()) "" else " DEFAULT ${if (isString) "'" else ""}${it.defaultVal}${if (isString) "'" else ""}"}${if (table.primaryKeyNames.contains(it.columnName)) "" else if (it.nullable) " NULL" else " NOT NULL"}"
    }

    private fun columnDefNull(it: Column, quote: String): String {
        val isString = it.typeName.startsWith("VARCHAR")
        return "$quote${it.columnName}$quote ${it.typeName}${if (it.length > 0) "(${it.length}${if (it.scale > 0) ",${it.scale}" else ""})" else ""}${if (it.defaultVal.isNullOrBlank()) "" else " DEFAULT ${if (isString) "'" else ""}${it.defaultVal}${if (isString) "'" else ""}"}"
    }

    private fun appendPrimaryKeys(table: Table, out: File, pw: PrintWriter, quote: String) {
        if (table.primaryKeyNames.isNotEmpty()) {
            pw.println("  PRIMARY KEY (${table.primaryKeyNames.joinToString(",") { "$quote$it$quote" }})")
        } else {
            pw.flush()
            out.writeText(out.readText().substringBeforeLast(","))
            pw.println()
        }
    }
}
