package cn.bestwu.generator.dom.java

import cn.bestwu.generator.database.domain.Column
import java.math.BigDecimal
import java.sql.Types
import java.util.*

/**
 * @author Peter Wu
 */
object JavaTypeResolver {

    var forceBigDecimals: Boolean = false
    var forceIntegers: Boolean = true

    private val typeMap: MutableMap<Int, JdbcTypeInformation> = HashMap()
    private val typeNameMap: MutableMap<String, Int> = HashMap()
    private val typeNameBundle: ResourceBundle = ResourceBundle.getBundle("jdbcTypeName")

    init {
        typeMap[Types.ARRAY] = JdbcTypeInformation("ARRAY",
                JavaType(java.lang.Object::class.java.name))
        typeMap[Types.BIGINT] = JdbcTypeInformation("BIGINT",
                JavaType(java.lang.Long::class.java.name))
        typeMap[Types.BINARY] = JdbcTypeInformation("BINARY",
                JavaType("byte[]"))
        typeMap[Types.BIT] = JdbcTypeInformation("BIT",
                JavaType(java.lang.Boolean::class.java.name))
        typeMap[Types.BLOB] = JdbcTypeInformation("BLOB",
                JavaType("byte[]"))
        typeMap[Types.BOOLEAN] = JdbcTypeInformation("BOOLEAN",
                JavaType(java.lang.Boolean::class.java.name))
        typeMap[Types.CHAR] = JdbcTypeInformation("CHAR",
                JavaType(java.lang.String::class.java.name))
        typeMap[Types.CLOB] = JdbcTypeInformation("CLOB",
                JavaType(java.lang.String::class.java.name))
        typeMap[Types.DATALINK] = JdbcTypeInformation("DATALINK",
                JavaType(java.lang.Object::class.java.name))
        typeMap[Types.DATE] = JdbcTypeInformation("DATE",
                JavaType(Date::class.java.name))
        typeMap[Types.DECIMAL] = JdbcTypeInformation("DECIMAL",
                JavaType(BigDecimal::class.java.name))
        typeMap[Types.DISTINCT] = JdbcTypeInformation("DISTINCT",
                JavaType(java.lang.Object::class.java.name))
        typeMap[Types.DOUBLE] = JdbcTypeInformation("DOUBLE",
                JavaType(java.lang.Double::class.java.name))
        typeMap[Types.FLOAT] = JdbcTypeInformation("FLOAT",
                JavaType(java.lang.Double::class.java.name))
        typeMap[Types.INTEGER] = JdbcTypeInformation("INTEGER",
                JavaType(java.lang.Integer::class.java.name))
        typeMap[Types.JAVA_OBJECT] = JdbcTypeInformation("JAVA_OBJECT",
                JavaType(java.lang.Object::class.java.name))
        typeMap[Types.LONGNVARCHAR] = JdbcTypeInformation("LONGNVARCHAR",
                JavaType(java.lang.String::class.java.name))
        typeMap[Types.LONGVARBINARY] = JdbcTypeInformation("LONGVARBINARY",
                JavaType("byte[]"))
        typeMap[Types.LONGVARCHAR] = JdbcTypeInformation("LONGVARCHAR",
                JavaType(java.lang.String::class.java.name))
        typeMap[Types.NCHAR] = JdbcTypeInformation("NCHAR",
                JavaType(java.lang.String::class.java.name))
        typeMap[Types.NCLOB] = JdbcTypeInformation("NCLOB",
                JavaType(java.lang.String::class.java.name))
        typeMap[Types.NVARCHAR] = JdbcTypeInformation("NVARCHAR",
                JavaType(java.lang.String::class.java.name))
        typeMap[Types.NULL] = JdbcTypeInformation("NULL",
                JavaType(java.lang.Object::class.java.name))
        typeMap[Types.NUMERIC] = JdbcTypeInformation("NUMERIC",
                JavaType(BigDecimal::class.java.name))
        typeMap[Types.OTHER] = JdbcTypeInformation("OTHER",
                JavaType(java.lang.Object::class.java.name))
        typeMap[Types.REAL] = JdbcTypeInformation("REAL",
                JavaType(java.lang.Float::class.java.name))
        typeMap[Types.REF] = JdbcTypeInformation("REF",
                JavaType(java.lang.Object::class.java.name))
        typeMap[Types.SMALLINT] = JdbcTypeInformation("SMALLINT",
                JavaType(java.lang.Short::class.java.name))
        typeMap[Types.STRUCT] = JdbcTypeInformation("STRUCT",
                JavaType(java.lang.Object::class.java.name))
        typeMap[Types.TIME] = JdbcTypeInformation("TIME",
                JavaType(Date::class.java.name))
        typeMap[Types.TIMESTAMP] = JdbcTypeInformation("TIMESTAMP",
                JavaType(Date::class.java.name))
        typeMap[Types.TINYINT] = JdbcTypeInformation("TINYINT",
                JavaType(java.lang.Byte::class.java.name))
        typeMap[Types.VARBINARY] = JdbcTypeInformation("VARBINARY",
                JavaType("byte[]"))
        typeMap[Types.VARCHAR] = JdbcTypeInformation("VARCHAR",
                JavaType(java.lang.String::class.java.name))

        typeMap.forEach { t, u ->
            typeNameMap[u.jdbcTypeName] = t
        }
    }

    fun calculateJdbcTypeName(typeName: String): String {
        return if (typeNameBundle.containsKey(typeName)) typeNameBundle.getString(typeName) else typeName
    }

    fun calculateDataType(jdbcTypeName: String): Int? {
        return typeNameMap[jdbcTypeName]
    }

    fun calculateJavaType(column: Column): JavaType? {
        var answer: JavaType? = null
        val jdbcTypeInformation = typeMap[column.dataType]

        if (jdbcTypeInformation != null) {
            answer = jdbcTypeInformation.javaType
            answer = overrideDefaultType(column, answer)
        }
        return if (forceIntegers && JavaType(java.lang.Short::class.java.name) == answer)
            JavaType(java.lang.Integer::class.java.name)
        else
            answer
    }

    fun calculateJdbcTypeName(column: Column): String? {
        var answer: String? = null
        val jdbcTypeInformation = typeMap[column.dataType]

        if (jdbcTypeInformation != null) {
            answer = jdbcTypeInformation.jdbcTypeName
        }

        return answer
    }

    private fun overrideDefaultType(column: Column, defaultType: JavaType?): JavaType? {
        var answer = defaultType

        when (column.dataType) {
            Types.BIT -> answer = calculateBitReplacement(column, defaultType)
            Types.DECIMAL, Types.NUMERIC -> answer = calculateBigDecimalReplacement(column, defaultType)
            else -> {
            }
        }

        return answer
    }

    private fun calculateBitReplacement(column: Column, defaultType: JavaType?): JavaType? {

        return if (column.length <= 1) {
            defaultType
        } else {
            JavaType("byte[]")
        }
    }

    private fun calculateBigDecimalReplacement(column: Column, defaultType: JavaType?): JavaType? {
        return if (column.scale > 0 || column.length > 20 || forceBigDecimals) {
            defaultType
        } else if (column.length > 11 || column.length == 0) {
            JavaType(java.lang.Long::class.java.name)
        } else if (column.length > 4) {
            JavaType(java.lang.Integer::class.java.name)
        } else {
            JavaType(java.lang.Short::class.java.name)
        }
    }

    class JdbcTypeInformation(val jdbcTypeName: String,
                              val javaType: JavaType)
}
