package cn.bestwu.generator.dsl

import cn.bestwu.generator.GeneratorException
import cn.bestwu.generator.GeneratorExtension
import cn.bestwu.generator.database.Column
import cn.bestwu.generator.database.Table
import cn.bestwu.generator.dom.java.JavaType
import cn.bestwu.generator.dom.java.JavaTypeResolver
import cn.bestwu.generator.dom.java.PrimitiveTypeWrapper
import org.atteo.evo.inflector.English
import java.util.*

/**
 * 模板基类
 */
abstract class Generator {
    companion object {
        fun toBoolean(obj: Any?): Boolean {
            when (obj) {
                is Boolean -> return obj
                is String -> {
                    if (obj.equals("true", true)) {
                        return true
                    }
                    when (obj.length) {
                        1 -> {
                            val ch0 = obj[0]
                            if (ch0 == 'y' || ch0 == 'Y' ||
                                    ch0 == 't' || ch0 == 'T' || ch0 == '1') {
                                return true
                            }
                            if (ch0 == 'n' || ch0 == 'N' ||
                                    ch0 == 'f' || ch0 == 'F' || ch0 == '0') {
                                return false
                            }
                        }
                        2 -> {
                            val ch0 = obj[0]
                            val ch1 = obj[1]
                            if ((ch0 == 'o' || ch0 == 'O') && (ch1 == 'n' || ch1 == 'N')) {
                                return true
                            }
                            if ((ch0 == 'n' || ch0 == 'N') && (ch1 == 'o' || ch1 == 'O')) {
                                return false
                            }
                        }
                        3 -> {
                            val ch0 = obj[0]
                            val ch1 = obj[1]
                            val ch2 = obj[2]
                            if ((ch0 == 'y' || ch0 == 'Y') &&
                                    (ch1 == 'e' || ch1 == 'E') &&
                                    (ch2 == 's' || ch2 == 'S')) {
                                return true
                            }
                            if ((ch0 == 'o' || ch0 == 'O') &&
                                    (ch1 == 'f' || ch1 == 'F') &&
                                    (ch2 == 'f' || ch2 == 'F')) {
                                return false
                            }
                        }
                        4 -> {
                            val ch0 = obj[0]
                            val ch1 = obj[1]
                            val ch2 = obj[2]
                            val ch3 = obj[3]
                            if ((ch0 == 't' || ch0 == 'T') &&
                                    (ch1 == 'r' || ch1 == 'R') &&
                                    (ch2 == 'u' || ch2 == 'U') &&
                                    (ch3 == 'e' || ch3 == 'E')) {
                                return true
                            }
                        }
                        5 -> {
                            val ch0 = obj[0]
                            val ch1 = obj[1]
                            val ch2 = obj[2]
                            val ch3 = obj[3]
                            val ch4 = obj[4]
                            if ((ch0 == 'f' || ch0 == 'F') &&
                                    (ch1 == 'a' || ch1 == 'A') &&
                                    (ch2 == 'l' || ch2 == 'L') &&
                                    (ch3 == 's' || ch3 == 'S') &&
                                    (ch4 == 'e' || ch4 == 'E')) {
                                return false
                            }
                        }
                        else -> {
                        }
                    }
                }
                is Number -> return obj.toInt() > 0
            }

            return false
        }

    }

    protected lateinit var table: Table
    protected lateinit var extension: GeneratorExtension
    protected val isOracleDatasource
        get() = extension.datasource.url.startsWith("jdbc:oracle")

    protected val basePackageName: String
        get() = extension.basePackageName

    protected val moduleName: String
        get() = extension.moduleName

    protected val moduleComment: String
        get() = extension.moduleComment

    protected val settings: Map<String, Any?>
        get() = extension.settings

    protected fun property(key: String): Any? = settings[key]

    protected fun <T> property(key: String, default: T): T {
        val any = settings[key]
        return if (any == null) {
            return default
        } else {
            @Suppress("UNCHECKED_CAST")
            any as T
        }
    }

    protected val className
        get() = extension.className(tableName)

    protected val entityName
        get() = className.decapitalize()

    protected val Column.javaType
        get() = JavaTypeResolver.calculateJavaType(this)!!

    protected val Column.jdbcType: String
        get() = JavaTypeResolver.calculateJdbcTypeName(this)!!

    protected val Column.randomValue: Any
        get() = when (javaType) {
            JavaType("java.sql.Timestamp::class") -> System.currentTimeMillis()
            JavaType.dateInstance, JavaType("java.sql.Date") -> System.currentTimeMillis()
            JavaType("java.sql.Time") -> System.currentTimeMillis()
            PrimitiveTypeWrapper.booleanInstance -> Random().nextBoolean()
            PrimitiveTypeWrapper.doubleInstance -> Random().nextDouble()
            PrimitiveTypeWrapper.longInstance -> Random().nextInt()
            PrimitiveTypeWrapper.integerInstance -> Random().nextInt()
            JavaType.stringInstance -> "test${Random().nextInt()}"
            else -> 1
        }

    protected val Column.randomValueToSet: String
        get() = when (javaType) {
            JavaType("java.sql.Timestamp::class") -> "new Timestamp(System.currentTimeMillis())"
            JavaType.dateInstance, JavaType("java.sql.Date") -> "new Date(System.currentTimeMillis())"
            JavaType("java.sql.Time") -> "new Time(System.currentTimeMillis())"
            PrimitiveTypeWrapper.booleanInstance -> "new Random().nextBoolean()"
            PrimitiveTypeWrapper.doubleInstance -> "new Random().nextDouble()"
            PrimitiveTypeWrapper.longInstance -> "(long) new Random().nextInt()"
            PrimitiveTypeWrapper.integerInstance -> "new Random().nextInt()"
            JavaType.stringInstance -> "\"test${Random().nextInt()}\""
            else -> "1"
        }

    protected val Column.testId: Any
        get() = when (javaType) {
            JavaType.stringInstance -> "\"1\""
            PrimitiveTypeWrapper.longInstance -> "1L"
            PrimitiveTypeWrapper.integerInstance -> 1
            else -> 1
        }

    protected val Column.javaName
        get() = extension.javaName(this.columnName)

    protected val Column.initializationString
        get() = when {
            javaType.shortName == "Boolean" -> toBoolean(defaultVal).toString()
            javaType.shortName == "Long" -> "${defaultVal}L"
            javaType.shortName == "Double" -> "${defaultVal}D"
            javaType.shortName == "Float" -> "${defaultVal}F"
            javaType.shortName == "BigDecimal" -> "new BigDecimal($defaultVal)"
            javaType.shortName == "String" -> "\"$defaultVal\""
            else -> defaultVal
        }

    /**
     * 表名
     */
    protected val tableName: String
        get() = table.tableName

    protected val catalog: String?
        get() = table.catalog

    protected val schema: String?
        get() = table.schema

    /**
     * 表类型
     */
    protected val tableType: String
        get() = table.tableType

    /**
     * 注释说明
     */
    protected val comment: String
        get() {
            val comment = if (table.comment.endsWith("表")) table.comment.substringBeforeLast("表") else table.comment
            return if (comment.isBlank()) {
                extension.comment
            } else {
                comment
            }
        }
    /**
     * 主键
     */
    protected val primaryKeys: List<Column>
        get() {
            val primaryKeys = table.primaryKeys
            return if (primaryKeys.isEmpty()) {
                val primaryKey = columns.find { it.columnName.equals(extension.primaryKeyName, true) }
                        ?: columns.find { it.comment.contains("主键") }
                if (primaryKey != null) {
                    listOf(primaryKey)
                } else {
                    emptyList()
                }
            } else {
                primaryKeys
            }
        }
    /**
     * 主键
     */
    protected val primaryKey: Column
        get() {
            if (primaryKeys.size == 1) {
                return primaryKeys[0]
            } else {
                throw GeneratorException("$tableName:没有单主键，$primaryKeyNames")
            }
        }
    /**
     * 字段
     */
    protected val columns: List<Column>
        get() = table.columns

    protected val primaryKeyNames: List<String>
        get() = primaryKeys.map { it.columnName }

    protected val Column.isPrimary
        get() = primaryKeyNames.contains(columnName)

    protected val pathName: String
        get() = English.plural(entityName)

    fun call(extension: GeneratorExtension, table: Table): Any? {
        this.extension = extension
        this.table = table
        return if (supports())
            doCall()
        else
            null
    }

    protected open fun supports(): Boolean {
        return true
    }

    protected abstract fun doCall(): Any
}