package cn.bestwu.generator

import cn.bestwu.generator.database.DatabaseMetaData
import cn.bestwu.generator.dom.java.element.JavaElement
import cn.bestwu.generator.dsl.Generator
import com.zaxxer.hikari.HikariConfig
import com.zaxxer.hikari.HikariDataSource
import java.sql.Connection
import java.util.*


/**
 * @author Peter Wu
 */
open class GeneratorExtension(
        /**
         * JDBC连接配置
         */
        val datasource: JDBCConnectionConfiguration = JDBCConnectionConfiguration(),
        /**
         * 生成文件基础路径
         */
        var path: String = "",
        /**
         * 覆盖所有已生成文件
         */
        var replaceAll: Boolean = false,
        /**
         * 生成代码包名
         */
        var packageName: String = "",
        /**
         * 基础代码包名
         */
        var basePackageName: String = "",
        /**
         * 子模块名称
         */
        var moduleName: String = "",
        /**
         * 子模块名称
         */
        var moduleComment: String = "",
        /**
         * 表前缀
         */
        var tablePrefix: String = "",
        /**
         * 注释说明
         */
        var comment: String = "",
        /**
         * 手动主键名
         */
        var primaryKeyName: String = "id",
        /**
         * 缩进
         */
        var indent: String = JavaElement.defaultIndent,
        /**
         * 模板
         */
        var generators: Array<Generator> = arrayOf(),
        /**
         * 相关数据表
         */
        var tableNames: Array<String> = arrayOf(),
        /**
         * 额外设置
         */
        var settings: Map<String, Any?> = mapOf()
) {
    /**
     * javaName
     */
    var javaName: (String) -> String = {
        javaName(it, false)
    }

    /**
     * ClassName
     */
    var className: (String) -> String = { str ->
        javaName(str.substringAfter(tablePrefix), true)
    }

    private fun javaName(str: String, capitalize: Boolean = false): String {
        val s = str.split(Regex("[^\\p{Alnum}]")).joinToString("") {
            it.toLowerCase().capitalize()
        }
        return if (capitalize) s else s.decapitalize()
    }


    fun <T> use(metaData: DatabaseMetaData.() -> T): T {
        val connection = datasource.getConnection()
        try {
            return metaData(DatabaseMetaData(connection.metaData, datasource.catalog, datasource.schema))
        } finally {
            connection.close()
        }
    }

    fun <T> run(connectionFun: Connection.() -> T): T {
        val connection = datasource.getConnection()
        try {
            return connectionFun(connection)
        } finally {
            connection.close()
        }
    }
}

class JDBCConnectionConfiguration(
        var url: String = "",
        var catalog: String? = null,
        val properties: Properties = Properties().apply {
            set("remarksReporting", "true") //oracle 读取表注释
            set("useInformationSchema", "true")//mysql 读取表注释
            set("characterEncoding", "utf8")
            set("user", "root")
            set("password", "root")
        }
) {
    var schema: String? = null
        get() {
            return if (field.isNullOrBlank() && url.startsWith("jdbc:oracle")) {
                username.toUpperCase()
            } else {
                field
            }
        }

    var driverClass: String = ""
        get() {
            return if (field.isBlank()) {
                when {
                    url.startsWith("jdbc:mysql") -> "com.mysql.jdbc.Driver"
                    url.startsWith("jdbc:oracle") -> "oracle.jdbc.OracleDriver"
                    else -> field
                }
            } else {
                field
            }

        }

    var username: String
        set(value) = properties.set("user", value)
        get() = properties.getProperty("user")
    var password: String by properties

    private var hikariDataSource: HikariDataSource? = null

    fun getConnection(): Connection {
        if (hikariDataSource == null) {
            val config = HikariConfig()
            config.maximumPoolSize = 10
            config.driverClassName = driverClass
            config.jdbcUrl = url
            config.username = username
            config.password = password
            hikariDataSource = HikariDataSource(config)
        }
        return hikariDataSource!!.connection
    }
}