package cn.bestwu.autodoc.gradle.plugin

import cn.bestwu.autodoc.core.AutodocExtension
import cn.bestwu.autodoc.core.HtmlGenerator
import cn.bestwu.autodoc.core.MDGenerator
import cn.bestwu.gradle.profile.ProfileExtension
import cn.bestwu.gradle.profile.ProfilePlugin
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.distribution.DistributionContainer
import org.gradle.api.distribution.plugins.DistributionPlugin
import org.gradle.api.plugins.ApplicationPlugin
import org.gradle.api.plugins.JavaPlugin
import org.gradle.api.plugins.JavaPluginConvention
import org.gradle.api.tasks.SourceSet
import org.gradle.api.tasks.application.CreateStartScripts
import org.gradle.language.jvm.tasks.ProcessResources
import java.io.File

/**
 *
 * 注册task
 *
 * @author Peter Wu
 */
class AutodocPlugin : Plugin<Project> {

    @Suppress("DEPRECATION")
    override fun apply(project: Project) {
        project.extensions.create("autodoc", AutodocExtension::class.java)

        project.afterEvaluate {
            val apidocExtension = project.extensions.findByType(AutodocExtension::class.java)!!
            if (apidocExtension.projectName.isBlank()) {
                apidocExtension.projectName = project.name
            }
            val pathNames = project.findProperty("autodoc.path-names") as? String
            if (!pathNames.isNullOrBlank()) {
                apidocExtension.pathNames = pathNames!!.split(",").toTypedArray()
            }

            apidocExtension.basePath = project.file("./")
            val docOutputDir = File((project.tasks.getByName("processResources") as ProcessResources).destinationDir.absolutePath, "doc")
            if (apidocExtension.output == null)
                apidocExtension.output = docOutputDir
            val docPath = apidocExtension.sourceFile

            val group = "autodoc"
            project.tasks.create("mddoc") {
                it.group = group
                it.mustRunAfter("clean")

                if (docPath.exists()) {
                    it.inputs.dir(docPath)
                }
                if (apidocExtension.outputFile.exists())
                    it.outputs.dir(apidocExtension.outputFile)
                it.doLast {
                    MDGenerator.call(project.extensions.findByType(AutodocExtension::class.java)!!)
                }
            }
            project.tasks.create("htmldoc") {
                it.group = group
                it.dependsOn("mddoc")

                if (docPath.exists()) {
                    it.inputs.dir(docPath)
                }
                if (apidocExtension.outputFile.exists())
                    it.outputs.dir(apidocExtension.outputFile)
                it.doLast {
                    HtmlGenerator.call(project.extensions.findByType(AutodocExtension::class.java)!!)
                }
            }
            project.tasks.create("reformatApi") {
                it.group = group
                if (docPath.exists()) {
                    it.inputs.dir(docPath)
                }
                it.doLast {
                    MDGenerator.reformatApi(project.extensions.findByType(AutodocExtension::class.java)!!)
                }
            }
        }
        project.plugins.apply(ProfilePlugin::class.java)

        val defaultVersion = "0.0.1"
        val version = project.findProperty("autodoc.version") ?: defaultVersion

        project.configurations.create(API_TEST_COMPILE_CONFIGURATION_NAME)
        project.dependencies.add(API_TEST_COMPILE_CONFIGURATION_NAME, "cn.bestwu.autodoc:test-page:$version")
        project.dependencies.add("compileOnly", "cn.bestwu.autodoc:gen:$version")
        project.dependencies.add("testCompile", "cn.bestwu.autodoc:gen:$version")

        project.extensions.configure(AutodocExtension::class.java) {
            val applicationHost = project.findProperty("autodoc.default-doc-host") as? String
            if (!applicationHost.isNullOrBlank())
                it.defaultDocHost = applicationHost!!
            val applicationName = project.findProperty("application.name") as? String
            if (!applicationName.isNullOrBlank())
                it.projectName = applicationName!!
        }

        project.tasks.getByName("compileJava") {
            it.dependsOn("htmldoc")
        }

        val configuration = project.configurations.getByName(API_TEST_COMPILE_CONFIGURATION_NAME)
        val addConfiguration = configuration - project.configurations.getByName(JavaPlugin.COMPILE_CONFIGURATION_NAME)
        val sourceSet = project.convention.getPlugin(JavaPluginConvention::class.java).sourceSets.getByName(SourceSet.MAIN_SOURCE_SET_NAME)
        sourceSet.runtimeClasspath += addConfiguration
        project.extensions.configure(ProfileExtension::class.java) {
            it.closure {
                if (!it.releases.contains(it.active) && !tasks.getByName(JavaPlugin.COMPILE_JAVA_TASK_NAME).state.executed) {
                    project.extensions.getByType(DistributionContainer::class.java).getAt(DistributionPlugin.MAIN_DISTRIBUTION_NAME).contents {
                        val startScripts = project.tasks.getByName(ApplicationPlugin.TASK_START_SCRIPTS_NAME) as CreateStartScripts
                        startScripts.classpath += addConfiguration
                        it.from(addConfiguration) {
                            it.into("lib")
                        }
                    }
                }
            }
            it.releaseClosure {
                project.tasks.getByName("mddoc").enabled = false
                project.tasks.getByName("htmldoc").enabled = false
                val processResources = project.tasks.getByName("processResources") as ProcessResources
                processResources.exclude("doc")
                sourceSet.runtimeClasspath -= addConfiguration
            }
        }
    }

    companion object {
        private const val API_TEST_COMPILE_CONFIGURATION_NAME = "apiTestCompile"
    }

}