package cn.bestwu.api.sign;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import javax.servlet.http.HttpServletRequest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.DigestUtils;
import org.springframework.util.MultiValueMap;
import org.springframework.util.StringUtils;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

/**
 * 签名适配
 * <p>
 * 默认签名算法为MD5
 * <p>
 * SIGN_TYPE = "MD5";
 *
 * @author Peter Wu
 */
public abstract class AbstractApiSignAlgorithm implements ApiSignAlgorithm {

  protected static Logger log = LoggerFactory.getLogger(AbstractApiSignAlgorithm.class);
  protected final ApiSignProperties properties;

  protected AbstractApiSignAlgorithm(ApiSignProperties properties) {
    this.properties = properties;
  }

  /**
   * @param requestParams 请求参数
   * @param client_id 客户端ID
   * @param client_secret 客户端密钥
   * @param timestamp 时间 毫秒数
   * @return 签名
   */
  public static String sign(Map<String, String[]> requestParams, String client_id,
      String client_secret,
      String timestamp) {
    List<String> keys = new ArrayList<>(requestParams.keySet());
    Collections.sort(keys);
    StringBuilder prestr = new StringBuilder("");
    for (String key : keys) {
      String[] values = requestParams.get(key);
      String value = "";
      int length = values.length;
      for (int i = 0; i < length; i++) {
        value += values[i];
        value += (i == length - 1) ? "" : ",";
      }
      if (value == null || value.equals("") || key.equalsIgnoreCase("sign")
          || key.equalsIgnoreCase("sign_type")) {
        continue;
      }
      prestr.append(key).append("=").append(value).append("&");
    }
    if (StringUtils.hasText(client_id)) {
      prestr.append(client_id).append("&").append(timestamp);
    } else {
      prestr.append(timestamp);
    }
    if (log.isDebugEnabled()) {
      log.debug("待签名参数字符串：{}", prestr);
    }
    prestr = prestr.append("{").append(client_secret).append("}");
    return DigestUtils.md5DigestAsHex(prestr.toString().getBytes());
  }

  /**
   * @param requestParams 请求参数
   * @param client_id 客户端ID
   * @param client_secret 客户端密钥
   * @param timestamp 时间 毫秒数
   * @return 签名
   */
  public static String sign(MultiValueMap<String, String> requestParams, String client_id,
      String client_secret, String timestamp) {
    List<String> keys = new ArrayList<>(requestParams.keySet());
    Collections.sort(keys);
    StringBuilder prestr = new StringBuilder("");
    for (String key : keys) {
      List<String> values = requestParams.get(key);
      String value = "";
      int length = values.size();
      for (int i = 0; i < length; i++) {
        value += values.get(i);
        value += (i == length - 1) ? "" : ",";
      }
      if (value == null || value.equals("") || key.equalsIgnoreCase("sign")
          || key.equalsIgnoreCase("sign_type")) {
        continue;
      }
      prestr.append(key).append("=").append(value).append("&");
    }
    if (StringUtils.hasText(client_id)) {
      prestr.append(client_id).append("&").append(timestamp);
    } else {
      prestr.append(timestamp);
    }
    if (log.isDebugEnabled()) {
      log.debug("待签名参数字符串：{}", prestr);
    }
    prestr = prestr.append("{").append(client_secret).append("}");
    return DigestUtils.md5DigestAsHex(prestr.toString().getBytes());
  }


  /**
   * @param request 请求
   * @return request 是否跳过
   */
  protected boolean skip(HttpServletRequest request) {
    if (properties.isCanSkip()) {
      Boolean skip_sign = (Boolean) request.getAttribute("SKIP_SIGN");
      return skip_sign == null ? false : skip_sign;
    } else {
      return false;
    }
  }

  /**
   * 过滤签名验证
   */
  public static void skip() {
    getRequest().setAttribute("SKIP_SIGN", true);
  }

  private static HttpServletRequest getRequest() {
    ServletRequestAttributes requestAttributes = (ServletRequestAttributes) RequestContextHolder
        .getRequestAttributes();
    if (requestAttributes == null) {
      return null;
    }
    return requestAttributes.getRequest();
  }

  @Override
  public String sign(Map<String, String[]> requestParams, String client_id) {
    String timestamp = String.valueOf(System.currentTimeMillis());
    return base64Sign(sign(requestParams, client_id, loadClientSignKeyByClientId(client_id),
        timestamp), client_id, timestamp);
  }

  @Override
  public String sign(MultiValueMap<String, String> requestParams, String client_id) {
    String timestamp = String.valueOf(System.currentTimeMillis());
    return base64Sign(sign(requestParams, client_id, loadClientSignKeyByClientId(client_id),
        timestamp), client_id, timestamp);
  }


  /**
   * @param client_id client_id
   * @return ClientSignKey
   */
  protected abstract String loadClientSignKeyByClientId(String client_id);

  /**
   * @param sign 一次签名
   * @param client_id 客户端ID
   * @param timestamp 时间 毫秒数
   * @return 签名
   */
  protected abstract String base64Sign(String sign, String client_id,
      String timestamp);
}
