package com.bambooclod.epasswork;

import android.content.Context;

import com.alibaba.fastjson.JSON;
import com.bambooclod.epassbase.api.EpassBaseSDK;
import com.bambooclod.epassbase.bean.BaseEncryptObserver;
import com.bambooclod.epassbase.bean.BaseObserver;
import com.bambooclod.epassbase.bean.EncrypteData;
import com.bambooclod.epassbase.bean.GetEncryptStringResponse;
import com.bambooclod.epassbase.config.InitConfigValue;
import com.bambooclod.epassbase.log.LogUtil;
import com.bambooclod.epassbase.sp.SPUtils;
import com.bambooclod.epasswork.bean.AdditionnalAuthData;
import com.bambooclod.epasswork.bean.GetAmTokenRequest;
import com.bambooclod.epasswork.bean.GetEncryptStringRequest;
import com.bambooclod.epasswork.bean.SelectSubAccountRequest;
import com.bambooclod.epasswork.bean.VerifyJwtRequest;
import com.bambooclod.epasswork.callback.RequestListener;
import com.bambooclod.epasswork.http.WorkService;

import io.reactivex.Observable;
import io.reactivex.Observer;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.schedulers.Schedulers;

/**
 * Comment: 工作台
 *
 * @author Vangelis.Wang in BambooCloud
 * @date 2019-07-02
 * Email:wangpei@bamboocloud.cn
 */
public class EpassWorkRequest {

    private EpassWorkRequest() {
    }

    public static final EpassWorkRequest authRequest = new EpassWorkRequest();

    public static EpassWorkRequest get() {
        return authRequest;
    }

    /**
     * 校验Jwt是否合法
     */
    public void verifyJwt(final Context context, final RequestListener listener) {
        String appID = (String) SPUtils.get(context, InitConfigValue.APP_ID, InitConfigValue.DEFAULT);
        String jwt = (String) SPUtils.get(context, InitConfigValue.USER_JWT, InitConfigValue.DEFAULT);
        if (InitConfigValue.DEFAULT.equals(jwt)) {
            listener.onCodeError("NoJwt", "当前内部无JWT数据，未登录，需要登录后才能调用此接口");
            return;
        }
        VerifyJwtRequest deleteGesturePasswordData = new VerifyJwtRequest(appID, jwt);

        //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
        SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);

        request(context, deleteGesturePasswordData, WorkService.verifyJwt
                , new RequestListener() {
                    @Override
                    public void onSuccess(Object body) {
                        if (listener != null) {
                            listener.onSuccess(body);
                        }
                        SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                    }

                    @Override
                    public void onCodeError(String code, String message) {
                        if (listener != null) {
                            listener.onCodeError(code, message);
                        }
                        SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                    }
                });

    }

    /**
     * 获取Jwt内容
     */
    public String getJwtString(Context context) {
        String jwt = (String) SPUtils.get(context, InitConfigValue.USER_JWT, InitConfigValue.DEFAULT);
        if (!InitConfigValue.DEFAULT.equals(jwt)) {
            return jwt;
        }
        return "";
    }

    /**
     * 查询子账号
     *
     * @param context  Context
     * @param subAppId 子账号ID
     */
    public void querySubAccount(final Context context, String subAppId, final RequestListener listener) {
        String jwt = (String) SPUtils.get(context, InitConfigValue.USER_JWT, InitConfigValue.DEFAULT);

        if (InitConfigValue.DEFAULT.equals(jwt)) {
            listener.onCodeError("NoJwt", "当前内部无JWT数据，未登录，需要登录后才能调用此接口");
            return;
        }
        SelectSubAccountRequest request = new SelectSubAccountRequest(subAppId);

        //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
        SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);

        request(context, request, WorkService.selectSubAccount
                , new RequestListener() {
                    @Override
                    public void onSuccess(Object body) {
                        if (listener != null) {
                            listener.onSuccess(body);
                        }
                        SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                    }

                    @Override
                    public void onCodeError(String code, String message) {
                        if (listener != null) {
                            listener.onCodeError(code, message);
                        }
                        SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                    }
                });
    }

    /**
     * 互信接口
     *
     * @param context  context
     * @param listener listener
     */
    public void getAmToken(final Context context, String ip, final RequestListener listener) {
        String jwt = (String) SPUtils.get(context, InitConfigValue.USER_JWT, InitConfigValue.DEFAULT);
        if (InitConfigValue.DEFAULT.equals(jwt)) {
            listener.onCodeError("NoJwt", "当前内部无JWT数据，未登录，需要登录后才能调用此接口");
            return;
        }
        //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
        SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);
        GetAmTokenRequest request = new GetAmTokenRequest(ip);
        request(context, request, WorkService.getAMToken, new RequestListener() {
            @Override
            public void onSuccess(Object body) {
                if (listener != null) {
                    listener.onSuccess(body);
                }
                SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
            }

            @Override
            public void onCodeError(String code, String message) {
                if (listener != null) {
                    listener.onCodeError(code, message);
                }
                SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
            }
        });

    }

    /**
     * 获取授权的应用列表
     *
     * @param context  context
     * @param listener listener
     */
    public void getAuthAppList(final Context context, final RequestListener listener) {
        String jwt = (String) SPUtils.get(context, InitConfigValue.USER_JWT, InitConfigValue.DEFAULT);
        if (InitConfigValue.DEFAULT.equals(jwt)) {
            listener.onCodeError("NoJwt", "当前内部无JWT数据，未登录，需要登录后才能调用此接口");
            return;
        }
        //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
        SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);

        request(context, new Object(), WorkService.getUserAuthAppList, new RequestListener() {
            @Override
            public void onSuccess(Object body) {
                if (listener != null) {
                    listener.onSuccess(body);
                }
                SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
            }

            @Override
            public void onCodeError(String code, String message) {
                if (listener != null) {
                    listener.onCodeError(code, message);
                }
                SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
            }
        });
    }

    /**
     * 获取加密串
     *
     * @param context    Context
     * @param subAccount 子账号
     */
    public void getEncryptString(final Context context, String appID, String subAccount, final RequestListener listener) {
        String jwt = (String) SPUtils.get(context, InitConfigValue.USER_JWT, InitConfigValue.DEFAULT);
        if (InitConfigValue.DEFAULT.equals(jwt)) {
            listener.onCodeError("NoJwt", "当前内部无JWT数据，未登录，需要登录后才能调用此接口");
            return;
        }
        GetEncryptStringRequest request = new GetEncryptStringRequest(appID
                , jwt, subAccount);

        //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
        SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);

        request(context, request, WorkService.getEncryptString
                , new RequestListener() {
                    @Override
                    public void onSuccess(Object body) {
                        GetEncryptStringResponse response = JSON.parseObject(body.toString()
                                , GetEncryptStringResponse.class);
                        if (listener != null) {
                            listener.onSuccess(response.getEncryptedString());
                        }
                        SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                    }

                    @Override
                    public void onCodeError(String code, String message) {
                        if (listener != null) {
                            listener.onCodeError(code, message);
                        }
                        SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                    }
                });
    }

    /**
     * 统一接口请求
     *
     * @param context  context
     * @param t        请求体
     * @param url      地址
     * @param listener 回调
     * @param <T>      请求体
     */
    private <T> void request(Context context, T t, String url, final RequestListener listener) {
        //预请求体
        Observer observer;
        Observable observable;

        //是否内容加密
        boolean isEncrypt = SPUtils.isEncrytContent(context);
        //Retrofit
        WorkService apiService = EpassBaseSDK.Builder().getAPi(context, WorkService.class);

        if (isEncrypt) {
            String body = JSON.toJSONString(t);
            String authKey = (String) SPUtils.get(context, InitConfigValue.KEY_ENCRYPT, InitConfigValue.DEFAULT);
            Object data = new EncrypteData(EpassBaseSDK.Builder().encryt(body, authKey));
            observable = apiService.commonRequestByEncrypt(url, data);
            observer = new BaseEncryptObserver(context) {
                @Override
                protected void onSuccess(Object body) {
                    LogUtil.getInstance().i("请求成功：" + body);
                    listener.onSuccess(body);
                }

                @Override
                protected void onCodeError(String code, String message, Object body) {
                    LogUtil.getInstance().d("请求失败：" + code);
                    if ("error.dispatchercenter.jwt.additional.auth".equals(code)) {
                        AdditionnalAuthData data = JSON.parseObject(body.toString(), AdditionnalAuthData.class);
                        listener.onCodeError(code, data.getNextAuthType());
                    } else {
                        listener.onCodeError(code, message);
                    }
                }

                @Override
                protected void onFailure(Throwable e, boolean isNetWorkError) {
                    super.onFailure(e, isNetWorkError);
                    listener.onCodeError(e.getMessage(), e.getMessage());
                }
            };
        } else {
            observable = apiService.commonRequest(url, t);
            observer = new BaseObserver<Object>(context) {
                @Override
                protected void onSuccess(Object body) {
                    LogUtil.getInstance().i("请求成功：" + body);
                    listener.onSuccess(body);
                }

                @Override
                protected void onCodeError(String code, String message, Object object) {
                    LogUtil.getInstance().d("请求失败：" + code);
                    if ("error.dispatchercenter.jwt.additional.auth".equals(code)) {
                        AdditionnalAuthData data = JSON.parseObject(object.toString(), AdditionnalAuthData.class);
                        listener.onCodeError(code, data.getNextAuthType());
                    } else {
                        listener.onCodeError(code, message);
                    }
                }

                @Override
                protected void onFailure(Throwable e, boolean isNetWorkError) {
                    super.onFailure(e, isNetWorkError);
                    listener.onCodeError(e.getMessage(), e.getMessage());
                }
            };
        }

        observable
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(observer);
    }
}
