package com.bambooclod.epasswork;

import android.content.Context;
import android.util.Base64;
import android.widget.Toast;

import com.alibaba.fastjson.JSON;
import com.bambooclod.epassbase.api.EpassBaseSDK;
import com.bambooclod.epassbase.bean.BaseEncryptObserver;
import com.bambooclod.epassbase.bean.BaseObserver;
import com.bambooclod.epassbase.bean.EncrypteData;
import com.bambooclod.epassbase.bean.GetEncryptStringResponse;
import com.bambooclod.epassbase.config.InitConfigValue;
import com.bambooclod.epassbase.log.LogUtil;
import com.bambooclod.epassbase.sp.SPUtils;
import com.bambooclod.epasswork.bean.AdditionnalAuthData;
import com.bambooclod.epasswork.bean.GetEncryptStringRequest;
import com.bambooclod.epasswork.bean.JwtDecodeData;
import com.bambooclod.epasswork.bean.SelectSubAccountRequest;
import com.bambooclod.epasswork.bean.VerifyJwtRequest;
import com.bambooclod.epasswork.callback.RequestListener;
import com.bambooclod.epasswork.http.WorkService;

import io.reactivex.Observable;
import io.reactivex.Observer;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.schedulers.Schedulers;

/**
 * Comment: 工作台
 *
 * @author Vangelis.Wang in BambooCloud
 * @date 2019-07-02
 * Email:wangpei@bamboocloud.cn
 */
public class EpassWorkRequest {

    private EpassWorkRequest() {
    }

    public static final EpassWorkRequest authRequest = new EpassWorkRequest();

    public static EpassWorkRequest get() {
        return authRequest;
    }

    /**
     * 检测Jwt是否存在
     *
     * @param context Context
     */
    private boolean checkJwtExist(Context context) {
        String jwt = (String) SPUtils.get(context, InitConfigValue.USER_JWT, InitConfigValue.DEFAULT);
        //使用JWT获取用户名
        if (InitConfigValue.DEFAULT.equals(jwt)) {
            Toast.makeText(context, "当前内部无JWT数据，未登录，需要登录后才能调用此接口"
                    , Toast.LENGTH_SHORT).show();
            return false;
        } else {
            return true;
        }
    }

    /**
     * 解析Jwt中的内容
     */
    private JwtDecodeData parseJwtToInfo(Context context) {
        String jwt = (String) SPUtils.get(context, InitConfigValue.USER_JWT, InitConfigValue.DEFAULT);

        String[] forma = jwt.split("\\.");
        String jwtContent = new String(Base64.decode(forma[1], Base64.DEFAULT));
        return JSON.parseObject(jwtContent, JwtDecodeData.class);
    }

    /**
     * 校验Jwt是否合法
     */
    public void verifyJwt(final Context context, final RequestListener listener) {
        String appID = (String) SPUtils.get(context, InitConfigValue.APP_ID, InitConfigValue.DEFAULT);
        String jwt = (String) SPUtils.get(context, InitConfigValue.USER_JWT, InitConfigValue.DEFAULT);

        if (checkJwtExist(context)) {
            JwtDecodeData decodeData = parseJwtToInfo(context);
            VerifyJwtRequest deleteGesturePasswordData = new VerifyJwtRequest(appID, jwt);

            //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);

            request(context, deleteGesturePasswordData, WorkService.verifyJwt
                    , new RequestListener() {
                        @Override
                        public void onSuccess(Object body) {
                            if (listener != null) {
                                listener.onSuccess(body);
                            }
                            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                        }

                        @Override
                        public void onCodeError(String code, String message) {
                            if (listener != null) {
                                listener.onCodeError(code, message);
                            }
                            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                        }
                    });
        }
    }

    /**
     * 获取Jwt内容
     */
    public String getJwtString(Context context) {
        String jwt = (String) SPUtils.get(context, InitConfigValue.USER_JWT, InitConfigValue.DEFAULT);
        if (checkJwtExist(context)) {
            return jwt;
        }
        return "";
    }

    /**
     * 查询子账号
     *
     * @param context  Context
     * @param subAppId 子账号ID
     */
    public void querySubAccount(final Context context, String subAppId, final RequestListener listener) {
        String jwt = (String) SPUtils.get(context, InitConfigValue.USER_JWT, InitConfigValue.DEFAULT);

        if (checkJwtExist(context)) {
            SelectSubAccountRequest request = new SelectSubAccountRequest(subAppId);

            //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);

            request(context, request, WorkService.selectSubAccount
                    , new RequestListener() {
                        @Override
                        public void onSuccess(Object body) {
                            if (listener != null) {
                                listener.onSuccess(body);
                            }
                            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                        }

                        @Override
                        public void onCodeError(String code, String message) {
                            if (listener != null) {
                                listener.onCodeError(code, message);
                            }
                            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                        }
                    });
        }
    }

    /**
     * 获取加密串
     *
     * @param context    Context
     * @param subAccount 子账号
     */
    public void getEncryptString(final Context context, String appID, String subAccount, final RequestListener listener) {
        String jwt = (String) SPUtils.get(context, InitConfigValue.USER_JWT, InitConfigValue.DEFAULT);

        if (checkJwtExist(context)) {
            GetEncryptStringRequest request = new GetEncryptStringRequest(appID
                    , jwt, subAccount);

            //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);

            request(context, request, WorkService.getEncryptString
                    , new RequestListener() {
                        @Override
                        public void onSuccess(Object body) {
                            GetEncryptStringResponse response = JSON.parseObject(body.toString()
                                    , GetEncryptStringResponse.class);
                            if (listener != null) {
                                listener.onSuccess(response.getEncryptedString());
                            }
                            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                        }

                        @Override
                        public void onCodeError(String code, String message) {
                            if (listener != null) {
                                listener.onCodeError(code, message);
                            }
                            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                        }
                    });
        }
    }

    /**
     * 统一接口请求
     *
     * @param context  context
     * @param t        请求体
     * @param url      地址
     * @param listener 回调
     * @param <T>      请求体
     */
    private <T> void request(Context context, T t, String url, final RequestListener listener) {
        //预请求体
        Observer observer;
        Observable observable;

        //是否内容加密
        boolean isEncrypt = SPUtils.isEncrytContent(context);
        //Retrofit
        WorkService apiService = EpassBaseSDK.Builder().getAPi(context, WorkService.class);

        if (isEncrypt) {
            String body = JSON.toJSONString(t);
            String authKey = (String) SPUtils.get(context, InitConfigValue.KEY_ENCRYPT, InitConfigValue.DEFAULT);
            Object data = new EncrypteData(EpassBaseSDK.Builder().encryt(body, authKey));
            observable = apiService.commonRequestByEncrypt(url, data);
            observer = new BaseEncryptObserver(context) {
                @Override
                protected void onSuccess(Object body) {
                    LogUtil.getInstance().i("请求成功：" + body);
                    listener.onSuccess(body);
                }

                @Override
                protected void onCodeError(String code, String message, Object body) {
                    LogUtil.getInstance().d("请求失败：" + code);
                    if ("error.dispatchercenter.jwt.additional.auth".equals(code)) {
                        AdditionnalAuthData data = JSON.parseObject(body.toString(), AdditionnalAuthData.class);
                        listener.onCodeError(code, "需要进行增强认证,认证类型为：" + data.getNextAuthType());
                    } else {
                        listener.onCodeError(code, message);
                    }
                }
            };
        } else {
            observable = apiService.commonRequest(url, t);
            observer = new BaseObserver<Object>(context) {
                @Override
                protected void onSuccess(Object body) {
                    LogUtil.getInstance().i("请求成功：" + body);
                    listener.onSuccess(body);
                }

                @Override
                protected void onCodeError(String code, String message, Object object) {
                    LogUtil.getInstance().d("请求失败：" + code);
                    if ("error.dispatchercenter.jwt.additional.auth".equals(code)) {
                        AdditionnalAuthData data = JSON.parseObject(object.toString(), AdditionnalAuthData.class);
                        listener.onCodeError(code, "需要进行增强认证,认证类型为：" + data.getNextAuthType());
                    } else {
                        listener.onCodeError(code, message);
                    }
                }
            };
        }

        observable
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(observer);
    }
}
