package com.bambooclod.epassself.api;

import android.content.Context;
import android.util.Base64;
import android.widget.Toast;

import com.alibaba.fastjson.JSON;
import com.bambooclod.epassbase.api.EpassBaseSDK;
import com.bambooclod.epassbase.bean.BaseEncryptObserver;
import com.bambooclod.epassbase.bean.BaseObserver;
import com.bambooclod.epassbase.bean.EncrypteData;
import com.bambooclod.epassbase.config.InitConfigValue;
import com.bambooclod.epassbase.log.LogUtil;
import com.bambooclod.epassbase.sp.SPUtils;
import com.bambooclod.epassself.bean.AddFaceData;
import com.bambooclod.epassself.bean.AddVoiceData;
import com.bambooclod.epassself.bean.DeleteFaceData;
import com.bambooclod.epassself.bean.DeleteGesturePasswordData;
import com.bambooclod.epassself.bean.DeleteTrushDeviceData;
import com.bambooclod.epassself.bean.DeleteVoiceData;
import com.bambooclod.epassself.bean.GetAuthStatusResponse;
import com.bambooclod.epassself.bean.GetTrustDeviceList;
import com.bambooclod.epassself.bean.GetTrustDeviceResponse;
import com.bambooclod.epassself.bean.GetUserInfoData;
import com.bambooclod.epassself.bean.GetUserNameByEncryptData;
import com.bambooclod.epassself.bean.JwtDecodeData;
import com.bambooclod.epassself.bean.LogoutSessionData;
import com.bambooclod.epassself.bean.LogoutSessionResponse;
import com.bambooclod.epassself.bean.ModifyDeviceRemarkData;
import com.bambooclod.epassself.bean.ModifyGestureData;
import com.bambooclod.epassself.bean.ModifyPasswordData;
import com.bambooclod.epassself.bean.SaveGesturePasswordData;
import com.bambooclod.epassself.bean.SendVerCodeByForgotPwdData;
import com.bambooclod.epassself.bean.UpdatePwdByVerCodeData;
import com.bambooclod.epassself.bean.ViewTokenResponse;
import com.bambooclod.epassself.callback.CommonResultListener;
import com.bambooclod.epassself.callback.GetAuthTypeStatusListener;
import com.bambooclod.epassself.callback.GetTrustDeviceListListener;
import com.bambooclod.epassself.callback.GetUserInfoListener;
import com.bambooclod.epassself.callback.GetViewTokenListener;
import com.bambooclod.epassself.http.SelfService;

import java.util.ArrayList;
import java.util.List;

import io.reactivex.Observable;
import io.reactivex.Observer;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.schedulers.Schedulers;

/**
 * Comment: 移动自服务请求类
 *
 * @author Vangelis.Wang in BambooCloud
 * @date 2019-06-17
 * Email:wangpei@bamboocloud.cn
 */
public class EpassSelfRequest {

    private EpassSelfRequest() {
    }

    public static final EpassSelfRequest authRequest = new EpassSelfRequest();

    public static EpassSelfRequest get() {
        return authRequest;
    }

    /**
     * 检测Jwt是否存在
     *
     * @param context Context
     */
    private boolean checkJwtExist(Context context) {
        String jwt = (String) SPUtils.get(context, InitConfigValue.USER_JWT, InitConfigValue.DEFAULT);
        //使用JWT获取用户名
        if (InitConfigValue.DEFAULT.equals(jwt)) {
            Toast.makeText(context, "当前内部无JWT数据，未登录，需要登录后才能调用此接口"
                    , Toast.LENGTH_SHORT).show();
            return false;
        } else {
            return true;
        }
    }

    /**
     * 解析Jwt中的内容
     */
    private JwtDecodeData parseJwtToInfo(Context context) {
        String jwt = (String) SPUtils.get(context, InitConfigValue.USER_JWT, InitConfigValue.DEFAULT);
        String[] forma = jwt.split("\\.");
        String jwtContent = new String(Base64.decode(forma[1], Base64.DEFAULT));
        return JSON.parseObject(jwtContent, JwtDecodeData.class);
    }

    /**
     * 获取会话列表
     *
     * @param context  Context
     * @param listener 回调
     */
    public void getViewTokens(final Context context, final GetViewTokenListener listener) {
        if (checkJwtExist(context)) {
            //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);

            request(context, new Object(), SelfService.getViewTokenDetail, new RequestListener() {
                @Override
                public void onSuccess(Object body) {
                    ViewTokenResponse response = JSON.parseObject(body.toString(), ViewTokenResponse.class);
                    if (response != null) {
                        if (listener != null) {
                            listener.getSessions(response.getSessionList());
                        }
                    }
                    SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                }

                @Override
                public void onCodeError(String code, String message) {
                    if (listener != null) {
                        listener.error(code, message);
                    }
                    SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                }
            });
        }
    }

    /**
     * 注销会话
     *
     * @param context  Context
     * @param listener 回调
     */
    public void logoutSession(final Context context, String tokenKey, final CommonResultListener listener) {
        if (checkJwtExist(context)) {
            JwtDecodeData decodeData = parseJwtToInfo(context);
            LogoutSessionData logoutSessionData = new LogoutSessionData(decodeData.getUserid(), tokenKey);

            //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);

            request(context, logoutSessionData, SelfService.logoutSession, new RequestListener() {
                @Override
                public void onSuccess(Object body) {
                    LogoutSessionResponse response = JSON.parseObject(body.toString(), LogoutSessionResponse.class);
                    if (listener != null) {
                        listener.success();
                    }
                    SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                }

                @Override
                public void onCodeError(String code, String message) {
                    if (listener != null) {
                        listener.error(code, message);
                    }
                    SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                }
            });
        }
    }

    /**
     * 获取认证状态
     */
    public void getAuthStatus(final Context context, final GetAuthTypeStatusListener listener) {
        //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
        SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);

        request(context, new Object(), SelfService.getAuthStatus, new RequestListener() {
            @Override
            public void onSuccess(Object body) {
                GetAuthStatusResponse response = JSON.parseObject(body.toString(), GetAuthStatusResponse.class);
                if (listener != null) {
                    listener.success(response.isVoice(), response.isFace(), response.isGesture());
                }
                SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
            }

            @Override
            public void onCodeError(String code, String message) {
                if (listener != null) {
                    listener.error(code, message);
                }
                SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
            }
        });
    }

    /**
     * 获取信任设备列表
     */
    public void getTrustDevices(Context context, String pageSize, String page
            , final GetTrustDeviceListListener listener) {
        getTrustDevices(context, "", pageSize, page, listener);
    }

    /**
     * 获取信任设备列表
     *
     * @param context   Context
     * @param condition 查询条件
     * @param pageSize  每页多少条
     * @param page      第几页
     * @param listener  回调
     */
    public void getTrustDevices(final Context context, String condition, final String pageSize, String page
            , final GetTrustDeviceListListener listener) {
        if (checkJwtExist(context)) {
            GetTrustDeviceList getTrustDeviceList = new GetTrustDeviceList(condition, pageSize
                    , page);

            //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);

            request(context, getTrustDeviceList, SelfService.getDeviceList, new RequestListener() {
                @Override
                public void onSuccess(Object body) {
                    GetTrustDeviceResponse response = JSON.parseObject(body.toString(), GetTrustDeviceResponse.class);
                    if (response.getData() != null) {
                        if (listener != null) {
                            listener.success(response.getData());
                        }
                    }
                    SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                }

                @Override
                public void onCodeError(String code, String message) {
                    if (listener != null) {
                        listener.error(code, message);
                    }
                    SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                }
            });
        }
    }

    /**
     * 修改默认设备备注
     *
     * @param context Context
     * @param id      设备ID
     * @param remarks 新的备注
     */
    public void modifyTrustDeviceRemark(final Context context, String id, String remarks
            , final CommonResultListener listener) {
        if (checkJwtExist(context)) {
            ModifyDeviceRemarkData modifyDeviceRemarkData = new ModifyDeviceRemarkData(id, remarks);

            //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);

            request(context, modifyDeviceRemarkData, SelfService.modifyDeviceRemark, new RequestListener() {
                @Override
                public void onSuccess(Object body) {
                    if (listener != null) {
                        listener.success();
                    }
                    SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                }

                @Override
                public void onCodeError(String code, String message) {
                    if (listener != null) {
                        listener.error(code, message);
                    }
                    SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                }
            });
        }
    }


    /**
     * 删除信任设备
     *
     * @param context  Context
     * @param id       设备ID
     * @param tokenID  TokenID
     * @param listener 回调
     */
    public void deleteTrustDevice(final Context context, String id, String tokenID, final CommonResultListener listener) {
        if (checkJwtExist(context)) {
            DeleteTrushDeviceData deleteTrushDeviceData = new DeleteTrushDeviceData(id, tokenID, EpassBaseSDK.Builder().getDeviceID(context));
            //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);

            request(context, deleteTrushDeviceData, SelfService.deleteDeviceList, new RequestListener() {
                @Override
                public void onSuccess(Object body) {
                    if (listener != null) {
                        listener.success();
                    }
                    SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                }

                @Override
                public void onCodeError(String code, String message) {
                    if (listener != null) {
                        listener.error(code, message);
                    }
                    SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                }
            });
        }


    }

    /**
     * 获取用户信息
     *
     * @param context  Context
     * @param listener 回调
     */
    public void getUserInfo(final Context context, int isForceLoad, final GetUserInfoListener listener) {
        GetUserInfoData getUserInfoData = new GetUserInfoData(isForceLoad);

        //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
        SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);

        request(context, getUserInfoData, SelfService.getUserInfo, new RequestListener() {
            @Override
            public void onSuccess(Object body) {
                if (listener != null) {
                    // TODO: 2019-06-28 这里的数据体可能会变，现在直接改为string返回
                    listener.success(body.toString());
                }
                SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
            }

            @Override
            public void onCodeError(String code, String message) {
                if (listener != null) {
                    listener.error(code, message);
                }
                SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
            }
        });
    }

    /**
     * 修改用户密码
     *
     * @param context     Context
     * @param oldPassword 旧密码
     * @param password    新密码
     * @param listener    回调
     */
    public void modifyPassword(final Context context, String oldPassword, String password
            , final CommonResultListener listener) {
        ModifyPasswordData data = new ModifyPasswordData(password, oldPassword);

        //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
        SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);

        request(context, data, SelfService.modifyUserPassword, new RequestListener() {
            @Override
            public void onSuccess(Object body) {
                if (listener != null) {
                    listener.success();
                }
                SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
            }

            @Override
            public void onCodeError(String code, String message) {
                if (listener != null) {
                    listener.error(code, message);
                }
                SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
            }
        });
    }

    /**
     * 修改手势密码
     *
     * @param context  Context
     * @param listener 回调
     */
    public void modifyGesturePassword(final Context context, String gesture, String newGestureData, final CommonResultListener listener) {
        if (checkJwtExist(context)) {
            ModifyGestureData data = new ModifyGestureData(gesture, newGestureData);
            //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);

            request(context, data, SelfService.modifyUserGesturePassword, new RequestListener() {
                @Override
                public void onSuccess(Object body) {
                    if (listener != null) {
                        listener.success();
                    }
                    SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                }

                @Override
                public void onCodeError(String code, String message) {
                    if (listener != null) {
                        listener.error(code, message);
                    }
                    SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                }
            });
        }
    }

    /**
     * 新增手势密码
     *
     * @param context  Context
     * @param gesture  手势密码
     * @param listener 回调
     */
    public void saveGesturePassword(final Context context, String gesture, final CommonResultListener listener) {
        if (checkJwtExist(context)) {
            JwtDecodeData decodeData = parseJwtToInfo(context);
            SaveGesturePasswordData data = new SaveGesturePasswordData(gesture, decodeData.getUserid());

            //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);

            request(context, data, SelfService.saveGesturePassword, new RequestListener() {
                @Override
                public void onSuccess(Object body) {
                    if (listener != null) {
                        listener.success();
                    }
                    SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                }

                @Override
                public void onCodeError(String code, String message) {
                    if (listener != null) {
                        listener.error(code, message);
                    }
                    SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                }
            });
        }
    }

    /**
     * 新增声纹信息
     * 需要3次
     *
     * @param context      Context
     * @param readContents 读取的内容
     * @param audioDatas   声纹信息（arm格式，base64编码）
     * @param listener     回调
     */
    public void addVoice(final Context context, List<String> readContents, List<String> audioDatas
            , final CommonResultListener listener) {
        if (checkJwtExist(context)) {
            //如果请求中含有JWT，表示为登录状态中的请求，则可以通过jwt去获取userinfo
            List<AddVoiceData.DatasBean> datasBeans = new ArrayList<>();
            for (int i = 0; i < readContents.size(); i++) {
                datasBeans.add(new AddVoiceData.DatasBean(audioDatas.get(i), readContents.get(i)));
            }
            AddVoiceData data = new AddVoiceData(EpassBaseSDK.Builder().getDeviceID(context), datasBeans);

            //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);

            request(context, data, SelfService.addVoice, new RequestListener() {
                @Override
                public void onSuccess(Object body) {
                    if (listener != null) {
                        listener.success();
                    }
                    SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                }

                @Override
                public void onCodeError(String code, String message) {
                    if (listener != null) {
                        listener.error(code, message);
                    }
                    SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                }
            });
        }
    }

    /**
     * 删除声纹信息
     *
     * @param context  Context
     * @param listener 回调
     */
    public void deleteVoice(final Context context, final CommonResultListener listener) {
        if (checkJwtExist(context)) {
            DeleteVoiceData data = new DeleteVoiceData(EpassBaseSDK.Builder().getDeviceID(context));
            //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);

            request(context, data, SelfService.deleteVoice, new RequestListener() {
                @Override
                public void onSuccess(Object body) {
                    if (listener != null) {
                        listener.success();
                    }
                    SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                }

                @Override
                public void onCodeError(String code, String message) {
                    if (listener != null) {
                        listener.error(code, message);
                    }
                    SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                }
            });
        }
    }

    /**
     * 新增人脸数据
     * <p>
     * 说明：只有新增的时候需要传递3张人脸照片
     *
     * @param context  Context
     * @param datas    人脸数据
     * @param listener 回调
     */
    public void addFace(final Context context, List<String> datas, final CommonResultListener listener) {
        if (checkJwtExist(context)) {
            List<AddFaceData.DatasBean> datasBeans = new ArrayList<>();
            for (String faceData :
                    datas) {
                datasBeans.add(new AddFaceData.DatasBean(faceData));
            }
            AddFaceData data = new AddFaceData(EpassBaseSDK.Builder().getDeviceID(context), datasBeans);

            //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);

            request(context, data, SelfService.addFace, new RequestListener() {
                @Override
                public void onSuccess(Object body) {
                    if (listener != null) {
                        listener.success();
                    }
                    SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                }

                @Override
                public void onCodeError(String code, String message) {
                    if (listener != null) {
                        listener.error(code, message);
                    }
                    SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                }
            });
        }
    }


    /**
     * 删除人脸数据
     *
     * @param context  Context
     * @param listener 回调
     */
    public void deleteFace(final Context context, final CommonResultListener listener) {
        if (checkJwtExist(context)) {
            DeleteFaceData data = new DeleteFaceData(EpassBaseSDK.Builder().getDeviceID(context));

            //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);

            request(context, data, SelfService.deleteFace, new RequestListener() {
                @Override
                public void onSuccess(Object body) {
                    if (listener != null) {
                        listener.success();
                    }
                    SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                }

                @Override
                public void onCodeError(String code, String message) {
                    if (listener != null) {
                        listener.error(code, message);
                    }
                    SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                }
            });
        }
    }

    /**
     * 在忘记密码的情况下，发送短信验证码
     *
     * @param context  Context
     * @param phone    手机号码
     * @param listener 回调
     */
    public void sendVerCodeForForgotPWD(final Context context, String phone
            , final CommonResultListener listener) {
        SendVerCodeByForgotPwdData data = new SendVerCodeByForgotPwdData(phone
                , SendVerCodeByForgotPwdData.FORGOT_PASSWORD);

        //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
        SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);

        request(context, data, SelfService.sendVerCodeForForgotPwd, new RequestListener() {
            @Override
            public void onSuccess(Object body) {
                if (listener != null) {
                    listener.success();
                }
                SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
            }

            @Override
            public void onCodeError(String code, String message) {
                if (listener != null) {
                    listener.error(code, message);
                }
                SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
            }
        });
    }

    /**
     * 重置密码 -- 忘记密码情况
     *
     * @param context     Context
     * @param newPassword 新密码
     * @param phone       手机号
     * @param verCode     验证码
     * @param listener    回调
     */
    public void updatePwdByVerCode(final Context context, String newPassword, String phone
            , String verCode, final CommonResultListener listener) {
        UpdatePwdByVerCodeData data = new UpdatePwdByVerCodeData(newPassword, phone, verCode);

        //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
        SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);

        request(context, data, SelfService.updatePwdFromVerCode, new RequestListener() {
            @Override
            public void onSuccess(Object body) {
                if (listener != null) {
                    listener.success();
                }
                SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
            }

            @Override
            public void onCodeError(String code, String message) {
                if (listener != null) {
                    listener.error(code, message);
                }
                SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
            }
        });
    }

    /**
     * 删除手势密码
     *
     * @param context    Context
     * @param gestureNum 手势密码
     * @param listener   回调
     */
    public void deleteGesturePassword(final Context context, String gestureNum
            , final CommonResultListener listener) {
        if (checkJwtExist(context)) {
            JwtDecodeData decodeData = parseJwtToInfo(context);
            DeleteGesturePasswordData deleteGesturePasswordData = new DeleteGesturePasswordData(gestureNum
                    , decodeData.getUserid());

            //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);

            request(context, deleteGesturePasswordData, SelfService.deleteGesturePassword
                    , new RequestListener() {
                        @Override
                        public void onSuccess(Object body) {
                            if (listener != null) {
                                listener.success();
                            }
                            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                        }

                        @Override
                        public void onCodeError(String code, String message) {
                            if (listener != null) {
                                listener.error(code, message);
                            }
                            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                        }
                    });
        }
    }

    /**
     * 获取用户信息
     *
     * @param context Context
     * @param encryptString 加密串
     */
    public void getUserNameByEncryptString(final Context context, String encryptString
            , final GetUserInfoListener listener) {
        if (checkJwtExist(context)) {
            JwtDecodeData decodeData = parseJwtToInfo(context);
            GetUserNameByEncryptData deleteGesturePasswordData = new GetUserNameByEncryptData(
                    EpassBaseSDK.Builder().getDeviceID(context), encryptString);

            //这个接口需要强制打开会话检查，只有打开了会话，服务器才会知道用户信息
            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, true);

            request(context, deleteGesturePasswordData, SelfService.getUserNameByEncryptString
                    , new RequestListener() {
                        @Override
                        public void onSuccess(Object body) {
                            if (listener != null) {
                                listener.success(body.toString());
                            }
                            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                        }

                        @Override
                        public void onCodeError(String code, String message) {
                            if (listener != null) {
                                listener.error(code, message);
                            }
                            SPUtils.put(context, InitConfigValue.BODY_IS_CHECK_SESSION, false);
                        }
                    });
        }
    }


    /**
     * 统一接口请求
     *
     * @param context  context
     * @param t        请求体
     * @param url      地址
     * @param listener 回调
     * @param <T>      请求体
     */
    private <T> void request(Context context, T t, String url, final RequestListener listener) {
        //预请求体
        Observer observer;
        Observable observable;

        //是否内容加密
        boolean isEncrypt = SPUtils.isEncrytContent(context);
        //Retrofit
        SelfService apiService = EpassBaseSDK.Builder().getAPi(context, SelfService.class);

        if (isEncrypt) {
            String body = JSON.toJSONString(t);
            String authKey = (String) SPUtils.get(context, InitConfigValue.KEY_ENCRYPT, InitConfigValue.DEFAULT);
            Object data = new EncrypteData(EpassBaseSDK.Builder().encryt(body, authKey));
            observable = apiService.commonRequestByEncrypt(url, data);
            observer = new BaseEncryptObserver(context) {
                @Override
                protected void onSuccess(Object body) {
                    LogUtil.getInstance().i("请求成功：" + body);
                    listener.onSuccess(body);
                }

                @Override
                protected void onCodeError(String code, String message, Object body) {
                    LogUtil.getInstance().d("请求失败：" + code);
                    listener.onCodeError(code, message);
                }
            };
        } else {
            observable = apiService.commonRequest(url, t);
            observer = new BaseObserver<Object>(context) {
                @Override
                protected void onSuccess(Object body) {
                    LogUtil.getInstance().i("请求成功：" + body);
                    listener.onSuccess(body);
                }

                @Override
                protected void onCodeError(String code, String message, Object object) {
                    LogUtil.getInstance().d("请求失败：" + code);
                    listener.onCodeError(code, message);
                }
            };
        }

        observable
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(observer);
    }


    /**
     * 统一接口请求
     *
     * @param context  context
     * @param url      地址
     * @param listener 回调
     * @param <T>      请求体
     */
    private <T> void requestByGet(Context context, String url, final RequestListener listener) {
        //预请求体
        Observer observer;
        Observable observable;


        //Retrofit
        SelfService apiService = EpassBaseSDK.Builder().getAPi(context, SelfService.class);

        //是否内容加密
        boolean isEncrypt = SPUtils.isEncrytContent(context);


        if (isEncrypt) {
            LogUtil.getInstance().d("加密");
            observable = apiService.commonRequestEncryptByGet(url);
            observer = new BaseEncryptObserver(context) {
                @Override
                protected void onSuccess(Object body) {
                    LogUtil.getInstance().i("请求成功：" + body);
                    listener.onSuccess(body);
                }

                @Override
                protected void onCodeError(String code, String message, Object body) {
                    LogUtil.getInstance().d("请求失败：" + code);
                    listener.onCodeError(code, message);
                }
            };
        } else {
            LogUtil.getInstance().d("非加密");
            observable = apiService.commonRequestByGet(url);
            observer = new BaseObserver<Object>(context) {
                @Override
                protected void onSuccess(Object body) {
                    LogUtil.getInstance().i("请求成功：" + body);
                    listener.onSuccess(body);
                }

                @Override
                protected void onCodeError(String code, String message, Object object) {
                    LogUtil.getInstance().d("请求失败：" + code);
                    listener.onCodeError(code, message);
                }
            };
        }


        observable
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(observer);
    }

    /**
     * 统一接口回调
     */
    private interface RequestListener {
        void onSuccess(Object body);

        void onCodeError(String code, String message);
    }
}
