package com.bambooclod.epassbase.bean;

import android.accounts.NetworkErrorException;
import android.content.Context;

import com.bambooclod.epassbase.http.CleanSaveSessionUtils;
import com.bambooclod.epassbase.log.LogUtil;

import java.net.ConnectException;
import java.net.UnknownHostException;
import java.util.concurrent.TimeoutException;

import io.reactivex.Observer;
import io.reactivex.disposables.Disposable;

/**
 * Comment: Retrofit请求过程管理
 *
 * @author Vangelis.Wang in BambooCloud
 * @date 2019-06-04
 * Email:wangpei@bamboocloud.cn
 */
public abstract class BaseObserver<T> implements Observer<BaseResponse<T>> {

    protected Context mContext;

    public BaseObserver(Context cxt) {
        this.mContext = cxt;
    }

    public BaseObserver() {

    }

    @Override
    public void onSubscribe(Disposable d) {
        onRequestStart();
    }

    @Override
    public void onNext(BaseResponse<T> tBaseResponse) {
        onRequestEnd();
        LogUtil.getInstance().e("获取数据：" + tBaseResponse.toString());

        if (tBaseResponse.isSuccess()) {
            try {
                //这个比较特殊
                //{"code":"user.password.expired.warn","message":"用户密码已过期，请尽快修改，以保证密码安全"}
                if("user.password.expired.warn".equals(tBaseResponse.getCode())){
                    onCodeError("-1",tBaseResponse.getMessage(),tBaseResponse.getBody());
                }
                onSuccess(tBaseResponse.getBody(),tBaseResponse.getMessage(),tBaseResponse.getCode());
            } catch (Exception e) {
                e.printStackTrace();
            }
        } else {
            try {
                //销毁Session时机
                CleanSaveSessionUtils.get().cleanSaveSession(mContext, tBaseResponse.getCode(), tBaseResponse);
                onCodeError(tBaseResponse.getCode(), tBaseResponse.getMessage(), tBaseResponse.getBody());
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    @Override
    public void onError(Throwable e) {
        //这里可以打印错误信息
        onRequestEnd();
        try {
            if (e instanceof ConnectException
                    || e instanceof TimeoutException
                    || e instanceof NetworkErrorException
                    || e instanceof UnknownHostException) {
                onFailure(e, true);
            } else {
                onFailure(e, false);
            }
        } catch (Exception e1) {
            e1.printStackTrace();
        }
    }

    @Override
    public void onComplete() {
    }

    /**
     * 返回成功
     */
    protected abstract void onSuccess(Object body,String message,String code) throws Exception;

    /**
     * 返回成功了,但是code错误
     */
    protected abstract void onCodeError(String code, String message, Object body);

    /**
     * 返回失败
     *
     * @param isNetWorkError 是否是网络错误
     */
    protected void onFailure(Throwable e, boolean isNetWorkError) {
        e.printStackTrace();
        LogUtil.getInstance().e("请求错误：" + e.getMessage());
    }

    /**
     * 请求开始
     */
    protected void onRequestStart() {
    }

    /**
     * 请求结束
     */
    protected void onRequestEnd() {
    }
}
