package com.bambooclod.epassbase.api;

import android.content.Context;
import android.widget.Toast;

import com.bambooclod.epassbase.config.InitConfigValue;
import com.bambooclod.epassbase.http.RetrofitFactory;
import com.bambooclod.epassbase.log.LogUtil;
import com.bambooclod.epassbase.otp.OtpUtils;
import com.bambooclod.epassbase.otp.Token;
import com.bambooclod.epassbase.otp.TokenCode;
import com.bambooclod.epassbase.otp.TokenPersistence;
import com.bambooclod.epassbase.sp.SPUtils;
import com.bambooclou.util.RandomUtil;
import com.bambooclou.util.auth.V2SignUtil;


/**
 * Comment: 对外调用类
 *
 * @author Vangelis.Wang in BambooCloud
 * @date 2019-06-04
 * Email:wangpei@bamboocloud.cn
 */
public class EpassBaseSDK {

    /**
     * OTP步长
     */
    private int mOtpStep = 30;
    private Context mContext;

    private EpassBaseSDK() {
    }

    // TODO: 2019-06-24 内存泄漏问题
    public static final EpassBaseSDK eaccount = new EpassBaseSDK();

    public static EpassBaseSDK Builder() {
        return eaccount;
    }

    /**
     * 指定后台地址
     */
    public EpassBaseSDK setBaseUrl(String baseUrl) {
        if ("".equals(baseUrl)) {
            throw new NullPointerException("后台地址不能为空!");
        } else {
            SPUtils.put(mContext, InitConfigValue.BASE_URL, baseUrl);
        }
        return this;
    }

    /**
     * 是否开启检查会话
     */
    public EpassBaseSDK isCheckSession(boolean checkSession) {
        SPUtils.put(mContext, InitConfigValue.BODY_IS_CHECK_SESSION, checkSession);
        return this;
    }

    /**
     * 是否开启内容加密传输
     */
    public EpassBaseSDK isEncrypt(boolean encrypt) {
        SPUtils.put(mContext, InitConfigValue.BODY_IS_ENCRYPT, encrypt);
        return this;
    }

    /**
     * 是否开启内容签名
     */
    public EpassBaseSDK isSign(boolean sign) {
        SPUtils.put(mContext, InitConfigValue.BODY_IS_SIGN, sign);
        return this;
    }

    /**
     * 设置OTP步长
     */
    public EpassBaseSDK setOtpStep(int step) {
        SPUtils.put(mContext, InitConfigValue.USER_OTP_STEP, mOtpStep);
        updateOtp(mContext);
        return this;
    }

    /**
     * 设置是否检测刷机
     */
    public EpassBaseSDK isBrushMachine(boolean bm) {
        SPUtils.put(mContext, InitConfigValue.DEVICE_FR_IS_CHECK_BM, bm);
        return this;
    }

    /**
     * 设置是否检测越狱
     */
    public EpassBaseSDK isCheckRoot(boolean root) {
        SPUtils.put(mContext, InitConfigValue.DEVICE_FR_IS_CHECK_ROOT, root);
        return this;
    }

    /**
     * 设置是否检测重装应用
     */
    public EpassBaseSDK isCheckReInstall(boolean install) {
        SPUtils.put(mContext, InitConfigValue.DEVICE_FR_IS_CHECK_REINSTALL, install);
        return this;
    }

    /**
     * 指定鉴权用户
     */
    public EpassBaseSDK setOperator(String operator) {
        if ("".equals(operator)) {
            throw new NullPointerException("鉴权用户不能为空!");
        } else {
            SPUtils.put(mContext, InitConfigValue.USER_OPERATOR, operator);
        }
        return this;
    }

    /**
     * 设置AppID
     */
    public EpassBaseSDK setAppID(String appID) {
        if ("".equals(appID)) {
            throw new NullPointerException("AppID不能为空!");
        } else {
            SPUtils.put(mContext, InitConfigValue.APP_ID, appID);
        }
        return this;
    }

    /**
     * 设置声纹录音倒计时
     */
    public EpassBaseSDK setAudioAuthTime(int time) {
        if (time < 3 || time > 20) {
            throw new NullPointerException("声纹认证倒计时不合法!");
        } else {
            SPUtils.put(mContext, InitConfigValue.AUDIO_RECORD_COUNTDOWN, time);
        }
        return this;
    }

    /**
     * 设置声纹认证类型
     */
    public EpassBaseSDK setAudioAuthType(String tpye) {
        if ("".equals(tpye)) {
            throw new NullPointerException("声纹认证类型不能为空!");
        } else {
            SPUtils.put(mContext, InitConfigValue.AUDIO_AUTH_TYPE, tpye);
        }
        return this;
    }

    /**
     * 设置鉴权秘钥
     */
    public EpassBaseSDK setAuthKey(String authKey) {
        if ("".equals(authKey)) {
            throw new NullPointerException("鉴权秘钥不能为空!");
        } else {
            SPUtils.put(mContext, InitConfigValue.KEY_AUTH, authKey);
        }
        return this;
    }

    /**
     * 设置加密秘钥
     */
    public EpassBaseSDK setEncryptKey(String encryptKey) {
        if ("".equals(encryptKey)) {
            throw new NullPointerException("加密秘钥不能为空!");
        } else {
            SPUtils.put(mContext, InitConfigValue.KEY_ENCRYPT, encryptKey);
        }
        return this;
    }


    /**
     * 设置签名秘钥
     */
    public EpassBaseSDK setSign(String sign) {
        if ("".equals(sign)) {
            throw new NullPointerException("签名秘钥不能为空!");
        } else {
            SPUtils.put(mContext, InitConfigValue.KEY_SIGN, sign);
        }
        return this;
    }

    /**
     * 设置声纹注册训练次数
     */
    public EpassBaseSDK setVoiceRegCount(int count) {
        SPUtils.put(mContext, InitConfigValue.VOICE_REG_REPEAT_COUNT, count);
        return this;
    }

    /**
     * 设置人脸注册训练次数
     */
    public EpassBaseSDK setFaceRegCount(int count) {
        SPUtils.put(mContext, InitConfigValue.FACE_REG_REPEAT_COUNT, count);
        return this;
    }

    /**
     * 设置Http请求超时时间
     */
    public EpassBaseSDK setHttpRequestTimeOut(long time) {
        SPUtils.put(mContext, InitConfigValue.HTTP_REQUEST_TIME_OUT, time);
        return this;
    }

    /**
     * 获取服务器地址
     */
    public String getBaseUrl(Context context) {
        return (String) SPUtils.get(context, InitConfigValue.BASE_URL, InitConfigValue.DEFAULT);
    }

    /**
     * 获取鉴权秘钥
     */
    public String getAuthKey(Context context) {
        return (String) SPUtils.get(context, InitConfigValue.KEY_AUTH, InitConfigValue.DEFAULT);
    }

    /**
     * 获取加密秘钥
     */
    public String getEncryptKey(Context context) {
        return (String) SPUtils.get(context, InitConfigValue.KEY_ENCRYPT, InitConfigValue.DEFAULT);
    }

    /**
     * 获取请求超时时间
     */
    public Long getRequestTimeOut(Context context) {
        return (Long) SPUtils.get(context, InitConfigValue.HTTP_REQUEST_TIME_OUT, 5L);
    }

    /**
     * 获取签名秘钥
     */
    public String getSignKey(Context context) {
        return (String) SPUtils.get(context, InitConfigValue.KEY_SIGN, InitConfigValue.DEFAULT);
    }

    /**
     * 初始化功能在这里写
     */
    public EpassBaseSDK init(Context context) {
        LogUtil.getInstance().e("初始化成功");
        this.mContext = context;

        //清空session
        LogUtil.getInstance().d("初始化的时候，清除SESSION");
        SPUtils.put(context, InitConfigValue.REQUEST_EP_SESSION, InitConfigValue.DEFAULT);

        updateOtp(context);
        return this;
    }

    /**
     * 更新OTP Key刷新配置
     */
    private void updateOtp(Context context) {
        //重新指定步长后，应该重新刷新配置
        String localOtpKey = (String) SPUtils.get(context, InitConfigValue.USER_OTP_KEY
                , InitConfigValue.DEFAULT);
        if (!InitConfigValue.DEFAULT.equals(localOtpKey) && localOtpKey != null) {
            OtpUtils.initOtp(context, localOtpKey);
        }
    }

    /**
     * 获取设备唯一标识码
     *
     * @param context Context
     */
    public String getDeviceID(Context context) {
        return EpassUtils.get().getDeviceID(context);
    }

    /**
     * 获取随机字符
     */
    public String getRandomString(int len) {
        return RandomUtil.getNumRandomString(len);
    }

    public String encryt(String val, String authKey) {
        return V2SignUtil.encryt(val, authKey);
    }

    public <T> T getAPi(Context context, Class clz) {
        return (T) RetrofitFactory.getInstance(context)
                .API().create(clz);
    }
}
